<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * Admin Bar Menu Handler
 *
 * Adds ProRank quick access menu to WordPress admin bar
 *
 * @package ProRank\SEO\Admin
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;


/**
 * AdminBarMenu class
 */
class AdminBarMenu {
    private const SEO_CACHE_KEY = 'prorank_adminbar_seo_summary';
    private const SEO_CACHE_TTL = 5 * MINUTE_IN_SECONDS;
    
    /**
     * Initialize admin bar menu
     *
     * @return void
     */
    public function init(): void {
        // Only show for users who can manage options
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Add admin bar menu
        add_action('admin_bar_menu', [$this, 'add_admin_bar_menu'], 100);

        // Handle AJAX debug toggle
        add_action('wp_ajax_prorank_toggle_debug', [$this, 'ajax_toggle_debug']);
        
        // Add inline styles and scripts
        add_action('admin_head', [$this, 'add_inline_assets']);
        add_action('wp_head', [$this, 'add_inline_assets']);
    }
    
    /**
     * Add admin bar menu items
     *
     * @param \WP_Admin_Bar $wp_admin_bar Admin bar instance
     * @return void
     */
    public function add_admin_bar_menu(\WP_Admin_Bar $wp_admin_bar): void {
        // Get SEO data
        $seo_data = $this->get_seo_dashboard_data();
        $score = $seo_data['seo_score'] ?? null;
        $issues = $seo_data['issues_count'] ?? 0;

        // Build main menu title with score badge
        $main_title = '<span class="ab-icon"></span><span class="ab-label">ProRank SEO</span>';
        if ($score !== null) {
            $score_class = $this->get_score_class($score);
            $main_title .= sprintf(' <span class="prorank-score-badge %s">%d</span>', $score_class, $score);
        }

        // ═══════════════════════════════════════════════════════════════
        // MAIN MENU
        // ═══════════════════════════════════════════════════════════════
        $wp_admin_bar->add_menu([
            'id' => 'prorank-seo',
            'title' => $main_title,
            'href' => admin_url('admin.php?page=prorank-seo'),
            'meta' => [
                'title' => __('ProRank SEO Dashboard', 'prorank-seo'),
                'class' => 'prorank-admin-bar-menu',
            ],
        ]);

        // ═══════════════════════════════════════════════════════════════
        // SEO OVERVIEW GROUP
        // ═══════════════════════════════════════════════════════════════
        $wp_admin_bar->add_group([
            'parent' => 'prorank-seo',
            'id' => 'prorank-seo-overview',
        ]);

        // SEO Score Display
        if ($score !== null) {
            $score_label = $this->get_score_label($score);
            $wp_admin_bar->add_menu([
                'parent' => 'prorank-seo-overview',
                'id' => 'prorank-score-display',
                'title' => sprintf(
                    '<span class="prorank-menu-icon">📊</span> SEO Score: <strong>%d/100</strong> <span class="prorank-score-label %s">%s</span>',
                    $score,
                    $this->get_score_class($score),
                    $score_label
                ),
                'href' => admin_url('admin.php?page=prorank-seo'),
            ]);
        }

        // Issues Alert
        if ($issues > 0) {
            $wp_admin_bar->add_menu([
                'parent' => 'prorank-seo-overview',
                'id' => 'prorank-issues-alert',
                'title' => sprintf(
                    '<span class="prorank-menu-icon">⚠️</span> <span class="prorank-issues-text">%d %s</span>',
                    $issues,
                    _n('Issue Found', 'Issues Found', $issues, 'prorank-seo')
                ),
                'href' => admin_url('admin.php?page=prorank-site-audit'),
            ]);
        }

        // ═══════════════════════════════════════════════════════════════
        // QUICK ACTIONS GROUP
        // ═══════════════════════════════════════════════════════════════
        $wp_admin_bar->add_group([
            'parent' => 'prorank-seo',
            'id' => 'prorank-quick-actions',
            'meta' => ['class' => 'ab-sub-secondary'],
        ]);

        // Dashboard
        $wp_admin_bar->add_menu([
            'parent' => 'prorank-quick-actions',
            'id' => 'prorank-dashboard',
            'title' => '<span class="prorank-menu-icon">🏠</span> ' . __('Dashboard', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-seo'),
        ]);

        // Site Audit
        $wp_admin_bar->add_menu([
            'parent' => 'prorank-quick-actions',
            'id' => 'prorank-site-audit',
            'title' => '<span class="prorank-menu-icon">🔍</span> ' . __('Site Audit', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-site-audit'),
        ]);

        // On-Page SEO
        $wp_admin_bar->add_menu([
            'parent' => 'prorank-quick-actions',
            'id' => 'prorank-on-page',
            'title' => '<span class="prorank-menu-icon">📝</span> ' . __('On-Page SEO', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-on-page-seo'),
        ]);

        // ═══════════════════════════════════════════════════════════════
        // PERFORMANCE TOOLS SUBMENU
        // ═══════════════════════════════════════════════════════════════
        $wp_admin_bar->add_menu([
            'parent' => 'prorank-seo',
            'id' => 'prorank-tools',
            'title' => '<span class="prorank-menu-icon">🛠️</span> ' . __('Performance Tools', 'prorank-seo'),
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-tools',
            'id' => 'prorank-pagespeed',
            'title' => __('PageSpeed Insights', 'prorank-seo'),
            'href' => '#',
            'meta' => ['onclick' => 'prorankTestPageSpeed(); return false;'],
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-tools',
            'id' => 'prorank-gtmetrix',
            'title' => __('GTmetrix Test', 'prorank-seo'),
            'href' => '#',
            'meta' => ['onclick' => 'prorankTestGTmetrix(); return false;'],
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-tools',
            'id' => 'prorank-mobile-test',
            'title' => __('Mobile-Friendly Test', 'prorank-seo'),
            'href' => '#',
            'meta' => ['onclick' => 'prorankMobileTest(); return false;'],
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-tools',
            'id' => 'prorank-schema-test',
            'title' => __('Schema Markup Test', 'prorank-seo'),
            'href' => '#',
            'meta' => ['onclick' => 'prorankSchemaTest(); return false;'],
        ]);

        // ═══════════════════════════════════════════════════════════════
        // SETTINGS SUBMENU
        // ═══════════════════════════════════════════════════════════════
        $wp_admin_bar->add_menu([
            'parent' => 'prorank-seo',
            'id' => 'prorank-settings',
            'title' => '<span class="prorank-menu-icon">⚙️</span> ' . __('Settings', 'prorank-seo'),
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-settings',
            'id' => 'prorank-settings-general',
            'title' => __('General Settings', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-seo'),
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-settings',
            'id' => 'prorank-settings-technical',
            'title' => __('Technical SEO', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-technical-seo'),
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-settings',
            'id' => 'prorank-settings-performance',
            'title' => __('Performance', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-performance'),
        ]);

        // Debug Mode Toggle (if WP_DEBUG)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $debug_mode = get_option('prorank_debug_mode', false);
            $wp_admin_bar->add_menu([
                'parent' => 'prorank-settings',
                'id' => 'prorank-debug-toggle',
                'title' => __('Debug Mode', 'prorank-seo') . ($debug_mode ? ' ✓' : ' ✗'),
                'href' => '#',
                'meta' => ['onclick' => 'prorankToggleDebug(); return false;'],
            ]);
        }

        // ═══════════════════════════════════════════════════════════════
        // HELP & SUPPORT
        // ═══════════════════════════════════════════════════════════════
        $wp_admin_bar->add_menu([
            'parent' => 'prorank-seo',
            'id' => 'prorank-help',
            'title' => '<span class="prorank-menu-icon">❓</span> ' . __('Help & Support', 'prorank-seo'),
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-help',
            'id' => 'prorank-documentation',
            'title' => __('Documentation', 'prorank-seo'),
            'href' => 'https://prorank.io/docs',
            'meta' => ['target' => '_blank'],
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-help',
            'id' => 'prorank-support-ticket',
            'title' => __('Submit Support Ticket', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-support-account'),
        ]);

        $wp_admin_bar->add_menu([
            'parent' => 'prorank-help',
            'id' => 'prorank-changelog',
            'title' => __('Changelog', 'prorank-seo'),
            'href' => admin_url('admin.php?page=prorank-support-account&tab=changelog'),
        ]);
    }
    
    /**
     * Add inline assets for admin bar functionality
     *
     * @return void
     */
    public function add_inline_assets(): void {
        // ProRank logo SVG - white version for dark admin bar background
        $admin_bar_svg = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1909 2048"><path transform="translate(959,1)" d="m0 0h10l6 3 5 8 12 32 8 19 9 24 7 20 16 40 10 26 3 10 13 34 4 9 12 31 7 20 10 25 9 23 7 18 6 18 8 21 7 16 11 29 3 10 14 36 5 12 11 28 4 13 12 36 3 16 1 13v13l-2 16-2 4-2 1-14 1-424 11h-14l-6 4-6 7-2 6-1 30-1 149v195l-3 8-7 6-35 12-28 11-27 10-37 13-16 6-12 5-30 11-11-2-11-9-16-13-11-8-9-8-11-9-5-4-5-5-14-12-10-8-12-9-13-11-11-9-5-4-5-5-13-11-14-11-9-7-13-11-14-11-8-8-14-11-9-8-6-5-5-3-13-11-11-9-10-9-13-11-11-9-8-6-10-8-13-11-11-9-12-11-10-8-11-9-11-8-15-13-10-8-12-11-22-18-11-8-15-13-10-8-12-11-11-9-4-5-1-2v-12l4-6 8-4 27-1 36-1 11-1 24-1 47-1 42-2 47-1 35-2 47-1 26-1 9-1 73-2 42-2h19l44-2 19-1 47-1 25-1 10-1 35-1h12l12-1 10-5 4-7 7-20 9-24 5-11 10-27 6-18 11-29 4-8 13-35 7-20 11-28 10-26 4-10 4-13 12-31 8-20 8-21 6-18 10-26 4-9 11-29 5-15 13-34 5-12 11-29 7-20 9-24 4-9 12-31 4-13 4-8z" fill="#ffffff"/><path transform="translate(728,646)" d="m0 0h324l24 2 24 4 20 5 20 7 16 7 19 10 15 10 14 11 13 12 11 12 10 13 10 15 11 21 7 18 7 25 4 25 1 11v62l-3 24-4 19-9 27-10 21-11 18-10 13-11 13-14 14-11 9-14 10-15 9-17 9-23 9-26 7-23 4-22 2h-185v275h-142z" fill="#ffffff"/><path transform="translate(1156,637)" d="m0 0h26l567 19 143 5 9 3 5 5 3 9-3 8-8 8-11 9-8 6-11 9-13 11-9 7-5 5-11 10-11 9-8 6-11 9-13 11-10 8-5 5-10 9-30 24-13 11-11 9-7 7-11 9-13 11-13 10-26 22-11 10-11 9-12 10-8 6-13 11-11 9-4 2-2 4-13 11-11 9-8 6-11 9-13 11-11 9-12 11-10 8-13 10-14 11-14 12-10 9-16 13-14 11-8 6-11 9-16 13-10 9-14 11-11 9-5 3-16 13-14 11-10 9-14 11-24 18-25 20-13 11-16 12-11 8-13 10-17 13-5 5-16 12-14 11-7 4-15 12-15 11-5 5-19 14-11 8-10 6-7 1-4-2h-12l-12 1-25 1-50 1-16 1h-17l-1-4v-31l1-24 1-59v-142l-3-1v-3h3l1 3 22 1 9 1 29 1h15l60 3h29l28-3 23-4 29-9 25-12 19-10 12-7 15-11 12-10 16-16 9-11 11-13 10-18 6-13 4-13 8-28 3-13 4-30 1-25v-39l-3-29-7-28-7-19-4-10-8-15-8-12-7-8-8-10-11-12-7-7v-2l-4-2-13-11-15-10-24-13-9-5-5-6-1-3v-11l3-6z" fill="#ffffff"/><path transform="translate(1369,1159)" d="m0 0 9 1 6 4 4 6 4 15 10 40 5 22 6 24 5 17 6 25 3 14 8 33 3 9 8 32 4 20 7 27 5 18 6 25 4 18 7 27 9 35 4 18 8 33 4 13 7 29 6 26 5 20 5 18 6 24 6 27 7 25 9 36v10l-3 6-6 5-9 2-8-3-16-8-19-10-18-11-29-16-12-6-21-12-21-13-24-14-12-6-22-13-15-10-27-16-16-9-21-13-13-9-24-15-14-8-23-15-16-11-17-11-11-7-9-5-20-13-10-7-23-15-14-9-9-5-17-11-12-8-10-7-25-16-14-8-18-12-5-6-3-9v-8l4-9 4-5 13-10 8-7 17-13 9-7 5-3 26-20 13-11 17-13 11-8 16-12 13-10 7-5 5-5 14-11 13-10 13-9 14-11 13-10 9-8 12-9 13-10 9-6 14-11 13-10 9-8 28-21 11-8 18-14 5-4h2l2-4 12-9 18-14 6-3z" fill="#ffffff"/><path transform="translate(714,1135)" d="m0 0h3l1 41v73l1 138-1 24h142l55 1h71l21 1-5 5-11 8-11 9-14 10-8 6-7 4-13 10-15 11h-2l-1 3-19 14-14 10-17 12-18 13-5 5-18 13-11 8-9 6-19 14-9 7-13 10-14 10-9 6-17 12-16 12-10 8-17 12-15 10-28 20h-2l-1 3-19 13-14 10-7 4-17 12-14 10-15 11-18 12-7 4-24 16-15 11-20 13-15 9-17 11-12 8-10 7-18 11-8 5-8 4-11 7-5-1-4-7-1-5v-13l7-30 4-13 7-29 3-15 7-28 4-14 6-25 3-15 7-28 4-15 6-24 3-15 7-28 4-15 6-24 3-15 7-28 4-15 6-24 5-23 5-20 4-15 6-24 3-15 7-28 4-15 6-24 3-15 7-28 5-18 2-5 25-10 48-17 10-3 37-13 15-6 37-13z" fill="#ffffff"/><path transform="translate(915,789)" d="m0 0h44l53 1 22 2 12 3 19 9 10 8 12 12 8 14 5 14 2 14v28l-3 19-7 16-7 12-4 2-2 4-10 9-13 8-11 5-12 3-8 1-31 1-36 1h-25l-29-1-12-1-1-23v-45l1-115z" fill="#ffffff"/></svg>';
        $icon_data = 'data:image/svg+xml;base64,' . base64_encode($admin_bar_svg);
        ?>
        <style>
            /* ═══════════════════════════════════════════════════════════════ */
            /* ProRank SEO Free Admin Bar Styles                               */
            /* ═══════════════════════════════════════════════════════════════ */

            /* Main Menu Icon - ProRank Logo */
            #wpadminbar #wp-admin-bar-prorank-seo > .ab-item .ab-icon {
                width: 20px !important;
                height: 20px !important;
                margin-right: 6px !important;
            }
            #wpadminbar #wp-admin-bar-prorank-seo > .ab-item .ab-icon::before {
                content: '' !important;
                display: block !important;
                width: 20px !important;
                height: 20px !important;
                background-image: url('<?php echo esc_attr($icon_data); ?>') !important;
                background-size: contain !important;
                background-repeat: no-repeat !important;
                background-position: center !important;
                font-family: inherit !important;
            }

            /* Score Badge in Top Bar */
            .prorank-score-badge {
                display: inline-block;
                padding: 2px 6px;
                border-radius: 3px;
                font-size: 11px;
                font-weight: 700;
                margin-left: 6px;
                vertical-align: middle;
            }
            .prorank-score-badge.good { background: linear-gradient(135deg, #28a745, #20c997); color: #fff; }
            .prorank-score-badge.ok { background: linear-gradient(135deg, #f97316, #fd7e14); color: #fff; }
            .prorank-score-badge.poor { background: linear-gradient(135deg, #dc3545, #e74c3c); color: #fff; }

            /* Score Label in Dropdown */
            .prorank-score-label {
                padding: 2px 8px;
                border-radius: 3px;
                font-size: 11px;
                font-weight: 600;
                margin-left: 8px;
            }
            .prorank-score-label.good { background: rgba(40, 167, 69, 0.2); color: #28a745; }
            .prorank-score-label.ok { background: rgba(249, 115, 22, 0.2); color: #f97316; }
            .prorank-score-label.poor { background: rgba(220, 53, 69, 0.2); color: #dc3545; }

            /* Issues Alert Text */
            .prorank-issues-text {
                color: #f97316 !important;
                font-weight: 600;
            }

            /* Menu Icons */
            .prorank-menu-icon {
                margin-right: 6px;
                font-size: 14px;
            }

            /* Dropdown Groups Styling */
            #wpadminbar .ab-sub-secondary {
                border-top: 1px solid rgba(255,255,255,0.1);
                padding-top: 6px;
                margin-top: 6px;
            }

            /* Hover Effects */
            #wpadminbar #wp-admin-bar-prorank-seo .ab-submenu a:hover {
                background: rgba(0, 212, 255, 0.1) !important;
                color: #00d4ff !important;
            }

            /* Submenu Headers */
            #wpadminbar #wp-admin-bar-prorank-tools > .ab-item,
            #wpadminbar #wp-admin-bar-prorank-settings > .ab-item,
            #wpadminbar #wp-admin-bar-prorank-help > .ab-item {
                font-weight: 600 !important;
                border-bottom: 1px solid rgba(255,255,255,0.05);
            }

            /* External link indicator */
            #wpadminbar #wp-admin-bar-prorank-documentation .ab-item:after {
                content: " ↗";
                font-size: 10px;
                opacity: 0.6;
            }
        </style>
        <script>
            function prorankTestPageSpeed() {
                const url = window.location.href;
                window.open('https://pagespeed.web.dev/report?url=' + encodeURIComponent(url), '_blank');
            }

            function prorankTestGTmetrix() {
                const url = window.location.href;
                window.open('https://gtmetrix.com/?url=' + encodeURIComponent(url), '_blank');
            }

            function prorankMobileTest() {
                const url = window.location.href;
                window.open('https://search.google.com/test/mobile-friendly?url=' + encodeURIComponent(url), '_blank');
            }

            function prorankSchemaTest() {
                const url = window.location.href;
                window.open('https://validator.schema.org/#url=' + encodeURIComponent(url), '_blank');
            }

            function prorankToggleDebug() {
                fetch(ajaxurl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=prorank_toggle_debug&_ajax_nonce=' + <?php echo json_encode(wp_create_nonce('prorank_toggle_debug')); ?>
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    }
                });
            }
        </script>
        <?php
    }
    
    /**
     * Handle AJAX debug toggle
     *
     * @return void
     */
    public function ajax_toggle_debug(): void {
        // Verify nonce
        if (!check_ajax_referer('prorank_toggle_debug', '_ajax_nonce', false)) {
            wp_send_json_error('Invalid security token');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Toggle debug mode
        $current_debug = get_option('prorank_debug_mode', false);
        update_option('prorank_debug_mode', !$current_debug);
        
        wp_send_json_success([
            'debug_mode' => !$current_debug,
            'message' => !$current_debug ? __('Debug mode enabled', 'prorank-seo') : __('Debug mode disabled', 'prorank-seo'),
        ]);
    }

    /**
     * Get SEO dashboard data from cache
     *
     * @return array
     */
    private function get_seo_dashboard_data(): array {
        $cached_dashboard = get_transient('prorank_dashboard_stats');
        if ($cached_dashboard !== false && is_array($cached_dashboard)) {
            $audit = $cached_dashboard['audit'] ?? null;
            if (is_array($audit) && !empty($audit['id']) && (($audit['state'] ?? '') === 'completed')) {
                return $cached_dashboard;
            }
        }

        $cached = get_transient(self::SEO_CACHE_KEY);
        if ($cached !== false && is_array($cached)) {
            return $cached;
        }

        $summary = $this->get_latest_completed_audit_summary();
        if ($summary !== null) {
            $data = [
                'seo_score' => $summary['score'],
                'issues_count' => $summary['issues_total'],
            ];
            set_transient(self::SEO_CACHE_KEY, $data, self::SEO_CACHE_TTL);
            return $data;
        }

        return [
            'seo_score' => null,
            'issues_count' => 0,
        ];
    }

    /**
     * Get latest completed audit summary (score + total issues).
     *
     * @return array{id: string, score: int, issues_total: int}|null
     */
    private function get_latest_completed_audit_summary(): ?array {
        global $wpdb;
        $table_audits = $wpdb->prefix . 'prorank_audits';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_audits)) !== $table_audits) {
            return null;
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT audit_id, score, stats
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             FROM {$table_audits}
             WHERE status = %s
             ORDER BY completed_at DESC, started_at DESC, id DESC
             LIMIT 1",
            'completed'
        ));

        if (!$row) {
            return null;
        }

        $score = is_numeric($row->score) ? (int) $row->score : null;
        if ($score === null) {
            return null;
        }

        $issues_total = 0;
        if (!empty($row->stats)) {
            $decoded = json_decode((string) $row->stats, true);
            if (is_array($decoded) && isset($decoded['unique_issues']['total'])) {
                $issues_total = (int) $decoded['unique_issues']['total'];
            }
        }

        return [
            'id' => (string) $row->audit_id,
            'score' => $score,
            'issues_total' => $issues_total,
        ];
    }

    /**
     * Get score class for styling
     *
     * @param int $score SEO score
     * @return string CSS class name
     */
    private function get_score_class(int $score): string {
        if ($score >= 80) return 'good';
        if ($score >= 50) return 'ok';
        return 'poor';
    }

    /**
     * Get score label text
     *
     * @param int $score SEO score
     * @return string Human-readable label
     */
    private function get_score_label(int $score): string {
        if ($score >= 90) return __('Excellent', 'prorank-seo');
        if ($score >= 80) return __('Good', 'prorank-seo');
        if ($score >= 60) return __('Fair', 'prorank-seo');
        if ($score >= 40) return __('Needs Work', 'prorank-seo');
        return __('Poor', 'prorank-seo');
    }
}
