<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * AJAX Handlers for ProRank SEO
 *
 * Handles all AJAX requests from the admin interface
 * Ensures all toggles and settings actually work
 *
 * @package ProRank\Admin
 * @since 3.0.0
 */

declare(strict_types=1);

namespace ProRank\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\Core\Cache\PageCacheManager;
use ProRank\SEO\Core\Services\LlmsTxtService;
use ProRank\Modules\Performance\ImageOptimizationModule;

class AjaxHandlers {

    /**
     * Initialize AJAX handlers
     */
    public function __construct() {
        // Universal settings handler
        add_action('wp_ajax_prorank_save_setting', [$this, 'handle_save_setting']);

        // Module management
        add_action('wp_ajax_prorank_toggle_module', [$this, 'handle_toggle_module']);

        // LLMs optimization - Individual handlers for each toggle
        add_action('wp_ajax_prorank_save_llms_setting', [$this, 'handle_save_llms_setting']);
        add_action('wp_ajax_prorank_schedule_llms_generation', [$this, 'handle_schedule_llms_generation']);
        add_action('wp_ajax_prorank_clear_llms_schedule', [$this, 'handle_clear_llms_schedule']);
        add_action('wp_ajax_prorank_llms_generate', [$this, 'handle_llms_generate']);
        add_action('wp_ajax_prorank_llms_status', [$this, 'handle_llms_status']);

        // Cache management
        add_action('wp_ajax_prorank_cache_control', [$this, 'handle_cache_control']);
        add_action('wp_ajax_prorank_clear_cache', [$this, 'handle_clear_cache']);

        // Database optimization
        add_action('wp_ajax_prorank_optimize_database', [$this, 'handle_optimize_database']);

        // Image optimization
        add_action('wp_ajax_prorank_optimize_images', [$this, 'handle_optimize_images']);
    }

    /**
     * Universal setting save handler
     * Makes sure ALL toggles and settings actually save
     */
    public function handle_save_setting() {
        // Security check
        if (!check_ajax_referer('prorank_nonce', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $setting = filter_input( INPUT_POST, 'setting', FILTER_UNSAFE_RAW );
        $setting = is_string( $setting ) ? sanitize_text_field( $setting ) : '';

        $raw_value = filter_input( INPUT_POST, 'value', FILTER_UNSAFE_RAW, FILTER_REQUIRE_ARRAY );
        if ( null === $raw_value || false === $raw_value ) {
            $raw_value = filter_input( INPUT_POST, 'value', FILTER_UNSAFE_RAW );
        }
        if ( null === $raw_value || false === $raw_value ) {
            $raw_value = '';
        }

        if ( is_array( $raw_value ) ) {
            $value = array_map( 'sanitize_text_field', $raw_value );
        } else {
            $value = sanitize_text_field( (string) $raw_value );
        }

        if (empty($setting)) {
            wp_send_json_error(['message' => __('Setting name required', 'prorank-seo')]);
            return;
        }

        // Handle boolean values
        if ($value === 'true' || $value === 'false') {
            $value = ($value === 'true');
        }

        // Save to database
        $option_name = 'prorank_' . $setting;
        update_option($option_name, $value);

        // Execute feature-specific actions
        $this->execute_feature_action($setting, $value);

        wp_send_json_success([
            'message' => __('Setting saved successfully', 'prorank-seo'),
            'setting' => $setting,
            'value' => $value
        ]);
    }

    /**
     * Handle module toggle
     */
    public function handle_toggle_module() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $module_slug = isset( $_POST['module'] ) ? sanitize_text_field( wp_unslash( $_POST['module'] ) ) : '';
        $enabled = isset( $_POST['enabled'] ) ? filter_var( wp_unslash( $_POST['enabled'] ), FILTER_VALIDATE_BOOLEAN ) : false;

        if (empty($module_slug)) {
            wp_send_json_error(['message' => __('Module slug required', 'prorank-seo')]);
            return;
        }

        // Get current modules
        $modules = get_option('prorank_modules', []);

        // Update module status
        $modules[$module_slug] = [
            'enabled' => $enabled,
            'updated_at' => current_time('mysql')
        ];

        update_option('prorank_modules', $modules);

        // Activate/deactivate module features
        if ($enabled) {
            $this->activate_module_features($module_slug);
        } else {
            $this->deactivate_module_features($module_slug);
        }

        wp_send_json_success([
            'message' => $enabled
                ? __('Module enabled successfully', 'prorank-seo')
                : __('Module disabled successfully', 'prorank-seo'),
            'module' => $module_slug,
            'enabled' => $enabled
        ]);
    }

    /**
     * Save individual LLMs setting
     */
    public function handle_save_llms_setting() {
        // Security check
        if (!check_ajax_referer('prorank_ajax', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $setting = isset( $_POST['setting'] ) ? sanitize_text_field( wp_unslash( $_POST['setting'] ) ) : '';
        $raw_value = isset( $_POST['value'] ) ? sanitize_text_field( wp_unslash( $_POST['value'] ) ) : '';
        $value = $raw_value === 'true' || $raw_value === true;

        // Get current settings using standardized key
        $llms_settings = get_option('prorank_llms_txt_settings', [
            'enabled' => false,
            'mode' => 'automatic',
            'pages' => [],
            'include_posts' => true,
            'include_pages' => true,
            'include_products' => true,
            'max_pages' => 20
        ]);

        // Update specific setting
        $llms_settings[$setting] = $value;
        update_option('prorank_llms_txt_settings', $llms_settings);

        // Log for debugging (dev only)
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('[ProRank LLMs] Setting ' . $setting . ' changed to ' . ($value ? 'true' : 'false'));
        }

        wp_send_json_success([
            'message' => __('Setting saved', 'prorank-seo'),
            'setting' => $setting,
            'value' => $value
        ]);
    }

    /**
     * Schedule weekly LLMs generation
     */
    public function handle_schedule_llms_generation() {
        // Security check
        if (!check_ajax_referer('prorank_ajax', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        // Clear any existing schedule
        wp_clear_scheduled_hook('prorank_llmstxt_regenerate');

        // Schedule weekly generation
        if (!wp_next_scheduled('prorank_llmstxt_regenerate')) {
            wp_schedule_event(time() + WEEK_IN_SECONDS, 'weekly', 'prorank_llmstxt_regenerate');
        }

        // Also generate immediately
        $this->generate_llms_files();

        wp_send_json_success([
            'message' => __('LLMs generation scheduled and files generated', 'prorank-seo'),
            'next_run' => wp_next_scheduled('prorank_llmstxt_regenerate')
        ]);
    }

    /**
     * Clear LLMs generation schedule
     */
    public function handle_clear_llms_schedule() {
        // Security check
        if (!check_ajax_referer('prorank_ajax', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        // Clear scheduled hook
        wp_clear_scheduled_hook('prorank_llmstxt_regenerate');

        // Also delete the generated files
        $upload_dir = wp_upload_dir();
        $llms_dir = $upload_dir['basedir'] . '/prorank-llms';

        if (file_exists($llms_dir . '/llms.txt')) {
            wp_delete_file($llms_dir . '/llms.txt');
        }
        if (file_exists($llms_dir . '/llms-full.txt')) {
            wp_delete_file($llms_dir . '/llms-full.txt');
        }

        // Remove symlinks from root
        if (file_exists(ABSPATH . 'llms.txt')) {
            wp_delete_file(ABSPATH . 'llms.txt');
        }
        if (file_exists(ABSPATH . 'llms-full.txt')) {
            wp_delete_file(ABSPATH . 'llms-full.txt');
        }

        wp_send_json_success([
            'message' => __('LLMs generation schedule cleared and files removed', 'prorank-seo')
        ]);
    }

    /**
     * Generate LLMs files immediately
     */
    private function generate_llms_files() {
        $settings = get_option('prorank_llms_txt_settings', [
            'enabled' => false,
            'mode' => 'automatic',
            'pages' => [],
            'include_posts' => true,
            'include_pages' => true,
            'include_products' => true,
            'max_pages' => 20
        ]);

        if (!$settings['enabled']) {
            return false;
        }

        // Gather pages based on settings
        $pages = [];

        if ($settings['mode'] === 'automatic') {
            $post_types = [];

            if ($settings['include_posts']) {
                $post_types[] = 'post';
            }
            if ($settings['include_pages']) {
                $post_types[] = 'page';
            }
            if ($settings['include_products'] && class_exists('WooCommerce')) {
                $post_types[] = 'product';
            }

            if (!empty($post_types)) {
                $query_args = [
                    'post_type' => $post_types,
                    'posts_per_page' => intval($settings['max_pages']),
                    'post_status' => 'publish',
                    'orderby' => 'date',
                    'order' => 'DESC'
                ];

                $query = new \WP_Query($query_args);
                $pages = $query->posts;
            }
        } else {
            // Manual mode - use selected pages
            if (!empty($settings['pages'])) {
                $pages = get_posts([
                    'post__in' => $settings['pages'],
                    'post_type' => 'any',
                    'posts_per_page' => -1,
                    'post_status' => 'publish'
                ]);
            }
        }

        // Generate content
        $this->create_llms_txt_files($pages);

        // Update last generated timestamp
        update_option('prorank_llms_last_generated', time());

        return true;
    }

    /**
     * Create the actual llms.txt files
     */
    private function create_llms_txt_files($pages) {
        $upload_dir = wp_upload_dir();
        $llms_dir = $upload_dir['basedir'] . '/prorank-llms';

        // Create directory if needed
        if (!file_exists($llms_dir)) {
            wp_mkdir_p($llms_dir);
        }

        // Generate index content (llms.txt)
        $site_name = get_bloginfo('name');
        $site_desc = get_bloginfo('description');
        $site_url = home_url();

        $index_content = "# {$site_name}\n\n";
        $index_content .= "## About\n";
        $index_content .= "{$site_desc}\n\n";
        $index_content .= "Website: {$site_url}\n\n";
        $index_content .= "## Key Pages\n\n";

        // Generate full content (llms-full.txt)
        $full_content = $index_content;

        foreach ($pages as $page) {
            $title = get_the_title($page);
            $url = get_permalink($page);
            $excerpt = wp_trim_words(get_the_excerpt($page), 50);

            // Add to index (simple list)
            $index_content .= "- [{$title}]({$url})\n";

            // Add detailed content to full version
            $full_content .= "\n---\n\n";
            $full_content .= "### {$title}\n";
            $full_content .= "URL: {$url}\n\n";
            if (!empty($excerpt)) {
                $full_content .= "Summary: {$excerpt}\n\n";
            }
        }

        // Add generation info
        $generated_date = gmdate('Y-m-d H:i:s');
        $footer = "\n---\nGenerated: {$generated_date} by ProRank SEO\n";

        $index_content .= $footer;
        $full_content .= $footer;

        // Write files
        file_put_contents($llms_dir . '/llms.txt', $index_content);
        file_put_contents($llms_dir . '/llms-full.txt', $full_content);

        // Create symlinks in root (for easy access)
        $root_llms = ABSPATH . 'llms.txt';
        $root_llms_full = ABSPATH . 'llms-full.txt';

        // Remove old symlinks if they exist
        if (is_link($root_llms) || file_exists($root_llms)) {
            wp_delete_file($root_llms);
        }
        if (is_link($root_llms_full) || file_exists($root_llms_full)) {
            wp_delete_file($root_llms_full);
        }

        // Create new symlinks
        // phpcs:ignore WordPress.WP.AlternativeFunctions.symlink_symlink
        @symlink($llms_dir . '/llms.txt', $root_llms);
        // phpcs:ignore WordPress.WP.AlternativeFunctions.symlink_symlink
        @symlink($llms_dir . '/llms-full.txt', $root_llms_full);

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank LLMs] Files generated successfully at ' . $llms_dir);
            }
        }
    }

    /**
     * Handle LLMs.txt generation
     */
    public function handle_llms_generate() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        try {
            // Generate llms.txt files using new service
            $service = new LlmsTxtService();
            $index_content = $service->generate_index();
            $full_content = $service->generate_full();

            if ($index_content && $full_content) {
                // Store in transients for virtual serving
                set_transient('prorank_llms_txt', $index_content, WEEK_IN_SECONDS);
                set_transient('prorank_llms_full_txt', $full_content, WEEK_IN_SECONDS);

                // Update last generated timestamp
                update_option('prorank_llms_last_generated', current_time('timestamp'));

                wp_send_json_success([
                    'message' => __('LLMs files generated successfully', 'prorank-seo'),
                    'urls' => [
                        'index' => home_url('/llms.txt'),
                        'full' => home_url('/llms-full.txt')
                    ]
                ]);
            } else {
                wp_send_json_error([
                    'message' => __('Failed to generate LLMs files', 'prorank-seo')
                ]);
            }
        } catch (\Exception $e) {
            wp_send_json_error([
                'message' => esc_html($e->getMessage())
            ]);
        }
    }

    /**
     * Get LLMs.txt status
     */
    public function handle_llms_status() {
        check_ajax_referer('prorank_nonce', 'nonce');

        $last_generated = get_option('prorank_llms_last_generated', 0);
        $next_update = $last_generated + (7 * DAY_IN_SECONDS); // Weekly updates

        wp_send_json_success([
            'last_generated' => $last_generated,
            'last_generated_formatted' => $last_generated ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_generated) : null,
            'next_update_formatted' => $last_generated ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $next_update) : null,
            'urls' => [
                'index' => home_url('/llms.txt'),
                'full' => home_url('/llms-full.txt')
            ],
            'cache_exists' => [
                'index' => file_exists(ABSPATH . 'llms.txt'),
                'full' => file_exists(ABSPATH . 'llms-full.txt')
            ]
        ]);
    }

    /**
     * Handle cache control
     */
    public function handle_cache_control() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $action = isset( $_POST['cache_action'] ) ? sanitize_text_field( wp_unslash( $_POST['cache_action'] ) ) : '';

        switch ($action) {
            case 'enable':
                $this->enable_page_cache();
                wp_send_json_success(['message' => __('Page cache enabled', 'prorank-seo')]);
                break;

            case 'disable':
                $this->disable_page_cache();
                wp_send_json_success(['message' => __('Page cache disabled', 'prorank-seo')]);
                break;

            case 'clear':
                $this->clear_page_cache();
                wp_send_json_success(['message' => __('Cache cleared successfully', 'prorank-seo')]);
                break;

            default:
                wp_send_json_error(['message' => __('Invalid cache action', 'prorank-seo')]);
        }
    }

    /**
     * Handle cache clearing
     */
    public function handle_clear_cache() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $type = isset( $_POST['cache_type'] ) ? sanitize_text_field( wp_unslash( $_POST['cache_type'] ) ) : 'all';

        try {
            $cleared = [];

            if ($type === 'all' || $type === 'page') {
                $this->clear_page_cache();
                $cleared[] = 'page';
            }

            if ($type === 'all' || $type === 'object') {
                wp_cache_flush();
                $cleared[] = 'object';
            }

            if ($type === 'all' || $type === 'transient') {
                $this->clear_transients();
                $cleared[] = 'transient';
            }

            wp_send_json_success([
                /* translators: %s: list of cache types cleared */
                'message' => sprintf(
                    /* translators: %s: placeholder value */
                    __('Cleared %s cache', 'prorank-seo'),
                    implode(', ', $cleared)
                ),
                'cleared' => $cleared
            ]);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => esc_html($e->getMessage())]);
        }
    }

    /**
     * Handle database optimization
     */
    public function handle_optimize_database() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        global $wpdb;

        try {
            $optimized_tables = [];

            // Get all database tables
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);

            foreach ($tables as $table) {
                $table_name = $table[0];

                // Only optimize WordPress tables
                if (strpos($table_name, $wpdb->prefix) === 0) {
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $wpdb->query("OPTIMIZE TABLE `$table_name`");
                    $optimized_tables[] = $table_name;
                }
            }

            // Clean up options
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_%'");

            // Clean up post meta
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("DELETE pm FROM {$wpdb->postmeta} pm LEFT JOIN {$wpdb->posts} p ON p.ID = pm.post_id WHERE p.ID IS NULL");

            wp_send_json_success([
                /* translators: %d: number of database tables optimized */
                'message' => sprintf(
                    /* translators: %s: placeholder value */
                    __('Optimized %d tables', 'prorank-seo'),
                    count($optimized_tables)
                ),
                'tables' => $optimized_tables
            ]);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => esc_html($e->getMessage())]);
        }
    }

    /**
     * Handle image optimization
     */
    public function handle_optimize_images() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $image_id = intval($_POST['image_id'] ?? 0);

        if (!$image_id) {
            wp_send_json_error(['message' => __('Image ID required', 'prorank-seo')]);
            return;
        }

        try {
            $optimizer = new \ProRank\Core\ImageOptimizer();
            $result = $optimizer->optimize($image_id);

            if ($result['success']) {
                wp_send_json_success([
                    'message' => __('Image optimized successfully', 'prorank-seo'),
                    'saved' => $result['saved_bytes'],
                    'percentage' => $result['percentage']
                ]);
            } else {
                wp_send_json_error(['message' => $result['message']]);
            }
        } catch (\Exception $e) {
            wp_send_json_error(['message' => esc_html($e->getMessage())]);
        }
    }

    /**
     * Execute feature-specific actions when settings change
     */
    private function execute_feature_action($setting, $value) {
        switch ($setting) {
            case 'page_cache':
            case 'cache_enabled':
                if ($value) {
                    $this->enable_page_cache();
                } else {
                    $this->disable_page_cache();
                }
                break;

            case 'browser_cache_enabled':
                if ($value) {
                    $this->enable_browser_cache();
                } else {
                    $this->disable_browser_cache();
                }
                break;

            case 'gzip_compression':
                if ($value) {
                    $this->enable_gzip_compression();
                } else {
                    $this->disable_gzip_compression();
                }
                break;

            case 'lazy_load_images':
                if ($value) {
                    add_filter('wp_lazy_loading_enabled', '__return_true');
                } else {
                    add_filter('wp_lazy_loading_enabled', '__return_false');
                }
                break;
        }
    }

    /**
     * Enable page caching
     */
    private function enable_page_cache() {
        // Create cache directory if it doesn't exist
        $cache_dir = WP_CONTENT_DIR . '/cache/prorank/';
        if (!file_exists($cache_dir)) {
            wp_mkdir_p($cache_dir);
        }

        // Update .htaccess for caching rules
        $this->update_htaccess_cache_rules(true);

        update_option('prorank_page_cache_enabled', true);
    }

    /**
     * Disable page caching
     */
    private function disable_page_cache() {
        // Clear existing cache
        $this->clear_page_cache();

        // Remove .htaccess rules
        $this->update_htaccess_cache_rules(false);

        update_option('prorank_page_cache_enabled', false);
    }

    /**
     * Clear page cache
     */
    private function clear_page_cache() {
        $cache_dir = WP_CONTENT_DIR . '/cache/prorank/';

        if (file_exists($cache_dir)) {
            $this->delete_directory($cache_dir);
            wp_mkdir_p($cache_dir);
        }
    }

    /**
     * Clear transients
     */
    private function clear_transients() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_%'");
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_site_transient_%'");
    }

    /**
     * Enable browser caching via .htaccess
     */
    private function enable_browser_cache() {
        $rules = $this->get_browser_cache_rules();
        $this->insert_htaccess_rules($rules, 'ProRank Browser Cache');
    }

    /**
     * Disable browser caching
     */
    private function disable_browser_cache() {
        $this->remove_htaccess_rules('ProRank Browser Cache');
    }

    /**
     * Enable GZIP compression
     */
    private function enable_gzip_compression() {
        $rules = $this->get_gzip_rules();
        $this->insert_htaccess_rules($rules, 'ProRank GZIP');
    }

    /**
     * Disable GZIP compression
     */
    private function disable_gzip_compression() {
        $this->remove_htaccess_rules('ProRank GZIP');
    }

    /**
     * Get browser cache rules for .htaccess
     */
    private function get_browser_cache_rules() {
        return '
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType application/x-shockwave-flash "access plus 1 month"
    ExpiresByType image/x-icon "access plus 1 year"
    ExpiresDefault "access plus 2 days"
</IfModule>';
    }

    /**
     * Get GZIP compression rules
     */
    private function get_gzip_rules() {
        return '
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css
    AddOutputFilterByType DEFLATE text/javascript application/javascript application/x-javascript
    AddOutputFilterByType DEFLATE application/xml application/xhtml+xml application/rss+xml
    AddOutputFilterByType DEFLATE application/font-woff application/font-woff2 application/vnd.ms-fontobject
    AddOutputFilterByType DEFLATE application/x-font-ttf font/opentype image/svg+xml image/x-icon
</IfModule>';
    }

    /**
     * Update .htaccess with cache rules
     */
    private function update_htaccess_cache_rules($enable) {
        if ($enable) {
            $rules = $this->get_browser_cache_rules();
            $this->insert_htaccess_rules($rules, 'ProRank Cache');
        } else {
            $this->remove_htaccess_rules('ProRank Cache');
        }
    }

    /**
     * Insert rules into .htaccess
     */
    private function insert_htaccess_rules($rules, $marker) {
        if (!function_exists('insert_with_markers')) {
            require_once(ABSPATH . 'wp-admin/includes/misc.php');
        }

        $htaccess = ABSPATH . '.htaccess';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ( $wp_filesystem && $wp_filesystem->is_writable( $htaccess ) ) {
            $lines = explode("\n", trim($rules));
            insert_with_markers($htaccess, $marker, $lines);
        }
    }

    /**
     * Remove rules from .htaccess
     */
    private function remove_htaccess_rules($marker) {
        if (!function_exists('insert_with_markers')) {
            require_once(ABSPATH . 'wp-admin/includes/misc.php');
        }

        $htaccess = ABSPATH . '.htaccess';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ( $wp_filesystem && $wp_filesystem->is_writable( $htaccess ) ) {
            insert_with_markers($htaccess, $marker, []);
        }
    }

    /**
     * Activate module features
     */
    private function activate_module_features($module_slug) {
        switch ($module_slug) {
            case 'page-cache':
                $this->enable_page_cache();
                break;

            case 'image-optimization':
                update_option('prorank_image_optimization_enabled', true);
                break;

            case 'database-optimization':
                wp_schedule_event(time(), 'weekly', 'prorank_optimize_database');
                break;

            case 'llms-optimization':
                update_option('prorank_llms_enabled', true);
                wp_schedule_event(time(), 'weekly', 'prorank_llmstxt_regenerate');
                break;
        }
    }

    /**
     * Deactivate module features
     */
    private function deactivate_module_features($module_slug) {
        switch ($module_slug) {
            case 'page-cache':
                $this->disable_page_cache();
                break;

            case 'image-optimization':
                update_option('prorank_image_optimization_enabled', false);
                break;

            case 'database-optimization':
                wp_clear_scheduled_hook('prorank_optimize_database');
                break;

            case 'llms-optimization':
                update_option('prorank_llms_enabled', false);
                wp_clear_scheduled_hook('prorank_llmstxt_regenerate');
                break;
        }
    }

    /**
     * Recursively delete a directory
     */
    private function delete_directory($dir) {
        if (!file_exists($dir)) {
            return;
        }

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        $files = array_diff(scandir($dir), ['.', '..']);

        foreach ($files as $file) {
            $path = $dir . '/' . $file;

            if (is_dir($path)) {
                $this->delete_directory($path);
            } else {
                wp_delete_file($path);
            }
        }

        if ( $wp_filesystem ) {
            $wp_filesystem->rmdir( $dir );
        }
    }
}

// Initialize the AJAX handlers
new AjaxHandlers();
