<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * WordPress Dashboard Widget for ProRank SEO (Free Version)
 *
 * Uses local database for audit data
 *
 * @package ProRank\SEO\Admin
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

/**
 * Dashboard Widget class
 */
class DashboardWidget {

    /**
     * Initialize the dashboard widget
     */
    public function init(): void {
        add_action('wp_dashboard_setup', [$this, 'add_dashboard_widget']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_widget_styles']);
    }

    /**
     * Add the widget to WordPress dashboard
     */
    public function add_dashboard_widget(): void {
        wp_add_dashboard_widget(
            'prorank_seo_dashboard_widget',
            __('ProRank SEO - Site Health', 'prorank-seo'),
            [$this, 'render_widget'],
            null,
            null,
            'normal',
            'high'
        );
    }

    /**
     * Enqueue widget-specific styles
     */
    public function enqueue_widget_styles($hook): void {
        if ($hook !== 'index.php') {
            return;
        }

        wp_add_inline_style('dashboard', $this->get_widget_styles());
    }

    /**
     * Get audit data from local database
     *
     * @return array
     */
    private function get_audit_data(): array {
        global $wpdb;
        $table_audits = $wpdb->prefix . 'prorank_audits';

        // Check if table exists
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_audits)) !== $table_audits) {
            return [
                'state' => 'idle',
                'score' => null,
                'last_updated' => null,
                'stats' => [],
                'progress' => [],
            ];
        }

        // Check for running audit first
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $running = $wpdb->get_row($wpdb->prepare(
            "SELECT audit_id, score, stats, started_at, progress
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             FROM {$table_audits}
             WHERE status IN (%s, %s)
             ORDER BY started_at DESC
             LIMIT 1",
            'running',
            'crawling'
        ));

        if ($running) {
            $progress = [];
            if (!empty($running->progress)) {
                $progress = json_decode($running->progress, true) ?: [];
            }

            return [
                'state' => 'running',
                'score' => null,
                'last_updated' => $running->started_at,
                'stats' => [],
                'progress' => $progress,
            ];
        }

        // Get most recent completed audit
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $completed = $wpdb->get_row($wpdb->prepare(
            "SELECT audit_id, score, stats, completed_at, pages_crawled
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             FROM {$table_audits}
             WHERE status = %s
             ORDER BY completed_at DESC, started_at DESC, id DESC
             LIMIT 1",
            'completed'
        ));

        if (!$completed) {
            return [
                'state' => 'idle',
                'score' => null,
                'last_updated' => null,
                'stats' => [],
                'progress' => [],
            ];
        }

        $score = is_numeric($completed->score) ? (int) $completed->score : null;
        $stats = [
            'critical_issues' => 0,
            'high_issues' => 0,
            'medium_issues' => 0,
            'low_issues' => 0,
            'total_urls' => (int) ($completed->pages_crawled ?? 0),
        ];

        if (!empty($completed->stats)) {
            $decoded = json_decode($completed->stats, true);
            if (is_array($decoded)) {
                // Parse issue counts from stats
                if (isset($decoded['unique_issues']['by_severity'])) {
                    $by_severity = $decoded['unique_issues']['by_severity'];
                    $stats['critical_issues'] = (int) ($by_severity['critical'] ?? 0);
                    $stats['high_issues'] = (int) ($by_severity['high'] ?? 0);
                    $stats['medium_issues'] = (int) ($by_severity['medium'] ?? 0);
                    $stats['low_issues'] = (int) ($by_severity['low'] ?? 0);
                }
                if (isset($decoded['pages_crawled'])) {
                    $stats['total_urls'] = (int) $decoded['pages_crawled'];
                }
            }
        }

        return [
            'state' => 'completed',
            'score' => $score,
            'last_updated' => $completed->completed_at,
            'stats' => $stats,
            'progress' => [],
        ];
    }

    /**
     * Render the widget content
     */
    public function render_widget(): void {
        // Fetch real audit data from external service
        $audit_data = $this->get_audit_data();

        $score = $audit_data['score'];
        $state = $audit_data['state'];
        $last_updated = $audit_data['last_updated'];
        $stats = $audit_data['stats'];
        $progress = $audit_data['progress'] ?? [];
        
        $total_issues = ($stats['critical_issues'] ?? 0) + ($stats['high_issues'] ?? 0) + ($stats['medium_issues'] ?? 0) + ($stats['low_issues'] ?? 0);
        $has_audit_data = ($state === 'completed' || $state === 'crawling' || $state === 'running') && ($score !== null || $total_issues > 0);

        ?>
        <div class="prorank-dashboard-widget">
            <?php if ($state === 'idle' || (!$has_audit_data && $state !== 'crawling' && $state !== 'running')): ?>
                <div class="prorank-no-audit">
                    <div class="no-audit-icon">
                        <span class="dashicons dashicons-chart-area"></span>
                    </div>
                    <h4><?php esc_html_e('No Audit Data Available', 'prorank-seo'); ?></h4>
                    <p class="description"><?php esc_html_e('Run a site audit to see your SEO health score and issues.', 'prorank-seo'); ?></p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=prorank-site-audit')); ?>" class="button button-primary">
                        <?php esc_html_e('Run Site Audit', 'prorank-seo'); ?>
                    </a>
                </div>
            <?php elseif ($state === 'crawling' || $state === 'running'): ?>
                <div class="prorank-audit-running">
                    <div class="running-indicator">
                        <span class="dashicons dashicons-update spin"></span>
                        <span class="running-text"><?php esc_html_e('Audit in Progress', 'prorank-seo'); ?></span>
                    </div>
                    <?php if (!empty($progress)): ?>
                        <div class="progress-bar-container">
                            <div class="progress-bar" style="width: <?php echo esc_attr($progress['percent'] ?? 0); ?>%"></div>
                        </div>
                        <p class="progress-text">
                            <?php
                            $checked = $progress['checked_urls'] ?? 0;
                            $total = $progress['total_urls'] ?? 0;
                            if ($total > 0) {
                                /* translators: %1$d: checked URLs count, %2$d: total URLs count, %3$d: percentage */
                                echo esc_html( sprintf( __('Analyzing %1$d of %2$d URLs (%3$d%%)', 'prorank-seo'), (int) $checked, (int) $total, (int) ( $progress['percent'] ?? 0 ) ) );
                            } else {
                                esc_html_e('Discovering URLs...', 'prorank-seo');
                            }
                            ?>
                        </p>
                    <?php endif; ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=prorank-site-audit')); ?>" class="button">
                        <?php esc_html_e('View Progress', 'prorank-seo'); ?>
                    </a>
                </div>
            <?php else: ?>
                <div class="prorank-audit-score">
                    <div class="score-circle <?php echo esc_attr($this->get_score_class((int)$score)); ?>">
                        <svg viewBox="0 0 120 120" class="score-svg">
                            <circle class="score-bg" cx="60" cy="60" r="52" />
                            <circle class="score-progress" cx="60" cy="60" r="52"
                                stroke-dasharray="<?php echo esc_attr(((int)$score / 100) * 327); ?> 327"
                                style="stroke: <?php echo esc_attr($this->get_score_color((int)$score)); ?>" />
                        </svg>
                        <div class="score-text">
                            <div class="score-value"><?php echo esc_html($score ?? 0); ?></div>
                            <div class="score-label"><?php echo esc_html($this->get_score_label((int)($score ?? 0))); ?></div>
                        </div>
                    </div>

                    <div class="score-details">
                        <div class="issue-grid">
                            <div class="issue-item critical">
                                <span class="count"><?php echo esc_html($stats['critical_issues'] ?? 0); ?></span>
                                <span class="label"><?php esc_html_e('Critical', 'prorank-seo'); ?></span>
                            </div>
                            <div class="issue-item high">
                                <span class="count"><?php echo esc_html($stats['high_issues'] ?? 0); ?></span>
                                <span class="label"><?php esc_html_e('High', 'prorank-seo'); ?></span>
                            </div>
                            <div class="issue-item medium">
                                <span class="count"><?php echo esc_html($stats['medium_issues'] ?? 0); ?></span>
                                <span class="label"><?php esc_html_e('Medium', 'prorank-seo'); ?></span>
                            </div>
                            <div class="issue-item low">
                                <span class="count"><?php echo esc_html($stats['low_issues'] ?? 0); ?></span>
                                <span class="label"><?php esc_html_e('Low', 'prorank-seo'); ?></span>
                            </div>
                        </div>

                        <?php if (!empty($stats['total_urls'])): ?>
                            <p class="pages-scanned">
                                <span class="dashicons dashicons-admin-page"></span>
                                <?php
                                /* translators: %d: number of pages analyzed */
                                echo esc_html( sprintf( __('%d pages analyzed', 'prorank-seo'), (int) $stats['total_urls'] ) );
                                ?>
                            </p>
                        <?php endif; ?>

                        <?php if ($last_updated): ?>
                            <p class="last-updated">
                                <span class="dashicons dashicons-clock"></span>
                                <?php
                                $updated_time = strtotime($last_updated);
                                if ($updated_time) {
                                    /* translators: %s: human-readable time difference */
                                    echo esc_html( sprintf(
                                        /* translators: %s: placeholder value */
                                        __('Last audit: %s ago', 'prorank-seo'),
                                        human_time_diff($updated_time, current_time('timestamp'))
                                    ) );
                                }
                                ?>
                            </p>
                        <?php endif; ?>

                        <div class="widget-actions">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=prorank-site-audit')); ?>" class="button button-primary">
                                <?php esc_html_e('View Details', 'prorank-seo'); ?>
                            </a>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=prorank-site-audit')); ?>" class="button">
                                <?php esc_html_e('Run New Audit', 'prorank-seo'); ?>
                            </a>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Get score color
     */
    private function get_score_color(int $score): string {
        if ($score >= 90) return '#10b981';
        if ($score >= 70) return '#84cc16';
        if ($score >= 50) return '#f59e0b';
        if ($score >= 30) return '#f97316';
        return '#ef4444';
    }
    
    /**
     * Get score class for styling
     */
    private function get_score_class(int $score): string {
        if ($score >= 90) return 'excellent';
        if ($score >= 70) return 'good';
        if ($score >= 50) return 'fair';
        if ($score >= 30) return 'poor';
        return 'critical';
    }
    
    /**
     * Get score label
     */
    private function get_score_label(int $score): string {
        if ($score >= 90) return __('Excellent', 'prorank-seo');
        if ($score >= 70) return __('Good', 'prorank-seo');
        if ($score >= 50) return __('Fair', 'prorank-seo');
        if ($score >= 30) return __('Poor', 'prorank-seo');
        return __('Critical', 'prorank-seo');
    }
    
    /**
     * Get widget styles
     */
    private function get_widget_styles(): string {
        return '
        .prorank-dashboard-widget {
            padding: 12px 0;
        }

        /* No Audit State */
        .prorank-no-audit {
            text-align: center;
            padding: 30px 20px;
        }

        .prorank-no-audit .no-audit-icon {
            margin-bottom: 15px;
        }

        .prorank-no-audit .no-audit-icon .dashicons {
            font-size: 48px;
            width: 48px;
            height: 48px;
            color: #94a3b8;
        }

        .prorank-no-audit h4 {
            margin: 0 0 10px;
            font-size: 16px;
            color: #1e293b;
        }

        .prorank-no-audit .description {
            color: #64748b;
            margin-bottom: 20px;
        }

        /* Running State */
        .prorank-audit-running {
            text-align: center;
            padding: 25px 20px;
        }

        .prorank-audit-running .running-indicator {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            margin-bottom: 20px;
        }

        .prorank-audit-running .running-indicator .dashicons {
            color: #3b82f6;
            font-size: 24px;
            width: 24px;
            height: 24px;
        }

        .prorank-audit-running .running-text {
            font-size: 16px;
            font-weight: 600;
            color: #1e293b;
        }

        .progress-bar-container {
            background: #e2e8f0;
            border-radius: 4px;
            height: 8px;
            overflow: hidden;
            margin-bottom: 10px;
        }

        .progress-bar {
            background: linear-gradient(90deg, #3b82f6, #22c55e);
            height: 100%;
            border-radius: 4px;
            transition: width 0.3s ease;
        }

        .progress-text {
            color: #64748b;
            font-size: 13px;
            margin-bottom: 15px;
        }

        /* Score Display */
        .prorank-audit-score {
            display: flex;
            gap: 20px;
            align-items: flex-start;
        }

        .score-circle {
            flex-shrink: 0;
            width: 100px;
            height: 100px;
            position: relative;
        }

        .score-svg {
            width: 100%;
            height: 100%;
            transform: rotate(-90deg);
        }

        .score-bg {
            fill: none;
            stroke: #e2e8f0;
            stroke-width: 8;
        }

        .score-progress {
            fill: none;
            stroke-width: 8;
            stroke-linecap: round;
            transition: stroke-dasharray 0.5s ease;
        }

        .score-text {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            text-align: center;
        }

        .score-value {
            font-size: 28px;
            font-weight: bold;
            line-height: 1;
            color: #1e293b;
        }

        .score-label {
            font-size: 11px;
            margin-top: 2px;
            font-weight: 500;
            color: #64748b;
            text-transform: uppercase;
        }

        .score-details {
            flex: 1;
            min-width: 0;
        }

        /* Issue Grid */
        .issue-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 8px;
            margin-bottom: 15px;
        }

        .issue-item {
            text-align: center;
            padding: 8px 4px;
            border-radius: 6px;
            background: #f8fafc;
            border: 1px solid #e2e8f0;
        }

        .issue-item .count {
            display: block;
            font-weight: 700;
            font-size: 18px;
            line-height: 1.2;
        }

        .issue-item .label {
            display: block;
            font-size: 10px;
            text-transform: uppercase;
            font-weight: 500;
            margin-top: 2px;
        }

        .issue-item.critical {
            background: #fef2f2;
            border-color: #fecaca;
        }
        .issue-item.critical .count { color: #dc2626; }
        .issue-item.critical .label { color: #ef4444; }

        .issue-item.high {
            background: #fffbeb;
            border-color: #fde68a;
        }
        .issue-item.high .count { color: #d97706; }
        .issue-item.high .label { color: #f59e0b; }

        .issue-item.medium {
            background: #eff6ff;
            border-color: #bfdbfe;
        }
        .issue-item.medium .count { color: #2563eb; }
        .issue-item.medium .label { color: #3b82f6; }

        .issue-item.low {
            background: #f0fdf4;
            border-color: #bbf7d0;
        }
        .issue-item.low .count { color: #16a34a; }
        .issue-item.low .label { color: #22c55e; }

        /* Meta Info */
        .pages-scanned,
        .last-updated {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 12px;
            color: #64748b;
            margin: 8px 0;
        }

        .pages-scanned .dashicons,
        .last-updated .dashicons {
            font-size: 14px;
            width: 14px;
            height: 14px;
        }

        /* Actions */
        .widget-actions {
            margin-top: 15px;
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }

        .widget-actions .button {
            font-size: 12px;
            padding: 4px 12px;
        }

        /* Spinning animation */
        .dashicons.spin {
            animation: prorank-spin 1.5s linear infinite;
        }

        @keyframes prorank-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Responsive */
        @media (max-width: 782px) {
            .prorank-audit-score {
                flex-direction: column;
                align-items: center;
                text-align: center;
            }

            .issue-grid {
                grid-template-columns: repeat(2, 1fr);
            }

            .pages-scanned,
            .last-updated {
                justify-content: center;
            }

            .widget-actions {
                justify-content: center;
            }
        }
        ';
    }
}