<?php
/**
 * Editor Sidebar Handler
 *
 * Handles Block Editor sidebar integration and asset loading.
 *
 * @package ProRank\SEO\Admin
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;

/**
 * EditorSidebar class
 */
class EditorSidebar {
    
    /**
     * Plugin instance
     *
     * @var Plugin
     */
    private Plugin $plugin;
    
    /**
     * Constructor
     *
     * @param Plugin $plugin Plugin instance.
     */
    public function __construct(Plugin $plugin) {
        $this->plugin = $plugin;
    }
    
    /**
     * Initialize sidebar
     *
     * @return void
     */
    public function init(): void {
        // Enqueue block editor assets
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_sidebar_assets']);
    }
    
    /**
     * Enqueue sidebar assets
     *
     * @return void
     */
    public function enqueue_sidebar_assets(): void {
        // Only load on supported post types
        $current_screen = get_current_screen();
        if (!$current_screen || !$current_screen->is_block_editor()) {
            return;
        }
        
        // Get supported post types (default to post and page)
        $supported_post_types = apply_filters(
            'prorank_seo_sidebar_post_types',
            ['post', 'page']
        );
        
        if (!in_array($current_screen->post_type, $supported_post_types, true)) {
            return;
        }
        
        // Check if script file exists - try build directory first, then legacy path
        $script_path = PRORANK_PLUGIN_DIR . 'build/sidebar.js';
        $asset_path = PRORANK_PLUGIN_DIR . 'build/sidebar.asset.php';
        $script_url = PRORANK_PLUGIN_URL . 'build/sidebar.js';

        // Fallback to legacy path if build doesn't exist
        if (!file_exists($script_path)) {
            $script_path = PRORANK_PLUGIN_DIR . 'admin/assets/js/sidebar.js';
            $asset_path = PRORANK_PLUGIN_DIR . 'admin/assets/js/sidebar.asset.php';
            $script_url = PRORANK_PLUGIN_URL . 'admin/assets/js/sidebar.js';
        }

        if (!file_exists($script_path)) {
            return;
        }

        // Get script dependencies and version
        $script_asset = file_exists($asset_path)
            ? require($asset_path)
            : [
                'dependencies' => ['wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-data', 'wp-i18n', 'wp-editor', 'lodash'],
                'version' => PRORANK_VERSION
            ];

        // Enqueue sidebar script
        wp_enqueue_script(
            'prorank-seo-sidebar',
            $script_url,
            $script_asset['dependencies'],
            $script_asset['version'],
            true
        );
        
        // Enqueue sidebar CSS if available
        $css_path = PRORANK_PLUGIN_DIR . 'build/sidebar.css';
        if (file_exists($css_path)) {
            wp_enqueue_style(
                'prorank-seo-sidebar',
                PRORANK_PLUGIN_URL . 'build/sidebar.css',
                [],
                $script_asset['version']
            );
        }

        // Enqueue EditorSidebar CSS (contains HeadlineAnalyzer panel styles)
        $editor_css_path = PRORANK_PLUGIN_DIR . 'build/EditorSidebar.css';
        if (file_exists($editor_css_path)) {
            wp_enqueue_style(
                'prorank-seo-editor-sidebar',
                PRORANK_PLUGIN_URL . 'build/EditorSidebar.css',
                ['prorank-seo-sidebar'],
                $script_asset['version']
            );
        }

        // Localize script
        wp_localize_script('prorank-seo-sidebar', 'proRankSidebar', [
            'apiUrl' => home_url('/wp-json/prorank-seo/v1'),
            'nonce' => wp_create_nonce('wp_rest'),
            'postId' => get_the_ID(),
            'postType' => $current_screen->post_type,
            'licenseTier' => 'free',
            'settings' => $this->get_sidebar_settings(),
        ]);

        // Compatibility: ensure window.prorankSettings exists on editor screens so
        // tier checks don't mis-detect "pro" when the object is missing.
        $inline = 'window.prorankSettings = window.prorankSettings || {};';
        $inline .= 'window.prorankSettings.adminUrl = ' . wp_json_encode(admin_url()) . ';';
        $inline .= 'window.prorankSettings.license = window.prorankSettings.license || {};';
        $inline .= 'window.prorankSettings.license.tier = ' . wp_json_encode('free') . ';';
        wp_add_inline_script('prorank-seo-sidebar', $inline, 'before');
        
        // Set script translations
        wp_set_script_translations(
            'prorank-seo-sidebar',
            'prorank-seo',
            PRORANK_PLUGIN_DIR . 'languages'
        );
        
    }
    
    /**
     * Get current license tier
     *
     * @return string License tier.
     */
    private function get_license_tier(): string {
        return 'free';
    }
    
    /**
     * Get sidebar settings
     *
     * @return array Sidebar settings.
     */
    private function get_sidebar_settings(): array {
        $defaults = [
            'defaultPanelState' => [
                'readability' => true,
                'internalLinking' => false,
                'advancedAnalysis' => false,
            ],
            'features' => [
                'readability' => true,
                'internalLinking' => true,
                'advancedAnalysis' => false,
            ],
        ];

        $settings = [];
        try {
            $settings = $this->plugin->settings()->get_settings('editor_sidebar');
        } catch (\Throwable $e) {
            $settings = [];
        }

        if (!is_array($settings)) {
            $settings = [];
        }

        $merged = array_replace_recursive($defaults, $settings);
        $merged['features']['advancedAnalysis'] = false;
        $merged = apply_filters('prorank_seo_sidebar_settings', $merged, $this->get_license_tier());

        return is_array($merged) ? $merged : $defaults;
    }
}
