<?php
/**
 * Media Enhancements
 *
 * Adds ProRank SEO features to media library and attachment details
 *
 * @package ProRank\SEO\Admin
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;

/**
 * MediaEnhancements class
 */
class MediaEnhancements {
    
    /**
     * Plugin instance
     *
     * @var Plugin
     */
    private Plugin $plugin;
    
    /**
     * Constructor
     *
     * @param Plugin $plugin Plugin instance.
     */
    public function __construct(Plugin $plugin) {
        $this->plugin = $plugin;
        $this->init();
    }
    
    /**
     * Initialize media enhancements
     */
    private function init(): void {
        // Only load in admin
        if (!is_admin()) {
            return;
        }
        
        // Enqueue scripts for media library
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        
        // Add data to localize script
        add_filter('prorank_seo_admin_data', [$this, 'add_media_data']);
        
        // Debug module status on admin init
        add_action('admin_init', [$this, 'debug_module_status']);
    }
    
    /**
     * Enqueue media enhancement scripts
     *
     * @param string $hook_suffix Current admin page.
     */
    public function enqueue_scripts(string $hook_suffix): void {
        // Only load on pages that use media
        $media_pages = [
            'post.php',
            'post-new.php',
            'upload.php',
            'media-upload.php',
        ];
        
        if (!in_array($hook_suffix, $media_pages, true)) {
            return;
        }
        
        // Check if media features should be shown
        if (!MediaModuleHelper::should_show_media_features()) {
            return;
        }
        
        // Media enhancements are part of the admin bundle
        // The script will be included when admin.js is loaded
        wp_enqueue_media();
    }
    
    /**
     * Add media-related data to admin localization
     *
     * @param array $data Existing admin data.
     * @return array Modified admin data.
     */
    public function add_media_data(array $data): array {
        $modules = $this->plugin->modules();
        $license = $this->plugin->license();
        
        // Add module status using helper
        if (!isset($data['modules'])) {
            $data['modules'] = [];
        }
        
        // Merge with media module status
        $media_status = MediaModuleHelper::get_media_module_status();
        $data['modules'] = array_merge($data['modules'], $media_status);
        
        // In developer mode, force enable image optimization
        if ($license->is_developer_mode_active()) {
            $data['modules']['image-optimization'] = true;
            $data['modules']['image_optimization'] = true;
            $data['modules']['performance'] = true;
            $data['modules']['images'] = true;
        }
        
        // Add AI configuration status
        if ($modules->is_module_enabled('ai-image-attributes')) {
            $settings = $this->plugin->settings();
            $ai_provider = $settings->get('ai_config_provider');
            $ai_key = $settings->get("ai_config_{$ai_provider}_api_key");
            
            $data['ai'] = [
                'configured' => !empty($ai_provider) && !empty($ai_key),
                'provider' => $ai_provider,
            ];
        }
        
        return $data;
    }
    
}
