<?php
/**
 * Media Module Helper
 *
 * Ensures media modules are properly available
 *
 * @package ProRank\SEO\Admin
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;

/**
 * MediaModuleHelper class
 */
class MediaModuleHelper {
    
    /**
     * Check if media modules should be available
     *
     * @return bool
     */
    public static function should_show_media_features(): bool {
        $plugin = Plugin::get_instance();
        $modules = $plugin->modules();
        $license = $plugin->license();
        
        // In developer mode, always show
        if ($license->is_developer_mode_active()) {
            return true;
        }
        
        // Check if either module is enabled
        $ai_enabled = $modules->is_module_enabled('ai-image-attributes');
        $image_opt_enabled = $modules->is_module_enabled('image_optimization');
        
        return $ai_enabled || $image_opt_enabled;
    }
    
    /**
     * Get media module status for frontend
     *
     * @return array
     */
    public static function get_media_module_status(): array {
        $plugin = Plugin::get_instance();
        $modules = $plugin->modules();
        $license = $plugin->license();
        
        $ai_enabled = $modules->is_module_enabled('ai-image-attributes');
        $image_opt_enabled = $modules->is_module_enabled('image_optimization');
        
        // In developer mode, enable both
        if ($license->is_developer_mode_active()) {
            $ai_enabled = true;
            $image_opt_enabled = true;
        }
        
        return [
            'ai-image-attributes' => $ai_enabled,
            'image-optimization' => $image_opt_enabled,
            'image_optimization' => $image_opt_enabled,
            'developer_mode' => $license->is_developer_mode_active(),
        ];
    }
    
    /**
     * Ensure modules are registered
     * Called early to make sure modules exist before checking
     */
    public static function ensure_modules_registered(): void {
        // This will trigger module loading if not already done
        $plugin = Plugin::get_instance();
        $modules = $plugin->modules();
        
        // Log module registration status
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $all_modules = $modules->get_modules();
            $module_slugs = array_keys($all_modules);
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO Media Helper] Registered modules: ' . implode(', ', $module_slugs));
            }
        }
    }
}