<?php
/**
 * Admin Menu Handler
 *
 * Utilizes the MenuRegistry service to construct the WordPress admin menu
 * following the 11-item IA structure from ProRank_Combined_Reference_v1.md §3.1.
 *
 * @package ProRank\Admin
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;
use ProRank\SEO\Core\MenuRegistry;
use ProRank\SEO\Core\ModuleManager;
use ProRank\SEO\Admin\Pages\SupportAccountPage;

/**
 * MenuHandler class
 *
 * Registers admin menu items with conditional visibility
 */
class MenuHandler {

    /**
     * Plugin instance
     *
     * @var Plugin
     */
    private Plugin $plugin;

    /**
     * Support & Account page instance
     *
     * @var SupportAccountPage|null
     */
    private ?SupportAccountPage $support_account_page = null;



    /**
     * Constructor
     *
     * @param Plugin $plugin Plugin instance
     */
    public function __construct(Plugin $plugin) {
        $this->plugin = $plugin;

        // Initialize pages
        $this->support_account_page = new SupportAccountPage($this->plugin->license());
        add_action('admin_enqueue_scripts', [$this->support_account_page, 'enqueue_assets']);
    }

    /**
     * Initialize menu registration
     *
     * @return void
     */
    public function init(): void {
        // Handle OAuth callbacks early
        add_action('admin_init', [$this, 'handle_oauth_callbacks']);

        // Remove third-party admin notices on ProRank pages before they render
        add_action('current_screen', [$this, 'suppress_third_party_notices'], 0);

        // Hide third-party admin notices on ProRank pages
        add_action('admin_head', [$this, 'hide_third_party_notices'], 1);

        $menu_registry = $this->plugin->menus();

        // Register top-level ProRank SEO menu
        $menu_registry->register_top_level(
            'prorank-seo',
            'ProRank SEO Dashboard',
            'ProRank SEO',
            'manage_options',
            [$this, 'render_dashboard_page'],
            $this->get_menu_icon(),
            null, // No condition for main menu
            26
        );

        // 1. Dashboard (always visible as main page)
        // Register with same slug as parent to replace auto-generated submenu
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-seo', // Same as parent for first item
            'Dashboard',
            'Dashboard',
            'manage_options',
            [$this, 'render_dashboard_page'],
            null,
            0 // Position 0 to ensure it's first
        );

        // 2. On-Page SEO (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-on-page-seo',
            'On-Page SEO',
            'On-Page SEO',
            'manage_options',
            [$this, 'render_on_page_seo_page'],
            null, // No special conditions for free tier
            10
        );

        // Hidden: Bulk Edit page (only accessible via posts list bulk action)
        // Using 'prorank-seo' as parent but with empty menu title to hide from menu
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-bulk-edit',
            __('Bulk Edit SEO', 'prorank-seo'), // page_title (browser title)
            '', // menu_title empty = hidden from menu
            'edit_posts',
            [$this, 'render_bulk_edit_page'],
            null,
            999
        );

        // 3. Performance (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-performance',
            'Performance',
            'Performance',
            'manage_options',
            [$this, 'render_performance_page'],
            null, // Base features are free
            20
        );

        // 4. Technical SEO (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-technical-seo',
            'Technical SEO',
            'Technical SEO',
            'manage_options',
            [$this, 'render_technical_seo_page'],
            null, // Base features are free
            30
        );

        // 9. Site Audit & Tools (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-site-audit',
            'Site Audit & Tools',
            'Site Audit & Tools',
            'manage_options',
            [$this, 'render_site_audit_page'],
            null, // Base features are free
            80
        );

        // 11. Support & Account (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-support-account',
            'Support & Account',
            'Support & Account',
            'manage_options',
            [$this->support_account_page, 'render'],
            null, // Always visible
            100
        );

        // Actually register all menus with WordPress
        $menu_registry->register_menus();
    }

    /**
     * Get menu icon
     *
     * Returns the ProRank favicon as a data URI for WordPress admin menu.
     *
     * @return string
     */
    private function get_menu_icon(): string {
        $icon_path = PRORANK_PLUGIN_DIR . 'assets/prorank.svg';
        if ( file_exists( $icon_path ) ) {
            $svg = file_get_contents( $icon_path );
            if ( $svg !== false ) {
                return 'data:image/svg+xml;base64,' . base64_encode( $svg );
            }
        }

        // Fallback: simple hexagonal icon.
        $fallback = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none"><path d="M12 2L4 6.5V17.5L12 22L20 17.5V6.5L12 2Z" fill="#a7aaad" fill-opacity="0.15" stroke="#a7aaad" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';
        return 'data:image/svg+xml;base64,' . base64_encode( $fallback );
    }

    /**
     * Render dashboard page
     *
     * @return void
     */
    public function render_dashboard_page(): void {
        echo '<div id="prorank-seo-dashboard" class="wrap prorank-seo-admin-page">';
        echo '<h1>' . esc_html('ProRank SEO Dashboard') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render On-Page SEO page
     *
     * @return void
     */
    public function render_on_page_seo_page(): void {
        echo '<div id="prorank-seo-on-page" class="wrap prorank-seo-admin-page prorank-on-page-seo-page">';
        echo '<h1>' . esc_html('On-Page SEO') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Performance page
     *
     * @return void
     */
    public function render_performance_page(): void {
        echo '<div id="prorank-seo-performance" class="wrap prorank-seo-admin-page">';
        echo '<h1>' . esc_html('Performance') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Technical SEO page
     *
     * @return void
     */
    public function render_technical_seo_page(): void {
        echo '<div id="prorank-seo-technical" class="wrap prorank-seo-admin-page prorank-technical-seo-page">';
        echo '<h1>' . esc_html('Technical SEO') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';

        // Debug: Check if scripts are enqueued
        global $wp_scripts;
        if (isset($wp_scripts->registered['prorank-seo-admin'])) {
            echo '<!-- ProRank SEO Admin script is registered -->';
        } else {
            echo '<!-- WARNING: ProRank SEO Admin script is NOT registered -->';
        }
        echo '</div>';
    }


    /**
     * Handle OAuth callbacks early in admin_init
     *
     * Note: OAuth callbacks are only used by premium Analytics feature.
     * This method is kept as a stub for compatibility.
     *
     * @return void
     */
    public function handle_oauth_callbacks(): void {
        // OAuth callbacks are premium-only (Analytics feature)
        // No action needed in free version
    }

    /**
     * Render Site Audit & Tools page
     *
     * @return void
     */
    public function render_site_audit_page(): void {
        echo '<div id="prorank-seo-site-audit" class="wrap prorank-seo-admin-page prorank-site-audit-page">';

        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Support & Account page
     *
     * @return void
     */
    public function render_support_page(): void {
        echo '<div id="prorank-seo-support" class="wrap prorank-seo-admin-page">';
        echo '<h1>' . esc_html('Support & Account') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Import/Export page
     *
     * @return void
     */
    public function render_import_export_page(): void {
        // Always render the React container - React handles locked state per feature
        echo '<div id="prorank-seo-import-export" class="wrap prorank-seo-admin-page prorank-import-export-page">';
        echo '<div class="prorank-admin-container" id="prorank-admin-root"></div>';
        echo '</div>';
    }

    /**
     * Render Bulk Edit SEO page
     *
     * @return void
     */
    public function render_bulk_edit_page(): void {
        // Get post IDs from URL
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only URL args.
        $post_ids_raw = isset($_GET['post_ids']) ? sanitize_text_field( wp_unslash( $_GET['post_ids'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only URL args.
        $post_type = isset($_GET['post_type']) ? sanitize_key( wp_unslash( $_GET['post_type'] ) ) : 'post';
        $post_ids = array_filter(array_map('intval', explode(',', $post_ids_raw)));

        echo '<div class="wrap prorank-seo-admin-page prorank-bulk-edit-page">';
        echo '<h1>' . esc_html__('Bulk Edit SEO', 'prorank-seo') . '</h1>';

        if (empty($post_ids)) {
            echo '<div class="notice notice-warning"><p>' . esc_html__('No posts selected. Please select posts from the posts list and use the "Edit SEO Settings" bulk action.', 'prorank-seo') . '</p></div>';
            echo '<p><a href="' . esc_url(admin_url('edit.php?post_type=' . $post_type)) . '" class="button">' . esc_html__('Go to Posts', 'prorank-seo') . '</a></p>';
            echo '</div>';
            return;
        }

        // Handle form submission
        if (isset($_POST['prorank_bulk_edit_submit']) && check_admin_referer('prorank_bulk_edit_nonce', 'prorank_bulk_edit_nonce')) {
            $updated = 0;
            foreach ($post_ids as $post_id) {
                if (!current_user_can('edit_post', $post_id)) {
                    continue;
                }
                $title_key = 'prorank_seo_title_' . $post_id;
                $desc_key = 'prorank_seo_desc_' . $post_id;
                $keyword_key = 'prorank_focus_keyword_' . $post_id;

                if (isset($_POST[$title_key])) {
                    update_post_meta($post_id, '_prorank_seo_title', sanitize_text_field( wp_unslash( $_POST[$title_key] ) ));
                }
                if (isset($_POST[$desc_key])) {
                    update_post_meta($post_id, '_prorank_seo_description', sanitize_textarea_field( wp_unslash( $_POST[$desc_key] ) ));
                }
                if (isset($_POST[$keyword_key])) {
                    update_post_meta($post_id, '_prorank_seo_focus_keyword', sanitize_text_field( wp_unslash( $_POST[$keyword_key] ) ));
                }
                $updated++;
            }
            /* translators: %d: number of posts updated */
            echo '<div class="notice notice-success is-dismissible"><p>' . sprintf(esc_html__('Updated SEO settings for %d posts.', 'prorank-seo'), (int) $updated) . '</p></div>';
        }

        /* translators: %d: number of posts being edited */
        echo '<p>' . sprintf(esc_html__('Editing SEO settings for %d posts.', 'prorank-seo'), count($post_ids)) . ' ';
        echo '<a href="' . esc_url(admin_url('edit.php?post_type=' . $post_type)) . '">' . esc_html__('Back to posts list', 'prorank-seo') . '</a></p>';

        echo '<form method="post">';
        wp_nonce_field('prorank_bulk_edit_nonce', 'prorank_bulk_edit_nonce');

        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th style="width:25%">' . esc_html__('Post Title', 'prorank-seo') . '</th>';
        echo '<th style="width:15%">' . esc_html__('Focus Keyword', 'prorank-seo') . '</th>';
        echo '<th style="width:25%">' . esc_html__('SEO Title', 'prorank-seo') . ' <small>(60 chars)</small></th>';
        echo '<th style="width:35%">' . esc_html__('Meta Description', 'prorank-seo') . ' <small>(160 chars)</small></th>';
        echo '</tr></thead>';
        echo '<tbody>';

        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if (!$post) continue;

            $seo_title = get_post_meta($post_id, '_prorank_seo_title', true);
            $seo_desc = get_post_meta($post_id, '_prorank_seo_description', true);
            $focus_keyword = get_post_meta($post_id, '_prorank_seo_focus_keyword', true);

            echo '<tr>';
            echo '<td><strong><a href="' . esc_url(get_edit_post_link($post_id)) . '">' . esc_html($post->post_title) . '</a></strong></td>';
            echo '<td><input type="text" name="prorank_focus_keyword_' . esc_attr($post_id) . '" value="' . esc_attr($focus_keyword) . '" style="width:100%"></td>';
            echo '<td><input type="text" name="prorank_seo_title_' . esc_attr($post_id) . '" value="' . esc_attr($seo_title) . '" maxlength="70" style="width:100%"></td>';
            echo '<td><textarea name="prorank_seo_desc_' . esc_attr($post_id) . '" rows="2" maxlength="200" style="width:100%">' . esc_textarea($seo_desc) . '</textarea></td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
        echo '<p class="submit"><input type="submit" name="prorank_bulk_edit_submit" class="button button-primary" value="' . esc_attr__('Save All Changes', 'prorank-seo') . '"></p>';
        echo '</form>';
        echo '</div>';
    }

    /**
     * Hide third-party admin notices on ProRank SEO pages
     *
     * Prevents other plugins/themes from displaying their notices
     * on top of our plugin interface for a cleaner UX.
     *
     * @return void
     */
    public function hide_third_party_notices(): void {
        // Only run on ProRank admin pages
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only page check.
        if (!isset($_GET['page']) || strpos( sanitize_text_field( wp_unslash( $_GET['page'] ) ), 'prorank') !== 0) {
            return;
        }

        // Output CSS to hide all admin notices except our own
        echo '<style>
            /* Hide all third-party admin notices on ProRank pages */
            .prorank-seo-admin-page ~ .notice:not(.prorank-notice),
            .prorank-seo-admin-page ~ .update-nag,
            .prorank-seo-admin-page ~ .updated:not(.prorank-notice),
            .prorank-seo-admin-page ~ .error:not(.prorank-notice),
            .prorank-seo-admin-page ~ .notice-warning:not(.prorank-notice),
            .prorank-seo-admin-page ~ .notice-error:not(.prorank-notice),
            .prorank-seo-admin-page ~ .notice-info:not(.prorank-notice),
            .prorank-seo-admin-page ~ .notice-success:not(.prorank-notice),
            #wpbody-content > .notice:not(.prorank-notice),
            #wpbody-content > .update-nag,
            #wpbody-content > .updated:not(.prorank-notice),
            #wpbody-content > .error:not(.prorank-notice),
            #wpbody-content > div.notice:not(.prorank-notice),
            /* Specifically hide common plugin/theme notices */
            .td-admin-notice,
            .tagdiv-admin-notice,
            [class*="td-"][class*="notice"],
            /* Generic third-party notice patterns */
            .wrap > .notice:not(.prorank-notice):not(:has(+ .prorank-seo-admin-page)):first-child,
            .wrap > .update-nag:first-child,
            /* Hide notices that appear before our content */
            body.prorank-seo_page_prorank-seo #wpbody-content > .notice:not(.prorank-notice),
            body.prorank-seo_page_prorank-performance #wpbody-content > .notice:not(.prorank-notice),
            body.prorank-seo_page_prorank-on-page-seo #wpbody-content > .notice:not(.prorank-notice),
            body.prorank-seo_page_prorank-technical-seo #wpbody-content > .notice:not(.prorank-notice),
            body.prorank-seo_page_prorank-site-audit #wpbody-content > .notice:not(.prorank-notice),
            body.prorank-seo_page_prorank-support-account #wpbody-content > .notice:not(.prorank-notice),
            body[class*="prorank-seo_page_prorank"] #wpbody-content > .notice:not(.prorank-notice),
            body[class*="prorank-seo_page_prorank"] #wpbody-content > .update-nag,
            body[class*="prorank-seo_page_prorank"] #wpbody-content > .updated:not(.prorank-notice),
            body[class*="prorank-seo_page_prorank"] #wpbody-content > .error:not(.prorank-notice) {
                display: none !important;
            }
        </style>';
    }

    /**
     * Suppress third-party admin notices on ProRank pages.
     *
     * @param \WP_Screen|null $screen Current screen (provided by hook).
     * @return void
     */
    public function suppress_third_party_notices($screen = null): void {
        if (!$this->is_prorank_screen($screen)) {
            return;
        }

        $hooks = [
            'admin_notices',
            'all_admin_notices',
            'network_admin_notices',
            'user_admin_notices',
        ];

        foreach ($hooks as $hook) {
            $this->remove_non_prorank_callbacks($hook);
        }
    }

    /**
     * Check whether the current admin screen is a ProRank page.
     *
     * @param \WP_Screen|null $screen
     * @return bool
     */
    private function is_prorank_screen($screen = null): bool {
        if (!$screen && function_exists('get_current_screen')) {
            $screen = get_current_screen();
        }

        if ($screen && isset($screen->id) && strpos((string) $screen->id, 'prorank') !== false) {
            return true;
        }

        if ($screen && isset($screen->base) && strpos((string) $screen->base, 'prorank') !== false) {
            return true;
        }

        return false;
    }

    /**
     * Remove non-ProRank callbacks from a notice hook.
     *
     * @param string $hook
     * @return void
     */
    private function remove_non_prorank_callbacks(string $hook): void {
        global $wp_filter;

        if (empty($wp_filter[$hook]) || !($wp_filter[$hook] instanceof \WP_Hook)) {
            return;
        }

        $to_remove = [];

        foreach ($wp_filter[$hook]->callbacks as $priority => $callbacks) {
            foreach ($callbacks as $callback) {
                $function = $callback['function'] ?? null;
                if (!$this->is_prorank_callback($function)) {
                    $to_remove[] = [$hook, $function, $priority];
                }
            }
        }

        foreach ($to_remove as $removal) {
            [$hook_name, $function, $priority] = $removal;
            if ($function !== null) {
                remove_action($hook_name, $function, $priority);
            }
        }
    }

    /**
     * Determine whether a callback belongs to ProRank.
     *
     * @param mixed $callback
     * @return bool
     */
    private function is_prorank_callback($callback): bool {
        if ($callback === null) {
            return false;
        }

        if (is_string($callback)) {
            return strpos($callback, 'prorank') !== false;
        }

        if (is_array($callback)) {
            $target = $callback[0] ?? null;
            $class = '';
            if (is_object($target)) {
                $class = get_class($target);
            } elseif (is_string($target)) {
                $class = $target;
            }

            if ($class !== '' && strpos($class, 'ProRank\\SEO\\') === 0) {
                return true;
            }

            return $this->is_prorank_reflection_target($target);
        }

        if ($callback instanceof \Closure) {
            return $this->is_prorank_reflection_target($callback);
        }

        if (is_object($callback) && method_exists($callback, '__invoke')) {
            return $this->is_prorank_reflection_target($callback);
        }

        return false;
    }

    /**
     * Check callback file location to confirm it belongs to ProRank.
     *
     * @param mixed $target
     * @return bool
     */
    private function is_prorank_reflection_target($target): bool {
        try {
            if ($target instanceof \Closure) {
                $ref = new \ReflectionFunction($target);
                return $this->is_prorank_path($ref->getFileName());
            }

            if (is_object($target)) {
                $ref = new \ReflectionClass($target);
                return $this->is_prorank_path($ref->getFileName());
            }

            if (is_string($target) && class_exists($target)) {
                $ref = new \ReflectionClass($target);
                return $this->is_prorank_path($ref->getFileName());
            }
        } catch (\ReflectionException $e) {
            return false;
        }

        return false;
    }

    /**
     * Determine if a file path belongs to ProRank plugin directories.
     *
     * @param string|null $file
     * @return bool
     */
    private function is_prorank_path(?string $file): bool {
        if (!$file) {
            return false;
        }

        $path = str_replace('\\', '/', $file);
        $markers = [
            '/prorank-seo/',
            '/prorank-seo-premium/',
            '/free/',
            '/premium/',
        ];

        foreach ($markers as $marker) {
            if (strpos($path, $marker) !== false) {
                return true;
            }
        }

        return false;
    }


}
