<?php
/**
 * Meta Box Handler
 *
 * Handles the SEO meta box in the post editor.
 *
 * @package ProRank\SEO\Admin
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;

/**
 * MetaBox class
 */
class MetaBox {
    /**
     * Plugin instance
     *
     * @var Plugin
     */
    private Plugin $plugin;

    /**
     * Constructor
     *
     * @param Plugin $plugin Plugin instance.
     */
    public function __construct(Plugin $plugin) {
        $this->plugin = $plugin;
    }

    /**
     * Initialize meta box
     *
     * @return void
     */
    public function init(): void {
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Add meta box to post editor
     *
     * @return void
     */
    public function add_meta_box(): void {
        $screen = get_current_screen();
        
        // Only add to post edit screens
        if (!$screen || $screen->base !== 'post') {
            return;
        }

        // Get publicly queryable post types
        $post_types = get_post_types(['public' => true], 'names');
        
        add_meta_box(
            'prorank-seo-meta-box',
            __('ProRank SEO', 'prorank-seo'),
            [$this, 'render_meta_box'],
            $post_types,
            'normal',
            'high'
        );
    }

    /**
     * Render meta box content
     *
     * @param \WP_Post $post Current post object.
     * @return void
     */
    public function render_meta_box(\WP_Post $post): void {
        // Output container for React app
        echo '<div id="prorank-seo-metabox-root"></div>';
        
        // Add nonce field for security
        wp_nonce_field('prorank_seo_metabox', 'prorank_seo_metabox_nonce');
    }

    /**
     * Enqueue meta box assets
     *
     * @param string $hook Current admin page hook.
     * @return void
     */
    public function enqueue_assets(string $hook): void {
        // Only load on post edit screens
        if (!in_array($hook, ['post.php', 'post-new.php'], true)) {
            return;
        }

        $screen = get_current_screen();
        if (!$screen || $screen->base !== 'post') {
            return;
        }

        // Check if we're on a supported post type
        $post_type = $screen->post_type;
        $public_post_types = get_post_types(['public' => true], 'names');
        
        if (!in_array($post_type, $public_post_types, true)) {
            return;
        }

        // Enqueue meta box script from build directory
        $script_url = PRORANK_PLUGIN_URL . 'build/meta-box.js';

        // Get version from asset file if available
        $asset_file = PRORANK_PLUGIN_DIR . 'build/meta-box.asset.php';
        $asset_data = file_exists($asset_file) ? require $asset_file : [];
        $dependencies = $asset_data['dependencies'] ?? ['wp-element', 'wp-components', 'wp-i18n', 'wp-data', 'wp-api-fetch', 'wp-dom-ready', 'lodash'];
        $version = $asset_data['version'] ?? PRORANK_VERSION;

        wp_enqueue_script(
            'prorank-seo-meta-box',
            $script_url,
            $dependencies,
            $version,
            true
        );

        // Enqueue meta box CSS
        $css_url = PRORANK_PLUGIN_URL . 'build/meta-box.css';
        wp_enqueue_style(
            'prorank-seo-meta-box',
            $css_url,
            [],
            $version
        );

        // Get current post
        global $post;
        $post_id = $post ? $post->ID : 0;

        $license_tier = 'free';

        // Localize script data
        wp_localize_script(
            'prorank-seo-meta-box',
            'prorankMetaboxData',
            [
                'licenseTier' => $license_tier,
                'postId' => $post_id,
                'nonce' => wp_create_nonce('prorank_seo_metabox'),
                'apiUrl' => rest_url('prorank-seo/v1'),
            ]
        );

        // Compatibility + safety: some bundles rely on window.prorankSettings being present.
        // Without it, checks like `window.prorankSettings?.license?.tier !== 'free'` become
        // true when the object is missing, which wrongly enables Pro-only UI and triggers 404s.
        $inline = 'window.prorankSettings = window.prorankSettings || {};';
        $inline .= 'window.prorankSettings.adminUrl = ' . wp_json_encode(admin_url()) . ';';
        $inline .= 'window.prorankSettings.license = window.prorankSettings.license || {};';
        $inline .= 'window.prorankSettings.license.tier = ' . wp_json_encode($license_tier) . ';';
        wp_add_inline_script('prorank-seo-meta-box', $inline, 'before');

        // Set script translations
        wp_set_script_translations(
            'prorank-seo-meta-box',
            'prorank-seo',
            PRORANK_PLUGIN_DIR . 'languages'
        );
    }
}
