<?php
/**
 * Onboarding functionality
 *
 * @package ProRank\SEO\Admin
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\SettingsManager;

/**
 * Handles plugin onboarding
 */
class Onboarding {
    
    /**
     * Settings manager instance
     *
     * @var SettingsManager|null
     */
    private ?SettingsManager $settings_manager = null;
    
    /**
     * Constructor
     *
     * @param SettingsManager|null $settings_manager Settings manager instance
     */
    public function __construct(?SettingsManager $settings_manager = null) {
        $this->settings_manager = $settings_manager;
    }
    
    /**
     * Initialize onboarding
     *
     * @return void
     */
    public function init(): void {
        // Check for activation redirect
        add_action('admin_init', [$this, 'handle_activation_redirect']);
        
        // Enqueue onboarding assets if needed
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }
    
    /**
     * Handle activation redirect
     *
     * @return void
     */
    public function handle_activation_redirect(): void {
        // Check if we should redirect
        if (!get_transient('prorank_activation_redirect')) {
            return;
        }

        // Delete the redirect transient
        delete_transient('prorank_activation_redirect');
        
        // Don't redirect on network or bulk activation
        if (is_network_admin() || filter_input(INPUT_GET, 'activate-multi') !== null) {
            return;
        }
        
        // Don't redirect if user doesn't have capability
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Don't redirect during AJAX requests
        if (wp_doing_ajax()) {
            return;
        }
        
        // Check if onboarding was already completed
        if ($this->settings_manager && $this->settings_manager->get_setting('onboarding', 'completed', false)) {
            return;
        }
        
        // Set flag to show onboarding modal
        update_option('prorank_show_onboarding_modal', true);
    }
    
    /**
     * Enqueue onboarding assets
     *
     * @return void
     */
    public function enqueue_assets(): void {
        // Only enqueue if we should show the modal
        $show_modal = get_option('prorank_show_onboarding_modal');
        
        if (!$show_modal) {
            return;
        }
        
        // Only on admin pages
        if (!is_admin()) {
            return;
        }
        
        // Don't load during AJAX requests
        if (wp_doing_ajax()) {
            return;
        }
        
        // Check user capability
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Enqueue onboarding script (webpack/wp-scripts build outputs to build/)
        $built_file_path = PRORANK_PLUGIN_DIR . 'build/onboarding.js';
        if (!file_exists($built_file_path)) {
            return;
        }

        $asset_path = PRORANK_PLUGIN_DIR . 'build/onboarding.asset.php';
        $asset = file_exists($asset_path)
            ? require $asset_path
            : [
                'dependencies' => ['wp-element', 'wp-i18n', 'wp-components', 'wp-api-fetch'],
                'version' => PRORANK_VERSION,
            ];

        // Enqueue the onboarding script
        wp_enqueue_script(
            'prorank-seo-onboarding',
            PRORANK_PLUGIN_URL . 'build/onboarding.js',
            $asset['dependencies'] ?? ['wp-element', 'wp-i18n', 'wp-components', 'wp-api-fetch'],
            $asset['version'] ?? PRORANK_VERSION,
            true
        );
        
        // Localize script
        wp_localize_script('prorank-seo-onboarding', 'prorankSEOOnboarding', [
            'apiUrl' => rest_url(),
            'nonce' => wp_create_nonce('wp_rest'),
        ]);
        
        // Set script translations
        wp_set_script_translations('prorank-seo-onboarding', 'prorank-seo');

        // Optional: enqueue onboarding CSS if present in build/ (some builds only output JS)
        $css_path = PRORANK_PLUGIN_DIR . 'build/onboarding.css';
        if (file_exists($css_path)) {
            wp_enqueue_style(
                'prorank-seo-onboarding',
                PRORANK_PLUGIN_URL . 'build/onboarding.css',
                [],
                $asset['version'] ?? PRORANK_VERSION
            );
        }
    }
}
