<?php
declare(strict_types=1);

/**
 * Support & Account Page
 *
 * @package ProRank\Admin\Pages
 */

namespace ProRank\SEO\Admin\Pages;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\LicenseManager;
use ProRank\SEO\Core\MenuRegistry;

/**
 * Support & Account page loader
 */
class SupportAccountPage {
    
    /**
     * License Manager instance
     *
     * @var LicenseManager
     */
    private LicenseManager $license_manager;
    
    /**
     * Constructor
     *
     * @param LicenseManager $license_manager
     */
    public function __construct(LicenseManager $license_manager) {
        $this->license_manager = $license_manager;
        
        // Handle license reset request
        add_action('admin_init', [$this, 'handle_license_reset']);
    }
    
    /**
     * Handle license reset request
     *
     * @return void
     */
    public function handle_license_reset(): void {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified below.
        if (!isset($_GET['prorank_reset_license']) || sanitize_text_field( wp_unslash( $_GET['prorank_reset_license'] ) ) !== '1') {
            return;
        }
        
        // Verify nonce
        $nonce = isset($_GET['_wpnonce']) ? sanitize_key( wp_unslash( $_GET['_wpnonce'] ) ) : '';
        if (!wp_verify_nonce($nonce, 'prorank_reset_license')) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Reset and reactivate license
        $result = $this->license_manager->reset_and_reactivate();
        
        if (is_wp_error($result)) {
            set_transient('prorank_admin_notice', [
                'type' => 'error',
                'message' => $result->get_error_message(),
            ], 30);
        } else {
            set_transient('prorank_admin_notice', [
                'type' => 'success',
                'message' => __('License has been reset and reactivated successfully.', 'prorank-seo'),
            ], 30);
        }
        
        // Redirect back to support page
        wp_safe_redirect(admin_url('admin.php?page=prorank-support-account&license_reset=1'));
        exit;
    }
    
    /**
     * Register page
     *
     * @param MenuRegistry $menu_registry
     * @return void
     */
    public function register(MenuRegistry $menu_registry): void {
        // Page is already registered in MenuHandler
        // This method is kept for interface compatibility
    }
    
    /**
     * Render page
     *
     * @return void
     */
    public function render(): void {
        // Hide default WordPress admin notices for this page
        ?>
        <style>
            /* Hide WordPress admin notices that appear above our custom header */
            .prorank-seo_page_prorank-support-account .notice:not(.prorank-notice),
            .prorank-seo_page_prorank-support-account .error:not(.prorank-notice),
            .prorank-seo_page_prorank-support-account .updated:not(.prorank-notice),
            .prorank-seo_page_prorank-support-account .update-nag {
                display: none !important;
            }
            
            /* Ensure our React app takes full width */
            .prorank-seo_page_prorank-support-account #wpcontent {
                padding-left: 0;
            }
            
            .prorank-seo_page_prorank-support-account #wpbody-content {
                padding-bottom: 0;
            }
            
            .prorank-seo_page_prorank-support-account .wrap {
                margin: 0;
            }
        </style>
        <div id="prorank-admin-root" data-page="support-account" class="wrap prorank-admin-wrapper"></div>
        <?php
    }
    
    /**
     * Enqueue assets
     *
     * @param string $hook_suffix
     * @return void
     */
    public function enqueue_assets(string $hook_suffix): void {
        if (!str_contains($hook_suffix, 'prorank-support-account')) {
            return;
        }
        
        // Enqueue main admin app (Support & Account is routed in admin.js)
        $asset_file = PRORANK_PLUGIN_DIR . 'build/admin.asset.php';
        
        if (!file_exists($asset_file)) {
            return;
        }
        
        $asset = require $asset_file;
        
        wp_enqueue_script(
            'prorank-admin',
            PRORANK_PLUGIN_URL . 'build/admin.js',
            $asset['dependencies'],
            $asset['version'],
            true
        );

        // Enqueue main CSS (Vite uses index/main, webpack uses admin-support-account.css)
        $main_css = PRORANK_PLUGIN_DIR . 'build/main.css';
        $index_css = PRORANK_PLUGIN_DIR . 'build/index.css';
        $support_css = PRORANK_PLUGIN_DIR . 'build/admin-support-account.css';

        $loaded_main_css = false;

        if (file_exists($main_css)) {
            wp_enqueue_style(
                'prorank-admin',
                PRORANK_PLUGIN_URL . 'build/main.css',
                ['wp-components'],
                filemtime($main_css)
            );
            $loaded_main_css = true;
        } elseif (file_exists($index_css)) {
            wp_enqueue_style(
                'prorank-admin',
                PRORANK_PLUGIN_URL . 'build/index.css',
                ['wp-components'],
                filemtime($index_css)
            );
            $loaded_main_css = true;
        } elseif (file_exists($support_css)) {
            wp_enqueue_style(
                'prorank-admin',
                PRORANK_PLUGIN_URL . 'build/admin-support-account.css',
                ['wp-components'],
                filemtime($support_css)
            );
        }

        // Get build fingerprint
        $build_fingerprint = defined('PRORANK_BUILD_FINGERPRINT') ? PRORANK_BUILD_FINGERPRINT : 'development';
        
        // Check for license notices
        $notices = [];
        
        // Fingerprint error is now handled by license-reset-notice.php
        
        // Check for reset success
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only notice display.
        if (isset($_GET['license_reset']) && sanitize_text_field( wp_unslash( $_GET['license_reset'] ) ) === '1') {
            $notices[] = [
                'type' => 'success',
                'message' => __('License activation has been reset. Your license should now work properly.', 'prorank-seo'),
            ];
        }
        
        // Check for transient notices
        if (get_transient('prorank_license_reset_error')) {
            $notices[] = [
                'type' => 'error',
                'message' => get_transient('prorank_license_reset_error'),
            ];
            delete_transient('prorank_license_reset_error');
        }
        
        if (get_transient('prorank_license_reset_success')) {
            $notices[] = [
                'type' => 'success',
                'message' => __('Your license has been reactivated with the new server fingerprint.', 'prorank-seo'),
            ];
            delete_transient('prorank_license_reset_success');
        }
        
        // Get license data once
        $license_data = $this->license_manager->get_license_data();

        // Pass fingerprint error info to JavaScript
        $fingerprint_error = null;
        $last_error = $license_data['last_error'] ?? '';
        if ($last_error && strpos($last_error, 'fingerprint') !== false) {
            $fingerprint_error = [
                'message' => $last_error,
                'reset_url' => wp_nonce_url(
                    admin_url('admin.php?page=prorank-support-account&prorank_reset_license=1'),
                    'prorank_reset_license'
                ),
            ];
        }

        wp_localize_script('prorank-support-account', 'prorankSupport', [
            'apiUrl' => rest_url('prorank/v1'),
            'nonce' => wp_create_nonce('wp_rest'),
            'license' => $license_data,
            'pluginUrl' => PRORANK_PLUGIN_URL,
            'fingerprintError' => $fingerprint_error,
            'version' => PRORANK_VERSION,
            'buildFingerprint' => $build_fingerprint,
            'notices' => $notices,
            'supportLinks' => [
                'documentation' => 'https://docs.prorank.io',
                'feedback' => 'https://feedback.prorank.io',
                'changelog' => PRORANK_PLUGIN_URL . 'changelog.md',
            ],
        ]);

        // Additional CSS for support page (if generated separately)
        if ($loaded_main_css && file_exists($support_css)) {
            wp_enqueue_style(
                'prorank-support-account-extra',
                PRORANK_PLUGIN_URL . 'build/admin-support-account.css',
                ['prorank-support-account'],
                filemtime($support_css)
            );
        }
    }
}
