<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries throughout
/**
 * Site Audit Engine - Real Site Scanner
 *
 * @package ProRank\SEO\Core\Audits
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Audits;

defined( 'ABSPATH' ) || exit;

use WP_Error;
use Exception;

/**
 * Main audit engine for crawling and analyzing sites
 */
class SiteAuditEngine {
    
    /**
     * Audit ID
     */
    private string $audit_id;
    
    /**
     * Database tables
     */
    private string $audits_table;
    private string $audit_urls_table;
    private string $audit_issues_table;
    
    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->audits_table = $wpdb->prefix . 'prorank_audits';
        $this->audit_urls_table = $wpdb->prefix . 'prorank_audit_urls';
        $this->audit_issues_table = $wpdb->prefix . 'prorank_audit_issues';
        
        $this->maybe_create_tables();
    }
    
    /**
     * Start a new audit
     */
    public function start_audit(array $options = []): array {
        global $wpdb;
        
        try {
            // Generate audit ID
            $this->audit_id = 'audit_' . uniqid();
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: Starting audit with ID: ' . $this->audit_id);
            }
            
            // Insert audit record
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $result = $wpdb->insert($this->audits_table, [
                'audit_id' => $this->audit_id,
                'status' => 'crawling',
                'started_at' => current_time('mysql'),
                'options' => json_encode($options),
            ]);
            
            if ($result === false) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log('ProRank SEO: Failed to insert audit record. Error: ' . $wpdb->last_error);
                }
                throw new Exception('Failed to create audit record: ' . $wpdb->last_error);
            }

            // Run site-level checks (once per audit, not per-page)
            $this->run_site_level_checks($this->audit_id, $options);

            // Schedule crawler
            wp_schedule_single_event(time(), 'prorank_run_site_crawler', [$this->audit_id]);
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: Audit scheduled successfully');
            }
            
            // Run crawler immediately in the background
            wp_schedule_single_event(time() + 1, 'prorank_run_immediate_crawler', [$this->audit_id]);
            
            // Trigger cron immediately for faster response
            spawn_cron();
            
            // Also try to run crawler directly for immediate feedback
            if ((defined('DOING_AJAX') && DOING_AJAX) || (defined('REST_REQUEST') && REST_REQUEST)) {
                // If this is an AJAX/REST request, run crawler in same request
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log('ProRank SEO: Running crawler immediately in AJAX/REST context');
                }
                try {
                    $this->run_crawler($this->audit_id);
                } catch (Exception $e) {
                    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                        prorank_log('ProRank SEO: Error running crawler immediately: ' . $e->getMessage());
                    }
                }
            }
            
            return [
                'audit_id' => $this->audit_id,
                'status' => 'crawling',
                'message' => __('Site audit started. Crawling your site...', 'prorank-seo'),
            ];
        } catch (Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: Exception in start_audit: ' . $e->getMessage());
            }
            throw $e;
        }
    }
    
    /**
     * Run site-level checks (once per audit, not per-page)
     */
    private function run_site_level_checks(string $audit_id, array $options): void {
        $check_types = $options['check_types'] ?? [];
        $default_checks = [
            'xml_sitemap' => true,
            'wordpress_health' => true,
            'plugin_conflicts' => true,
            'database_optimization' => true,
        ];
        $enabled_checks = array_merge($default_checks, $check_types);

        // Create a pseudo URL entry for site-level issues
        global $wpdb;
        $site_url = home_url();

        // Insert site-level URL entry
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->insert($this->audit_urls_table, [
            'audit_id' => $audit_id,
            'url' => $site_url . '?_site_level_checks',
            'status' => 'pending',
            'depth' => 0,
            'created_at' => current_time('mysql'),
        ]);
        $site_url_id = $wpdb->insert_id;

        $issues = [];
        $passed_checks = [];
        $warnings = [];

        // XML Sitemap check
        if (!empty($enabled_checks['xml_sitemap'])) {
            $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_xml_sitemap());
        }

        // WordPress Health check
        if (!empty($enabled_checks['wordpress_health'])) {
            $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_wordpress_health());
        }

        // Plugin conflicts check
        if (!empty($enabled_checks['plugin_conflicts'])) {
            $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_plugin_conflicts());
        }

        // Database optimization check
        if (!empty($enabled_checks['database_optimization'])) {
            $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_database_optimization());
        }

        // Insert all findings
        $all_findings = array_merge($issues, $warnings);
        $current_time = current_time('mysql');

        foreach ($all_findings as $finding) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->insert($this->audit_issues_table, [
                'audit_id' => $audit_id,
                'url_id' => $site_url_id,
                'type' => $finding['type'],
                'severity' => $finding['severity'],
                'message' => $finding['message'],
                'data' => json_encode($finding['data'] ?? []),
                'created_at' => $current_time,
            ]);
        }

        foreach ($passed_checks as $check) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->insert($this->audit_issues_table, [
                'audit_id' => $audit_id,
                'url_id' => $site_url_id,
                'type' => $check,
                'severity' => 'passed',
                'message' => 'Check passed successfully',
                'data' => '{}',
                'created_at' => $current_time,
            ]);
        }

        // Mark site-level URL as checked
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update($this->audit_urls_table, [
            'status' => 'checked',
            'issues_count' => count($issues),
            'warnings_count' => count($warnings),
            'passed_count' => count($passed_checks),
            'checked_at' => current_time('mysql'),
        ], ['id' => $site_url_id]);
    }

    /**
     * Run the site crawler
     */
    public function run_crawler(string $audit_id): void {
        global $wpdb;
        
        $this->audit_id = $audit_id;
        
        try {
            // Get audit options
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $audit = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->audits_table} WHERE audit_id = %s",
                $audit_id
            ));
            
            $options = [];
            if ($audit && !empty($audit->options)) {
                $options = json_decode($audit->options, true) ?: [];
            }
            
            // Get settings from options or defaults - read from audit settings
            $settings = $options['settings'] ?? [];
            $max_urls = $settings['max_urls'] ?? $settings['max_pages'] ?? (int) get_option('prorank_audit_max_urls', 100);
            
            // Update status
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->update($this->audits_table, 
                ['status' => 'crawling'],
                ['audit_id' => $audit_id]
            );
            
            // Get all URLs to crawl with limit
            $urls = $this->get_site_urls($max_urls, $settings);
            
            // Batch insert URLs for better performance
            $url_values = [];
            $current_time = current_time('mysql');
            
            foreach ($urls as $url) {
                $url_values[] = $wpdb->prepare(
                    "(%s, %s, %s, %s)",
                    $audit_id,
                    $url,
                    'pending',
                    $current_time
                );
            }
            
            if (!empty($url_values)) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $wpdb->query(
                    "INSERT INTO {$this->audit_urls_table} 
                    (audit_id, url, status, created_at) 
                    VALUES " . implode(',', $url_values)
                );
            }
            
            // Update audit with URL count
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->update($this->audits_table, [
                'total_urls' => count($urls),
                'status' => 'checking',
            ], ['audit_id' => $audit_id]);
            
            // Schedule URL checks
            $this->schedule_url_checks($audit_id);
            
            // Also schedule continuous processing
            wp_schedule_single_event(time() + 5, 'prorank_schedule_url_checks', [$audit_id]);
            
            // Force cron to run
            spawn_cron();
            
        } catch (Exception $e) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->update($this->audits_table, [
                'status' => 'failed',
                'error' => esc_html($e->getMessage()),
            ], ['audit_id' => $audit_id]);
        }
    }
    
    /**
     * Get all site URLs
     */
    private function get_site_urls(int $max_urls = 100, array $settings = []): array {
        $urls = [];
        
        // Get homepage
        $urls[] = home_url('/');
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO: Homepage URL: ' . home_url('/'));
        }
        
        // Get all published posts and pages
        $post_types = get_post_types(['public' => true], 'names');
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO: Post types found: ' . implode(', ', $post_types));
        }
        
        // Remove attachment post type
        unset($post_types['attachment']);
        
        // Query posts more reliably
        $args = [
            'post_type' => array_values($post_types),
            'post_status' => 'publish',
            'posts_per_page' => min($max_urls - 1, 50), // Limit based on settings, -1 for homepage
            'orderby' => 'modified',
            'order' => 'DESC',
            'fields' => 'ids',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ];
        
        $query = new \WP_Query($args);
        $post_ids = $query->posts;
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO: Found ' . count($post_ids) . ' posts/pages');
        }
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO: Query args: ' . json_encode($args));
        }
        
        if (empty($post_ids)) {
            // Fallback: Try querying each post type separately
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: No posts found with combined query, trying individually...');
            }
            foreach ($post_types as $post_type) {
                $type_query = new \WP_Query([
                    'post_type' => $post_type,
                    'post_status' => 'publish',
                    'posts_per_page' => -1,
                    'fields' => 'ids',
                ]);
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log("ProRank SEO: Found {$type_query->post_count} posts for type: $post_type");
                }
                $post_ids = array_merge($post_ids, $type_query->posts);
            }
        }
        
        foreach ($post_ids as $post_id) {
            $permalink = get_permalink($post_id);
            if ($permalink && !in_array($permalink, $urls)) {
                $urls[] = $permalink;
            } else if (!$permalink) {
                // Log failed permalink generation
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log("ProRank SEO: Failed to get permalink for post ID: $post_id");
                }
                
                // Try alternative method
                $post = get_post($post_id);
                if ($post) {
                    $post_type_obj = get_post_type_object($post->post_type);
                    if ($post_type_obj && $post_type_obj->public) {
                        // Try to construct URL manually as fallback
                        $url = home_url('/?p=' . $post_id);
                        if (!in_array($url, $urls)) {
                            $urls[] = $url;
                            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                                prorank_log("ProRank SEO: Using fallback URL for post ID $post_id: $url");
                            }
                        }
                    }
                }
            }
        }
        
        // Get archive pages
        $archive = get_post_type_archive_link('post');
        if ($archive && !in_array($archive, $urls)) {
            $urls[] = $archive;
        }
        
        // Get category pages (limit to prevent too many URLs)
        $categories = get_categories(['number' => 20]);
        foreach ($categories as $category) {
            $link = get_category_link($category->term_id);
            if ($link && !in_array($link, $urls)) {
                $urls[] = $link;
            }
        }
        
        // Get tag pages (limit to prevent too many URLs)
        $tags = get_tags(['number' => 20]);
        foreach ($tags as $tag) {
            $link = get_tag_link($tag->term_id);
            if ($link && !in_array($link, $urls)) {
                $urls[] = $link;
            }
        }
        
        // Remove any false or empty values
        $urls = array_filter($urls);
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO: Total URLs to audit: ' . count($urls));
        }
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO: URLs: ' . implode(', ', array_slice($urls, 0, 5)) . '...');
        }
        
        return $urls;
    }
    
    /**
     * Schedule URL checks
     */
    public function schedule_url_checks(string $audit_id): void {
        global $wpdb;
        
        // Check if audit is still active and get settings
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $audit = $wpdb->get_row($wpdb->prepare(
            "SELECT status, options FROM {$this->audits_table} WHERE audit_id = %s",
            $audit_id
        ));
        
        if (!$audit || !in_array($audit->status, ['checking', 'crawling'])) {
            return; // Audit was stopped or completed
        }
        
        // Get settings from audit options
        $options = [];
        $settings = [];
        if (!empty($audit->options)) {
            $options = json_decode($audit->options, true) ?: [];
            $settings = $options['settings'] ?? [];
        }
        
        // Calculate batch size based on crawl speed
        $crawl_speed = $settings['crawl_speed'] ?? 10; // URLs per minute
        $concurrent_requests = $settings['concurrent_requests'] ?? 5; // Increase default concurrent requests
        // Process more URLs per batch for better performance
        $batch_size = max(10, min($crawl_speed, 50)); // Increased batch size for faster processing
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $urls = $wpdb->get_results($wpdb->prepare(
            "SELECT id, url FROM {$this->audit_urls_table} 
             WHERE audit_id = %s AND status = 'pending'
             LIMIT %d",
            $audit_id,
            intval($batch_size)
        ));
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO: Processing batch of ' . count($urls) . ' URLs for audit ' . $audit_id);
        }
        
        // Process more URLs immediately for faster feedback
        $immediate_checks = array_slice($urls, 0, max(5, $concurrent_requests));

        foreach ($immediate_checks as $index => $url_data) {
            // Schedule immediate check via WP-Cron for parallel processing
            wp_schedule_single_event(
                time() + $index, // Stagger by 1 second each
                'prorank_check_audit_url',
                [$audit_id, (int)$url_data->id, $url_data->url]
            );
        }
        
        // Schedule remaining URLs with faster timing
        $scheduled_urls = array_slice($urls, $concurrent_requests);
        if (count($scheduled_urls) > 0) {
            // Much faster URL processing - minimal delays for speed
            $delay_between_urls = max(0.2, 10 / $crawl_speed); // Much faster processing
            $delay = $concurrent_requests; // Start after immediate checks
            
            foreach ($scheduled_urls as $url_data) {
                wp_schedule_single_event(
                    time() + intval($delay),
                    'prorank_check_audit_url',
                    [$audit_id, (int)$url_data->id, $url_data->url]
                );
                $delay += $delay_between_urls;
            }
        }
        
        // Check if more URLs to process
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $remaining = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->audit_urls_table} 
             WHERE audit_id = %s AND status = 'pending'",
            $audit_id
        ));
        
        if ($remaining > 0) {
            // Schedule next batch much more quickly for better performance
            // Process batches with minimal delay
            $next_batch_delay = min(2, max(1, 5 / $crawl_speed)); // Much faster batch processing
            
            wp_schedule_single_event(
                time() + intval($next_batch_delay),
                'prorank_schedule_url_checks',
                [$audit_id]
            );
        } else {
            // All URLs processed, complete audit
            $this->complete_audit($audit_id);
        }
    }
    
    /**
     * Check a single URL
     */
    public function check_url(string $audit_id, int $url_id, string $url): void {
        global $wpdb;
        
        // Get audit with options
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $audit = $wpdb->get_row($wpdb->prepare(
            "SELECT started_at, status, options FROM {$this->audits_table} WHERE audit_id = %s",
            $audit_id
        ));
        
        // Get settings from audit options
        $options = [];
        $settings = [];
        $check_types = [];
        if ($audit && !empty($audit->options)) {
            $options = json_decode($audit->options, true) ?: [];
            $settings = $options['settings'] ?? [];
            $check_types = $options['check_types'] ?? [];
        }

        // Default check types - all enabled unless explicitly disabled
        $default_checks = [
            'meta_tags' => true,
            'headings_structure' => true,
            'canonical_tags' => true,
            'open_graph' => true,
            'schema_validation' => true,
            'image_optimization' => true,
            'https_status' => true,
            'broken_links' => true,
        ];
        $enabled_checks = array_merge($default_checks, $check_types);

        // Get timeout and max duration from settings
        $request_timeout = $settings['request_timeout'] ?? 2; // Reduced default timeout
        $max_duration = ($settings['max_duration'] ?? 30) * 60; // Convert to seconds
        
        if ($audit && $audit->started_at) {
            $elapsed = time() - strtotime($audit->started_at);
            if ($elapsed > $max_duration) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log("ProRank SEO: Audit {$audit_id} exceeded maximum time ({$max_duration}s), auto-completing");
                }
                $this->complete_audit($audit_id);
                return;
            }
        }
        
        // Check if audit was stopped
        if ($audit && $audit->status === 'stopped') {
            return;
        }
        
        $issues = [];
        $passed_checks = [];
        $warnings = [];
        
        // Respect robots.txt before fetching
        if ($this->is_disallowed_by_robots($url)) {
            $this->add_issue($audit_id, $url, 'disallowed_by_robots', 'URL blocked by robots.txt', 'warning');
            $this->mark_url_checked($url_id);
            return;
        }

        // Rate-limit per host to avoid hammering
        $this->rate_limit_host($url);

        // Enhanced fetch with more data
        $start_time = microtime(true);
        $response = wp_remote_get($url, [
            'timeout' => max(4, $request_timeout), // Ensure a reasonable timeout
            'redirection' => 5,
            'user-agent' => 'ProRank SEO Audit Bot/1.1',
            'sslverify' => true,
            'headers' => [
                'Accept' => 'text/html,application/xhtml+xml',
                'Accept-Encoding' => 'gzip, deflate, br',
            ],
            'httpversion' => '1.1',
            'compress' => true,
            'reject_unsafe_urls' => true,
        ]);
        $load_time = microtime(true) - $start_time;
        
        if (is_wp_error($response)) {
            $issues[] = [
                'type' => 'connectivity',
                'severity' => 'critical',
                'message' => 'Failed to fetch URL: ' . $response->get_error_message(),
                'data' => ['error_code' => $response->get_error_code()],
            ];
        } else {
            $status_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            $headers = wp_remote_retrieve_headers($response);
            $response_size = strlen($body);
            
            // Check status code

            // Optionally fetch Core Web Vitals for a small subset of URLs
            $this->maybe_record_core_web_vitals($audit_id, $url, $headers);
            $this->check_structured_data_enhanced($audit_id, $url, $body);
            if ($status_code >= 400) {
                $issues[] = [
                    'type' => 'http_error',
                    'severity' => 'critical',
                    'message' => sprintf('HTTP %d error', $status_code),
                    'data' => ['status_code' => $status_code],
                ];
            } elseif ($status_code >= 300) {
                $warnings[] = [
                    'type' => 'redirect',
                    'severity' => 'medium',
                    'message' => sprintf('Page redirects with HTTP %d', $status_code),
                    'data' => ['status_code' => $status_code, 'location' => $headers['location'] ?? ''],
                ];
            } else {
                $passed_checks[] = 'http_status';
            }
            
            // Parse HTML
            if (!empty($body) && $status_code < 400) {
                $dom = new \DOMDocument();
                libxml_use_internal_errors(true);
                @$dom->loadHTML(mb_convert_encoding($body, 'HTML-ENTITIES', 'UTF-8'));
                libxml_clear_errors();

                // Core SEO checks - conditionally run based on enabled_checks
                if (!empty($enabled_checks['meta_tags'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_title($dom));
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_meta_description($dom));
                }

                if (!empty($enabled_checks['headings_structure'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_headings($dom));
                }

                if (!empty($enabled_checks['image_optimization'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_images($dom, $url));
                }

                // Technical SEO checks
                if (!empty($enabled_checks['canonical_tags'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_canonical($dom, $url));
                }

                if (!empty($enabled_checks['robots_txt'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_robots_meta($dom));
                }

                // Social and structured data checks
                if (!empty($enabled_checks['open_graph']) || !empty($enabled_checks['twitter_cards'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_open_graph($dom));
                }

                if (!empty($enabled_checks['schema_validation'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_schema_markup($dom));
                }

                // Security check (HTTPS)
                if (!empty($enabled_checks['https_status']) || !empty($enabled_checks['ssl_certificate'])) {
                    if (strpos($url, 'https://') === 0) {
                        $passed_checks[] = 'https_enabled';
                    } else {
                        $issues[] = [
                            'type' => 'missing_https',
                            'severity' => 'high',
                            'message' => 'Page is not using HTTPS',
                            'data' => []
                        ];
                    }
                }

                // Content quality check
                if (!empty($enabled_checks['content_quality'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_content_quality($dom, $body));
                }

                // Link checks (broken links, internal/external linking)
                if (!empty($enabled_checks['broken_links']) || !empty($enabled_checks['internal_linking']) || !empty($enabled_checks['external_links'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_links($dom, $url));
                }

                // Page speed check
                if (!empty($enabled_checks['page_speed'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_page_speed($url, $headers, $load_time, $response_size));
                }

                // Core Web Vitals check
                if (!empty($enabled_checks['core_web_vitals'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_core_web_vitals($url));
                }

                // Mobile optimization check
                if (!empty($enabled_checks['mobile_friendly'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_mobile_optimization($dom));
                }

                // Accessibility check
                if (!empty($enabled_checks['accessibility'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_accessibility($dom));
                }

                // Mixed content check
                if (!empty($enabled_checks['mixed_content'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_mixed_content($dom, $url));
                }

                // Security headers check (uses direct insert pattern)
                if (!empty($enabled_checks['security_headers'])) {
                    $this->check_security_headers($audit_id, $url, $headers);
                }

                // Redirect chains check (uses direct insert pattern)
                if (!empty($enabled_checks['redirect_chains'])) {
                    $this->check_redirect_chain($audit_id, $url);
                }

                // Thin content check
                if (!empty($enabled_checks['thin_content'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_thin_content($dom, $body));
                }

                // Duplicate content check
                if (!empty($enabled_checks['duplicate_content'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_duplicate_content($audit_id, $url, $dom, $body));
                }

                // Crawl depth check
                if (!empty($enabled_checks['crawl_depth'])) {
                    $this->merge_check_results($issues, $passed_checks, $warnings, $this->check_crawl_depth($audit_id, $url_id));
                }

                // Store internal links for orphaned pages detection
                if (!empty($enabled_checks['orphaned_pages'])) {
                    $this->store_internal_links_for_orphan_detection($audit_id, $dom, $url);
                }
            }
        }

        // Batch insert all findings for better performance
        $all_findings = array_merge($issues, $warnings);
        $insert_values = [];
        $current_time = current_time('mysql');
        
        // Prepare all findings for batch insert
        foreach ($all_findings as $finding) {
            $insert_values[] = $wpdb->prepare(
                "(%s, %d, %s, %s, %s, %s, %s)",
                $audit_id,
                $url_id,
                $finding['type'],
                $finding['severity'],
                $finding['message'],
                json_encode($finding['data'] ?? []),
                $current_time
            );
        }
        
        // Add passed checks
        foreach ($passed_checks as $check) {
            $insert_values[] = $wpdb->prepare(
                "(%s, %d, %s, %s, %s, %s, %s)",
                $audit_id,
                $url_id,
                $check,
                'passed',
                'Check passed successfully',
                '{}',
                $current_time
            );
        }
        
        // Batch insert all issues at once
        if (!empty($insert_values)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query(
                "INSERT INTO {$this->audit_issues_table} 
                (audit_id, url_id, type, severity, message, data, created_at) 
                VALUES " . implode(',', $insert_values)
            );
        }
        
        // Calculate page score
        $page_score = $this->calculate_page_score($issues, $warnings, $passed_checks);
        
        // Update URL status with enhanced data
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update($this->audit_urls_table, [
            'status' => 'checked',
            'issues_count' => count($issues),
            'warnings_count' => count($warnings),
            'passed_count' => count($passed_checks),
            'score' => $page_score,
            'checked_at' => current_time('mysql'),
        ], ['id' => $url_id]);

        // Recursive link discovery with max_depth enforcement
        if (isset($dom) && isset($body) && !empty($body)) {
            $max_depth = $settings['max_depth'] ?? 5;
            $max_urls = $settings['max_urls'] ?? 500;
            $this->discover_and_queue_links($audit_id, $url, $url_id, $dom, $max_depth, $max_urls);
        }
    }

    /**
     * Discover internal links and queue them for crawling (respects max_depth)
     */
    private function discover_and_queue_links(string $audit_id, string $url, int $url_id, \DOMDocument $dom, int $max_depth, int $max_urls): void {
        global $wpdb;

        // Get current URL's depth
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $current_depth = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT depth FROM {$this->audit_urls_table} WHERE id = %d",
            $url_id
        ));

        // Don't discover if at max depth
        if ($current_depth >= $max_depth) {
            return;
        }

        // Check current URL count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $current_count = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->audit_urls_table} WHERE audit_id = %s",
            $audit_id
        ));

        if ($current_count >= $max_urls) {
            return;
        }

        // Get home URL for internal link detection
        $home_url = home_url();
        $home_host = wp_parse_url($home_url, PHP_URL_HOST);

        // Find all links
        $links = $dom->getElementsByTagName('a');
        $new_urls = [];
        $new_depth = $current_depth + 1;

        foreach ($links as $link) {
            $href = $link->getAttribute('href');

            // Skip empty, anchor-only, or javascript links
            if (empty($href) || $href[0] === '#' || strpos($href, 'javascript:') === 0 || strpos($href, 'mailto:') === 0) {
                continue;
            }

            // Make absolute URL
            $absolute_url = $this->make_absolute_url($href, $url);
            if (!$absolute_url) {
                continue;
            }

            // Check if internal link
            $link_host = wp_parse_url($absolute_url, PHP_URL_HOST);
            if ($link_host !== $home_host) {
                continue;
            }

            // Normalize URL (remove fragment, trailing slash consistency)
            $normalized = strtok($absolute_url, '#');
            $normalized = rtrim($normalized, '/');

            // Skip if already in queue
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$this->audit_urls_table} WHERE audit_id = %s AND url = %s",
                $audit_id,
                $normalized
            ));

            if (!$exists && !isset($new_urls[$normalized])) {
                $new_urls[$normalized] = $new_depth;
            }

            // Limit new URLs per page to avoid explosion
            if (count($new_urls) >= 20) {
                break;
            }
        }

        // Insert new URLs
        if (!empty($new_urls)) {
            $remaining_slots = $max_urls - $current_count;
            $urls_to_add = array_slice($new_urls, 0, $remaining_slots, true);

            foreach ($urls_to_add as $new_url => $depth) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $wpdb->insert($this->audit_urls_table, [
                    'audit_id' => $audit_id,
                    'url' => $new_url,
                    'status' => 'pending',
                    'depth' => $depth,
                    'created_at' => current_time('mysql'),
                ]);
            }
        }
    }
    
    /**
     * Check title tag
     */
    /**
     * Helper function to merge check results
     */
    private function merge_check_results(&$issues, &$passed_checks, &$warnings, $results): void {
        if (isset($results['issues'])) {
            $issues = array_merge($issues, $results['issues']);
        }
        if (isset($results['warnings'])) {
            $warnings = array_merge($warnings, $results['warnings']);
        }
        if (isset($results['passed'])) {
            $passed_checks = array_merge($passed_checks, $results['passed']);
        }
    }
    
    /**
     * Calculate page score based on findings
     */
    private function calculate_page_score($issues, $warnings, $passed_checks): int {
        $total_checks = count($issues) + count($warnings) + count($passed_checks);
        if ($total_checks === 0) return 0;
        
        // Weight factors
        $critical_weight = 25;
        $high_weight = 15;
        $medium_weight = 10;
        $low_weight = 5;
        
        $deductions = 0;
        foreach ($issues as $issue) {
            switch ($issue['severity']) {
                case 'critical':
                    $deductions += $critical_weight;
                    break;
                case 'high':
                    $deductions += $high_weight;
                    break;
                case 'medium':
                    $deductions += $medium_weight;
                    break;
                case 'low':
                    $deductions += $low_weight;
                    break;
            }
        }
        
        foreach ($warnings as $warning) {
            $deductions += $low_weight;
        }
        
        $score = max(0, 100 - $deductions);
        return (int) $score;
    }
    
    private function check_title(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $titles = $dom->getElementsByTagName('title');
        
        if ($titles->length === 0) {
            $result['issues'][] = [
                'type' => 'missing_title',
                'severity' => 'critical',
                'message' => 'Missing title tag - Critical for SEO',
                'data' => ['fix' => 'Add a unique <title> tag to your page'],
            ];
        } elseif ($titles->length > 1) {
            $result['issues'][] = [
                'type' => 'duplicate_title',
                'severity' => 'high',
                'message' => 'Multiple title tags found',
                'data' => ['count' => $titles->length, 'fix' => 'Keep only one <title> tag'],
            ];
        } else {
            $title = trim($titles->item(0)->textContent);
            
            if (empty($title)) {
                $result['issues'][] = [
                    'type' => 'empty_title',
                    'severity' => 'critical',
                    'message' => 'Empty title tag',
                    'data' => ['fix' => 'Add descriptive text to your title tag'],
                ];
            } else {
                // Title exists - pass basic check
                $result['passed'][] = 'has_title';
                
                $title_length = mb_strlen($title);
                
                // Length checks
                if ($title_length < 30) {
                    $result['warnings'][] = [
                        'type' => 'short_title',
                        'severity' => 'medium',
                        'message' => sprintf('Title too short (%d chars, recommended 30-60)', $title_length),
                        'data' => ['title' => $title, 'length' => $title_length],
                    ];
                } elseif ($title_length > 60) {
                    $result['warnings'][] = [
                        'type' => 'long_title',
                        'severity' => 'medium',
                        'message' => sprintf('Title too long (%d chars, may be truncated)', $title_length),
                        'data' => ['title' => $title, 'length' => $title_length],
                    ];
                } else {
                    $result['passed'][] = 'title_length_optimal';
                }
                
                // Keyword stuffing check
                if (preg_match('/(\b\w+\b).*\b\1\b.*\b\1\b/i', $title)) {
                    $result['warnings'][] = [
                        'type' => 'title_keyword_stuffing',
                        'severity' => 'medium',
                        'message' => 'Possible keyword stuffing in title',
                        'data' => ['title' => $title],
                    ];
                } else {
                    $result['passed'][] = 'no_keyword_stuffing';
                }
            }
        }
        
        return $result;
    }
    
    /**
     * Check meta description
     */
    private function check_meta_description(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        $descriptions = $xpath->query('//meta[@name="description"]');
        
        if ($descriptions->length === 0) {
            $result['issues'][] = [
                'type' => 'missing_meta_description',
                'severity' => 'high',
                'message' => 'Missing meta description',
            ];
        } else {
            $description = $descriptions->item(0)->getAttribute('content');
            
            if (empty($description)) {
                $result['issues'][] = [
                    'type' => 'empty_meta_description',
                    'severity' => 'high',
                    'message' => 'Empty meta description',
                ];
            } elseif (strlen($description) < 120) {
                $result['warnings'][] = [
                    'type' => 'short_meta_description',
                    'severity' => 'medium',
                    'message' => sprintf('Meta description too short (%d chars)', strlen($description)),
                    'data' => ['description' => $description, 'length' => strlen($description)],
                ];
            } elseif (strlen($description) > 160) {
                $result['warnings'][] = [
                    'type' => 'long_meta_description',
                    'severity' => 'medium',
                    'message' => sprintf('Meta description too long (%d chars)', strlen($description)),
                    'data' => ['description' => $description, 'length' => strlen($description)],
                ];
            } else {
                $result['passed'][] = 'meta_description_optimal';
            }
        }
        
        return $result;
    }
    
    /**
     * Check headings
     */
    private function check_headings(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $h1s = $dom->getElementsByTagName('h1');
        
        if ($h1s->length === 0) {
            $result['issues'][] = [
                'type' => 'missing_h1',
                'severity' => 'high',
                'message' => 'Missing H1 heading',
            ];
        } elseif ($h1s->length > 1) {
            $result['warnings'][] = [
                'type' => 'multiple_h1',
                'severity' => 'medium',
                'message' => sprintf('Multiple H1 headings found (%d)', $h1s->length),
            ];
        } else {
            $result['passed'][] = 'has_single_h1';
        }
        
        return $result;
    }
    
    /**
     * Check images
     */
    private function check_images(\DOMDocument $dom, string $base_url): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $images = $dom->getElementsByTagName('img');
        $missing_alt_count = 0;
        $broken_images = 0;
        
        foreach ($images as $img) {
            $alt = $img->getAttribute('alt');
            $src = $img->getAttribute('src');
            
            if (empty($alt)) {
                $missing_alt_count++;
            }
            
            // Check if image exists
            if (!empty($src)) {
                $image_url = $this->make_absolute_url($src, $base_url);
                $response = wp_remote_head($image_url, ['timeout' => 5]);
                
                if (is_wp_error($response) || wp_remote_retrieve_response_code($response) >= 400) {
                    $broken_images++;
                }
            }
        }
        
        if ($missing_alt_count > 0) {
            $result['issues'][] = [
                'type' => 'missing_alt_text',
                'severity' => 'medium',
                'message' => sprintf('%d images missing alt text', $missing_alt_count),
                'data' => ['count' => $missing_alt_count],
            ];
        }
        
        if ($broken_images > 0) {
            $result['issues'][] = [
                'type' => 'broken_image',
                'severity' => 'high',
                'message' => sprintf('%d broken images found', $broken_images),
                'data' => ['count' => $broken_images],
            ];
        }
        
        if ($images->length > 0 && $missing_alt_count === 0) {
            $result['passed'][] = 'all_images_have_alt';
        }
        
        return $result;
    }
    
    /**
     * Check links
     */
    private function check_links(\DOMDocument $dom, string $base_url): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $links = $dom->getElementsByTagName('a');
        $checked_urls = [];
        $broken_links = 0;
        $internal_links = 0;
        $external_links = 0;
        
        foreach ($links as $link) {
            $href = $link->getAttribute('href');
            
            if (empty($href) || $href === '#') {
                continue;
            }
            
            $absolute_url = $this->make_absolute_url($href, $base_url);
            
            // Skip if already checked
            if (in_array($absolute_url, $checked_urls)) {
                continue;
            }
            
            $checked_urls[] = $absolute_url;
            
            // Check if internal or external
            if (strpos($absolute_url, home_url()) === 0) {
                $internal_links++;
                $response = wp_remote_head($absolute_url, ['timeout' => 5]);
                
                if (is_wp_error($response) || wp_remote_retrieve_response_code($response) >= 400) {
                    $broken_links++;
                }
            } else {
                $external_links++;
            }
        }
        
        if ($broken_links > 0) {
            $result['issues'][] = [
                'type' => 'broken_link',
                'severity' => 'high',
                'message' => sprintf('%d broken internal links found', $broken_links),
                'data' => ['count' => $broken_links],
            ];
        }
        
        if ($internal_links === 0) {
            $result['warnings'][] = [
                'type' => 'no_internal_links',
                'severity' => 'medium',
                'message' => 'No internal links found',
            ];
        } else {
            $result['passed'][] = 'has_internal_links';
        }
        
        return $result;
    }
    
    /**
     * Process URLs synchronously (for testing/debugging)
     */
    public function process_urls_sync(string $audit_id, int $limit = 5): int {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $urls = $wpdb->get_results($wpdb->prepare(
            "SELECT id, url FROM {$this->audit_urls_table} 
             WHERE audit_id = %s AND status = 'pending'
             LIMIT %d",
            $audit_id,
            $limit
        ));
        
        $processed = 0;
        foreach ($urls as $url_data) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log("ProRank SEO: Processing URL sync: {$url_data->url}");
            }
            try {
                $this->check_url($audit_id, (int)$url_data->id, $url_data->url);
                $processed++;
            } catch (\Exception $e) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log("ProRank SEO: Error processing URL: " . $e->getMessage());
                }
            }
        }
        
        return $processed;
    }
    
    /**
     * Stop audit
     */
    public function stop_audit(string $audit_id = null): bool {
        global $wpdb;
        
        // If no audit_id provided, stop the latest audit
        if (!$audit_id) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $audit = $wpdb->get_row("
                SELECT audit_id FROM {$this->audits_table} 
                WHERE status IN ('crawling', 'checking')
                ORDER BY started_at DESC 
                LIMIT 1
            ");
            
            if (!$audit) {
                return false;
            }
            
            $audit_id = $audit->audit_id;
        }
        
        // Update audit status to stopped
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $result = $wpdb->update(
            $this->audits_table,
            [
                'status' => 'stopped',
                'completed_at' => current_time('mysql'),
            ],
            ['audit_id' => $audit_id]
        );
        
        // Clear any scheduled events for this audit
        $timestamp = wp_next_scheduled('prorank_run_site_crawler', [$audit_id]);
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'prorank_run_site_crawler', [$audit_id]);
        }
        
        $timestamp = wp_next_scheduled('prorank_schedule_url_checks', [$audit_id]);
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'prorank_schedule_url_checks', [$audit_id]);
        }
        
        $timestamp = wp_next_scheduled('prorank_complete_audit', [$audit_id]);
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'prorank_complete_audit', [$audit_id]);
        }
        
        // Clear all URL check events
        $crons = _get_cron_array();
        foreach ($crons as $timestamp => $cron) {
            foreach ($cron as $hook => $args) {
                if ($hook === 'prorank_check_audit_url') {
                    foreach ($args as $key => $arg) {
                        if (isset($arg['args'][0]) && $arg['args'][0] === $audit_id) {
                            wp_unschedule_event($timestamp, $hook, $arg['args']);
                        }
                    }
                }
            }
        }
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log("ProRank SEO: Audit {$audit_id} stopped by user");
        }
        
        return $result !== false;
    }
    
    /**
     * Complete audit
     */
    public function complete_audit(string $audit_id): array {
        global $wpdb;

        // Run orphaned pages check now that all links are collected
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $audit = $wpdb->get_row($wpdb->prepare(
            "SELECT options FROM {$this->audits_table} WHERE audit_id = %s",
            $audit_id
        ));
        $options = $audit && !empty($audit->options) ? json_decode($audit->options, true) : [];
        $check_types = $options['check_types'] ?? [];

        if (!isset($check_types['orphaned_pages']) || $check_types['orphaned_pages']) {
            $this->check_orphaned_pages($audit_id);
        }

        // Calculate statistics - count unique issues and affected URLs separately
        $stats = $wpdb->get_row($wpdb->prepare("
            SELECT 
                COUNT(DISTINCT u.id) as total_urls,
                COUNT(DISTINCT CASE WHEN u.issues_count > 0 THEN u.id END) as urls_with_issues,
                COUNT(DISTINCT CONCAT(i.type, ':', i.message)) as unique_issues,
                COUNT(CASE WHEN i.severity != 'passed' THEN 1 END) as total_issue_instances,
                COUNT(DISTINCT CASE WHEN i.severity = 'critical' THEN CONCAT(i.type, ':', i.message) END) as critical_issues,
                COUNT(DISTINCT CASE WHEN i.severity = 'high' THEN CONCAT(i.type, ':', i.message) END) as high_issues,
                COUNT(DISTINCT CASE WHEN i.severity = 'medium' THEN CONCAT(i.type, ':', i.message) END) as medium_issues,
                COUNT(DISTINCT CASE WHEN i.severity = 'low' THEN CONCAT(i.type, ':', i.message) END) as low_issues,
                SUM(CASE WHEN i.severity = 'passed' THEN 1 ELSE 0 END) as passed_checks,
                COUNT(CASE WHEN i.severity = 'critical' THEN 1 ELSE 0 END) as critical_issue_instances,
                COUNT(CASE WHEN i.severity = 'high' THEN 1 ELSE 0 END) as high_issue_instances,
                COUNT(CASE WHEN i.severity = 'medium' THEN 1 ELSE 0 END) as medium_issue_instances,
                COUNT(CASE WHEN i.severity = 'low' THEN 1 ELSE 0 END) as low_issue_instances
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            FROM {$this->audit_urls_table} u
            LEFT JOIN {$this->audit_issues_table} i ON u.id = i.url_id
            WHERE u.audit_id = %s
        ", $audit_id));
        
        // Calculate category stats
        $categories = $this->calculate_category_stats($audit_id);
        
        // Calculate overall score like professional SEO tools (Ahrefs-style)
        // Start with perfect score and deduct based on issue severity
        $score = 100;
        
        // Deduct points based on issue severity (with caps to prevent over-penalization)
        $critical_penalty = min(30, ($stats->critical_issues ?? 0) * 10); // Max -30 points
        $high_penalty = min(20, ($stats->high_issues ?? 0) * 5);         // Max -20 points
        $medium_penalty = min(20, ($stats->medium_issues ?? 0) * 2);     // Max -20 points
        $low_penalty = min(10, ($stats->low_issues ?? 0) * 1);           // Max -10 points
        
        $score -= $critical_penalty;
        $score -= $high_penalty;
        $score -= $medium_penalty;
        $score -= $low_penalty;
        
        // Bonus points for having many passed checks
        if ($stats->passed_checks > 0 && $stats->total_urls > 0) {
            $pass_rate = $stats->passed_checks / ($stats->total_urls * 10); // Assume ~10 checks per URL
            $bonus = min(10, round($pass_rate * 10)); // Up to 10 bonus points
            $score += $bonus;
        }
        
        // Ensure score is between 0 and 100
        $score = max(0, min(100, round($score)));
        
        // Update audit
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update($this->audits_table, [
            'status' => 'completed',
            'completed_at' => current_time('mysql'),
            'score' => $score,
            'stats' => json_encode($stats),
        ], ['audit_id' => $audit_id]);
        
        return [
            'audit_id' => $audit_id,
            'score' => $score,
            'stats' => $stats,
            'categories' => $categories,
            'completed_at' => current_time('mysql'),
        ];
    }
    
    /**
     * Calculate category statistics
     */
    private function calculate_category_stats(string $audit_id): array {
        global $wpdb;
        
        $categories = [
            'technical' => [
                'name' => 'Technical SEO',
                'types' => ['missing_title', 'duplicate_title', 'empty_title', 'missing_meta_description', 
                           'empty_meta_description', 'missing_h1', 'duplicate_h1', 'missing_canonical',
                           'duplicate_canonical', 'noindex_set', 'missing_robots_txt', 'missing_sitemap'],
            ],
            'on-page' => [
                'name' => 'On-Page SEO',
                'types' => ['short_title', 'long_title', 'short_meta_description', 'long_meta_description',
                           'title_keyword_stuffing', 'missing_og_tag', 'missing_twitter_cards', 'thin_content',
                           'keyword_optimization', 'missing_alt_text'],
            ],
            'performance' => [
                'name' => 'Performance',
                'types' => ['slow_load_time', 'large_page_size', 'no_compression', 'no_cache_headers',
                           'unoptimized_images', 'render_blocking_resources', 'large_dom_size'],
            ],
            'security' => [
                'name' => 'Security',
                'types' => ['missing_https', 'mixed_content', 'missing_security_header', 'vulnerable_js_library'],
            ],
            'content' => [
                'name' => 'Content Quality',
                'types' => ['thin_content', 'duplicate_content', 'long_sentences', 'missing_headings',
                           'broken_links', 'external_broken_links'],
            ],
        ];
        
        $results = [];
        
        foreach ($categories as $key => $category) {
            $type_list = "'" . implode("','", $category['types']) . "'";
            
            $stats = $wpdb->get_row($wpdb->prepare("
                SELECT 
                    COUNT(CASE WHEN severity = 'critical' THEN 1 END) as critical,
                    COUNT(CASE WHEN severity = 'high' THEN 1 END) as high,
                    COUNT(CASE WHEN severity = 'medium' THEN 1 END) as medium,
                    COUNT(CASE WHEN severity = 'low' THEN 1 END) as low,
                    COUNT(CASE WHEN severity = 'passed' THEN 1 END) as passed
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                FROM {$this->audit_issues_table}
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                WHERE audit_id = %s AND type IN ($type_list)
            ", $audit_id));
            
            $total = $stats->critical + $stats->high + $stats->medium + $stats->low + $stats->passed;
            $score = $total > 0 ? ($stats->passed / $total) * 100 : 100;
            
            $results[$key] = [
                'name' => $category['name'],
                'score' => round($score),
                'passed' => $stats->passed,
                'warnings' => $stats->medium + $stats->low,
                'critical' => $stats->critical + $stats->high,
                'total' => $total,
            ];
        }
        
        return $results;
    }
    
    /**
     * Get audit status
     */
    public function get_audit_status(string $audit_id = null): array {
        global $wpdb;
        
        // Get latest audit if no ID provided
        if (!$audit_id) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $audit = $wpdb->get_row("
                SELECT * FROM {$this->audits_table} 
                ORDER BY started_at DESC 
                LIMIT 1
            ");
        } else {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $audit = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->audits_table} WHERE audit_id = %s",
                $audit_id
            ));
        }
        
        if (!$audit) {
            return [
                'state' => 'idle',
                'stats' => [
                    'critical_issues' => 0,
                    'high_issues' => 0,
                    'medium_issues' => 0,
                    'low_issues' => 0,
                    'total_issues' => 0,
                    'unique_issues' => 0,
                    'total_instances' => 0,
                    'total_urls' => 0,
                    'passed' => 0,
                ],
            ];
        }
        
        $stats = $audit->stats ? json_decode($audit->stats, true) : [];
        
        // Get issues by type (excluding passed checks)
        $issues = [];
        if ($audit->status === 'completed') {
            $issue_types = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    i.type,
                    i.severity,
                    i.message,
                    COUNT(*) as count,
                    GROUP_CONCAT(DISTINCT u.url SEPARATOR '|||') as affected_urls,
                    MAX(i.data) as sample_data
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                FROM {$this->audit_issues_table} i
                JOIN {$this->audit_urls_table} u ON i.url_id = u.id
                WHERE i.audit_id = %s AND i.severity != 'passed'
                GROUP BY i.type, i.severity, i.message
                ORDER BY 
                    FIELD(i.severity, 'critical', 'high', 'medium', 'low'),
                    count DESC
            ", $audit->audit_id));
            
            foreach ($issue_types as $issue_type) {
                $affected_urls = $issue_type->affected_urls ? explode('|||', $issue_type->affected_urls) : [];
                $issues[] = [
                    'id' => $issue_type->type . '_' . md5($issue_type->message),
                    'type' => $issue_type->type,
                    'message' => $issue_type->message,
                    'severity' => $issue_type->severity,
                    'count' => (int)$issue_type->count,
                    'affected_urls' => array_slice($affected_urls, 0, 5),
                    'data' => json_decode($issue_type->sample_data, true),
                ];
            }
        }
        
        $result = [
            'state' => $audit->status,
            'audit_id' => $audit->audit_id,
            'started_at' => $audit->started_at,
            'completed_at' => $audit->completed_at,
            'score' => (int)($audit->score ?? 0),
            'stats' => [
                'critical_issues' => (int)($stats['unique_issues']['critical'] ?? 0),
                'high_issues' => (int)($stats['unique_issues']['high'] ?? 0),
                'medium_issues' => (int)($stats['unique_issues']['medium'] ?? 0),
                'low_issues' => (int)($stats['unique_issues']['low'] ?? 0),
                'total_issues' => (int)($stats['unique_issues']['total'] ?? 0),
                'unique_issues' => (int)($stats['unique_issues']['total'] ?? 0),
                'total_instances' => (int)($stats['total_instances'] ?? 0),
                'total_urls' => (int)($stats['total_urls'] ?? 0),
                'passed' => (int)($stats['total_urls'] ?? 0) - (int)($stats['urls_with_issues'] ?? 0),
            ],
            'issues' => $issues,
            'quick_fixes' => $this->get_quick_fixes($issues),
            'categories' => $this->calculate_category_scores($audit->audit_id),
        ];
        
        // Add progress data if audit is running
        if ($audit->status === 'crawling' || $audit->status === 'checking') {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $total_urls = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->audit_urls_table} WHERE audit_id = %s",
                $audit->audit_id
            ));
            
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $checked_urls = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->audit_urls_table} WHERE audit_id = %s AND status = 'checked'",
                $audit->audit_id
            ));
            
            // Prevent negative remaining / >100% progress if counts drift
            if ($checked_urls > $total_urls) {
                $total_urls = $checked_urls;
            }

            $result['progress'] = [
                'total_urls'   => (int)$total_urls,
                'checked_urls' => (int)$checked_urls,
                'percentage'   => $total_urls > 0 ? min(100, round(($checked_urls / $total_urls) * 100, 2)) : 0
            ];
        }
        
        return $result;
    }
    
    /**
     * Get quick fixes based on issues
     */
    private function get_quick_fixes(array $issues): array {
        $fixes = [];
        
        foreach ($issues as $issue) {
            switch ($issue['id']) {
                case 'missing_cache_headers':
                    $fixes[] = [
                        'id' => 'enable_caching',
                        'type' => 'enable_caching',
                        'title' => __('Enable browser caching', 'prorank-seo'),
                        'description' => __('Set proper cache headers for static resources', 'prorank-seo'),
                        'severity' => 'warning',
                        'impact' => __('Reduces server load and improves repeat visit speed', 'prorank-seo'),
                    ];
                    break;
                    
                case 'missing_compression':
                    $fixes[] = [
                        'id' => 'enable_compression',
                        'type' => 'enable_compression',
                        'title' => __('Enable GZIP compression', 'prorank-seo'),
                        'description' => __('Compress text-based resources to reduce file sizes', 'prorank-seo'),
                        'severity' => 'warning',
                        'impact' => __('Can reduce page size by 60-80%', 'prorank-seo'),
                    ];
                    break;
                    
                case 'broken_image':
                    $fixes[] = [
                        'id' => 'fix_broken_images',
                        'type' => 'fix_broken_images',
                        'title' => __('Fix broken images', 'prorank-seo'),
                        'description' => __('Remove or replace broken image references', 'prorank-seo'),
                        'severity' => 'high',
                        'impact' => __('Improves user experience and SEO', 'prorank-seo'),
                    ];
                    break;
            }
        }
        
        return array_slice($fixes, 0, 3); // Return top 3 fixes
    }
    
    /**
     * Clear audit data
     */
    public function clear_audit_data(): bool {
        global $wpdb;
        
        try {
            // Clear all scheduled audit tasks first
            $cron_array = _get_cron_array();
            $cleared_events = 0;
            foreach ($cron_array as $timestamp => $cron) {
                foreach ($cron as $hook => $values) {
                    if (strpos($hook, 'prorank_') === 0 && 
                        (strpos($hook, 'audit') !== false || 
                         strpos($hook, 'crawler') !== false || 
                         strpos($hook, 'check') !== false)) {
                        foreach ($values as $key => $args) {
                            wp_unschedule_event($timestamp, $hook, $args['args']);
                            $cleared_events++;
                        }
                    }
                }
            }
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: Cleared ' . $cleared_events . ' scheduled audit events');
            }
            
            // Clear all audit data
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("TRUNCATE TABLE {$this->audits_table}");
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("TRUNCATE TABLE {$this->audit_urls_table}");
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("TRUNCATE TABLE {$this->audit_issues_table}");
            
            // Clear transients
            delete_transient('prorank_audit_status');
            delete_transient('prorank_audit_issues');
            
            // Clear any audit-related options
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%prorank_audit%'");
            
            return true;
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: Clear audit data error: ' . $e->getMessage());
            }
            return false;
        }
    }
    
    /**
     * Get audit history
     */
    public function get_audit_history(int $limit = 10): array {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $audits = $wpdb->get_results($wpdb->prepare("
            SELECT * FROM {$this->audits_table}
            WHERE status = 'completed'
            ORDER BY completed_at DESC
            LIMIT %d
        ", $limit));
        
        $history = [];
        foreach ($audits as $audit) {
            $stats = $audit->stats ? json_decode($audit->stats, true) : [];
            
            $history[] = [
                'id' => $audit->audit_id,
                'date' => $audit->completed_at,
                'score' => (int)$audit->score,
                'critical' => (int)($stats['critical_issues'] ?? 0),
                'warnings' => (int)($stats['high_issues'] ?? 0),
                'total_issues' => (int)($stats['total_issues'] ?? 0),
                'totalUrls' => (int)($stats['total_urls'] ?? 0),
                'duration' => strtotime($audit->completed_at) - strtotime($audit->started_at),
                'stats' => $stats,
                'state' => 'completed',
            ];
        }
        
        return $history;
    }
    
    /**
     * Calculate category scores
     */
    private function calculate_category_scores(string $audit_id): array {
        global $wpdb;
        
        // Map issue types to categories (using actual issue types from the code)
        $category_mapping = [
            'technical-seo' => [
                'missing_title', 'duplicate_title', 'empty_title', 'missing_canonical', 'multiple_canonicals',
                'missing_h1', 'multiple_h1', 'http_error', 'connectivity', 'redirect', 'noindex_set',
                'missing_x_default_hreflang', 'missing_hreflang', 'missing_sitemap_reference',
                'missing_lang_attribute', 'robots_noindex'
            ],
            'on-page-seo' => [
                'short_title', 'long_title', 'title_keyword_stuffing', 'missing_meta_description',
                'empty_meta_description', 'short_meta_description', 'long_meta_description',
                'missing_alt_text', 'missing_h1', 'multiple_h1', 'missing_og_tags', 'missing_twitter_cards',
                'broken_link', 'broken_internal_links', 'no_internal_links'
            ],
            'performance' => [
                'no_cache_headers', 'no_compression', 'broken_image', 'broken_images', 'slow_load_time',
                'moderate_load_time', 'large_page_size', 'moderate_page_size', 'cwv_not_measured'
            ],
            'security' => [
                'missing_https', 'missing_security_header', 'mixed_content', 'missing_viewport'
            ],
            'content' => [
                'thin_content', 'short_content', 'missing_schema', 'missing_og_tags',
                'long_sentences', 'nofollow_set'
            ],
        ];
        
        $categories = [];
        
        foreach ($category_mapping as $category_id => $issue_types) {
            $types_string = "'" . implode("','", $issue_types) . "'";
            
            // Get unique issue counts for this category, including passed checks
            $result = $wpdb->get_row($wpdb->prepare("
                SELECT 
                    COUNT(DISTINCT CASE WHEN severity = 'critical' THEN CONCAT(type, ':', message) END) as critical,
                    COUNT(DISTINCT CASE WHEN severity = 'high' THEN CONCAT(type, ':', message) END) as high,
                    COUNT(DISTINCT CASE WHEN severity = 'medium' THEN CONCAT(type, ':', message) END) as medium,
                    COUNT(DISTINCT CASE WHEN severity = 'low' THEN CONCAT(type, ':', message) END) as low,
                    COUNT(DISTINCT CASE WHEN severity = 'passed' THEN type END) as passed,
                    COUNT(DISTINCT CASE WHEN severity != 'passed' THEN CONCAT(type, ':', message) END) as total_issues,
                    COUNT(DISTINCT type) as total
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                FROM {$this->audit_issues_table}
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                WHERE audit_id = %s AND type IN ($types_string)
            ", $audit_id));
            
            // Calculate totals properly
            $passed = (int)($result->passed ?? 0);
            $total_issues = (int)($result->total_issues ?? 0);
            $total_checks = (int)($result->total ?? 0);
            
            // If no data, use URL count for estimation
            if ($total_checks === 0) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $url_count = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$this->audit_urls_table} WHERE audit_id = %s AND status = 'checked'",
                    $audit_id
                ));
                // Estimate: each URL gets checked for each issue type in this category
                $total_checks = $url_count * count($issue_types);
                $passed = $total_checks; // All passed if no issues found
            }
            
            $categories[$category_id] = [
                'total' => $total_checks,
                'passed' => $passed,
                'warnings' => (int)(($result->medium ?? 0) + ($result->low ?? 0)),
                'critical' => (int)(($result->critical ?? 0) + ($result->high ?? 0)),
            ];
        }
        
        return $categories;
    }
    
    /**
     * Check canonical tags
     */
    private function check_canonical(\DOMDocument $dom, string $current_url): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        $canonicals = $xpath->query('//link[@rel="canonical"]');
        
        if ($canonicals->length === 0) {
            $result['warnings'][] = [
                'type' => 'missing_canonical',
                'severity' => 'medium',
                'message' => 'Missing canonical tag',
            ];
        } elseif ($canonicals->length > 1) {
            $result['issues'][] = [
                'type' => 'multiple_canonicals',
                'severity' => 'high',
                'message' => 'Multiple canonical tags found',
            ];
        } else {
            $result['passed'][] = 'has_canonical';
        }
        
        return $result;
    }
    
    /**
     * Check Open Graph tags
     */
    private function check_open_graph(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        
        $required_og_tags = ['og:title', 'og:description', 'og:image', 'og:url'];
        $missing_tags = [];
        
        foreach ($required_og_tags as $property) {
            $tags = $xpath->query("//meta[@property='$property']");
            if ($tags->length === 0) {
                $missing_tags[] = $property;
            }
        }
        
        if (count($missing_tags) > 0) {
            $result['warnings'][] = [
                'type' => 'missing_og_tags',
                'severity' => 'low',
                'message' => 'Missing Open Graph tags: ' . implode(', ', $missing_tags),
                'data' => ['missing' => $missing_tags],
            ];
        } else {
            $result['passed'][] = 'has_all_og_tags';
        }
        
        return $result;
    }
    
    /**
     * Check Schema markup
     */
    private function check_schema_markup(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        
        // Check for JSON-LD
        $scripts = $xpath->query('//script[@type="application/ld+json"]');
        if ($scripts->length === 0) {
            $result['warnings'][] = [
                'type' => 'missing_schema',
                'severity' => 'medium',
                'message' => 'No structured data (Schema.org) found',
            ];
        } else {
            $result['passed'][] = 'has_schema_markup';
        }
        
        return $result;
    }
    
    /**
     * Check content quality
     */
    private function check_content_quality(\DOMDocument $dom, string $body): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        // Remove script and style content
        $text_content = wp_strip_all_tags($body);
        $word_count = str_word_count($text_content);
        
        if ($word_count < 300) {
            $result['issues'][] = [
                'type' => 'thin_content',
                'severity' => 'high',
                'message' => sprintf('Thin content (%d words)', $word_count),
                'data' => ['word_count' => $word_count],
            ];
        } elseif ($word_count < 500) {
            $result['warnings'][] = [
                'type' => 'short_content',
                'severity' => 'medium',
                'message' => sprintf('Content could be longer (%d words)', $word_count),
                'data' => ['word_count' => $word_count],
            ];
        } else {
            $result['passed'][] = 'good_content_length';
        }
        
        return $result;
    }
    
    /**
     * Check security
     */
    private function check_security($headers, string $url): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        // Check HTTPS
        if (strpos($url, 'https://') !== 0) {
            $result['issues'][] = [
                'type' => 'missing_https',
                'severity' => 'critical',
                'message' => 'Not using HTTPS',
            ];
        } else {
            $result['passed'][] = 'uses_https';
        }
        
        // Check security headers
        if (!isset($headers['x-frame-options'])) {
            $result['warnings'][] = [
                'type' => 'missing_security_header',
                'severity' => 'medium',
                'message' => 'Missing X-Frame-Options header',
            ];
        } else {
            $result['passed'][] = 'has_x_frame_options';
        }
        
        return $result;
    }
    
    /**
     * Check accessibility
     */
    private function check_accessibility(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        // Check lang attribute
        $html = $dom->getElementsByTagName('html')->item(0);
        if ($html && !$html->hasAttribute('lang')) {
            $result['issues'][] = [
                'type' => 'missing_lang_attribute',
                'severity' => 'medium',
                'message' => 'Missing lang attribute on html element',
            ];
        } else if ($html) {
            $result['passed'][] = 'has_lang_attribute';
        }
        
        return $result;
    }
    
    /**
     * Make absolute URL
     */
    private function make_absolute_url(string $url, string $base): string {
        if (wp_parse_url($url, PHP_URL_SCHEME) !== null) {
            return $url;
        }
        
        if (strpos($url, '//') === 0) {
            return wp_parse_url($base, PHP_URL_SCHEME) . ':' . $url;
        }
        
        if (strpos($url, '/') === 0) {
            $parsed = wp_parse_url($base);
            return $parsed['scheme'] . '://' . $parsed['host'] . $url;
        }
        
        return rtrim($base, '/') . '/' . $url;
    }
    
    /**
     * Create database tables
     */
    private function maybe_create_tables(): void {
        global $wpdb;
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Audits table
        $sql1 = "CREATE TABLE {$this->audits_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(50) NOT NULL,
            status enum('idle','crawling','checking','completed','failed','stopped','stopping') DEFAULT 'idle',
            started_at datetime DEFAULT NULL,
            completed_at datetime DEFAULT NULL,
            total_urls int(11) DEFAULT 0,
            score int(3) DEFAULT NULL,
            stats longtext,
            options longtext,
            error text,
            PRIMARY KEY  (id),
            UNIQUE KEY audit_id (audit_id),
            KEY status (status),
            KEY started_at (started_at)
        ) $charset_collate;";
        
        // Audit URLs table
        $sql2 = "CREATE TABLE {$this->audit_urls_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(50) NOT NULL,
            url varchar(500) NOT NULL,
            status enum('pending','checking','checked') DEFAULT 'pending',
            issues_count int(11) DEFAULT 0,
            warnings_count int(11) DEFAULT 0,
            passed_count int(11) DEFAULT 0,
            score int(3) DEFAULT NULL,
            created_at datetime DEFAULT NULL,
            checked_at datetime DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY audit_id (audit_id),
            KEY status (status),
            KEY url (url(255))
        ) $charset_collate;";
        
        // Audit issues table
        $sql3 = "CREATE TABLE {$this->audit_issues_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(50) NOT NULL,
            url_id bigint(20) unsigned NOT NULL,
            type varchar(50) NOT NULL,
            severity enum('critical','high','medium','low','passed') DEFAULT 'medium',
            message text,
            data longtext,
            created_at datetime DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY audit_id (audit_id),
            KEY url_id (url_id),
            KEY type (type),
            KEY severity (severity)
        ) $charset_collate;";
        
        // Execute table creation
        dbDelta($sql1);
        dbDelta($sql2);
        dbDelta($sql3);
        
        // Log any errors
        if ($wpdb->last_error) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: Error creating audit tables: ' . $wpdb->last_error);
            }
        }
    }
    
    /**
     * Check Core Web Vitals
     */
    private function check_core_web_vitals(string $url): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        // In production, this would call PageSpeed Insights API
        // For now, we'll do basic performance checks
        $result['warnings'][] = [
            'type' => 'cwv_not_measured',
            'severity' => 'medium',
            'message' => 'Core Web Vitals not measured in this audit',
            'data' => ['recommendation' => 'Use Google PageSpeed Insights for detailed metrics'],
        ];
        
        return $result;
    }
    
    /**
     * Enhanced page speed check
     */
    private function check_page_speed(string $url, $headers, float $load_time, int $page_size): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        // Check load time
        if ($load_time > 3.0) {
            $result['issues'][] = [
                'type' => 'slow_load_time',
                'severity' => 'high',
                'message' => sprintf('Page loads slowly (%.2fs, should be under 3s)', $load_time),
                'data' => ['load_time' => $load_time, 'fix' => 'Optimize server response time and resources'],
            ];
        } elseif ($load_time > 2.0) {
            $result['warnings'][] = [
                'type' => 'moderate_load_time',
                'severity' => 'medium',
                'message' => sprintf('Page load time could be improved (%.2fs)', $load_time),
                'data' => ['load_time' => $load_time],
            ];
        } else {
            $result['passed'][] = 'fast_load_time';
        }
        
        // Check page size
        $size_mb = $page_size / (1024 * 1024);
        if ($size_mb > 3.0) {
            $result['issues'][] = [
                'type' => 'large_page_size',
                'severity' => 'high',
                'message' => sprintf('Page size too large (%.1fMB, should be under 3MB)', $size_mb),
                'data' => ['size_mb' => $size_mb, 'fix' => 'Optimize images and remove unnecessary resources'],
            ];
        } elseif ($size_mb > 2.0) {
            $result['warnings'][] = [
                'type' => 'moderate_page_size',
                'severity' => 'medium',
                'message' => sprintf('Page size could be reduced (%.1fMB)', $size_mb),
                'data' => ['size_mb' => $size_mb],
            ];
        } else {
            $result['passed'][] = 'optimal_page_size';
        }
        
        // Check compression
        if (!isset($headers['content-encoding']) || stripos($headers['content-encoding'], 'gzip') === false) {
            $result['issues'][] = [
                'type' => 'no_compression',
                'severity' => 'high',
                'message' => 'GZIP compression not enabled',
                'data' => ['fix' => 'Enable GZIP compression on your server'],
            ];
        } else {
            $result['passed'][] = 'gzip_enabled';
        }
        
        // Check caching
        if (!isset($headers['cache-control'])) {
            $result['issues'][] = [
                'type' => 'no_cache_headers',
                'severity' => 'high',
                'message' => 'No cache headers set',
                'data' => ['fix' => 'Set appropriate Cache-Control headers'],
            ];
        } else {
            $result['passed'][] = 'cache_headers_set';
        }
        
        return $result;
    }
    
    /**
     * Check robots meta tag
     */
    private function check_robots_meta(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        $robots = $xpath->query('//meta[@name="robots"]');
        
        if ($robots->length > 0) {
            $content = $robots->item(0)->getAttribute('content');
            if (stripos($content, 'noindex') !== false) {
                $result['issues'][] = [
                    'type' => 'noindex_set',
                    'severity' => 'critical',
                    'message' => 'Page is set to noindex',
                    'data' => ['content' => $content, 'fix' => 'Remove noindex if you want this page in search results'],
                ];
            } else {
                $result['passed'][] = 'indexable';
            }
            
            if (stripos($content, 'nofollow') !== false) {
                $result['warnings'][] = [
                    'type' => 'nofollow_set',
                    'severity' => 'medium',
                    'message' => 'Page is set to nofollow',
                    'data' => ['content' => $content],
                ];
            }
        } else {
            $result['passed'][] = 'no_blocking_robots_meta';
        }
        
        return $result;
    }
    
    /**
     * Check hreflang tags
     */
    private function check_hreflang(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        $hreflangs = $xpath->query('//link[@rel="alternate"][@hreflang]');
        
        if ($hreflangs->length > 0) {
            $found_self_reference = false;
            $found_x_default = false;
            
            foreach ($hreflangs as $tag) {
                $hreflang = $tag->getAttribute('hreflang');
                if ($hreflang === 'x-default') {
                    $found_x_default = true;
                }
                // Check for self-referencing hreflang
                // This is simplified - in production would check actual URL match
            }
            
            if (!$found_x_default) {
                $result['warnings'][] = [
                    'type' => 'missing_x_default_hreflang',
                    'severity' => 'low',
                    'message' => 'No x-default hreflang tag found',
                    'data' => ['fix' => 'Add x-default hreflang for international targeting'],
                ];
            }
            
            $result['passed'][] = 'has_hreflang_tags';
        }
        
        return $result;
    }
    
    /**
     * Check sitemap reference
     */
    private function check_sitemap_reference(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        $sitemap_links = $xpath->query('//link[@rel="sitemap"]');
        
        if ($sitemap_links->length > 0) {
            $result['passed'][] = 'has_sitemap_reference';
        }
        
        // Note: Full sitemap check would verify robots.txt
        
        return $result;
    }
    
    /**
     * Check Twitter Cards
     */
    private function check_twitter_cards(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        
        $required_twitter_tags = ['twitter:card', 'twitter:title', 'twitter:description'];
        $missing = [];
        
        foreach ($required_twitter_tags as $name) {
            $tags = $xpath->query("//meta[@name='$name']");
            if ($tags->length === 0) {
                $missing[] = $name;
            }
        }
        
        if (count($missing) > 0) {
            $result['warnings'][] = [
                'type' => 'missing_twitter_cards',
                'severity' => 'low',
                'message' => 'Missing Twitter Card tags for better social sharing',
                'data' => ['missing' => $missing, 'fix' => 'Add Twitter Card meta tags'],
            ];
        } else {
            $result['passed'][] = 'has_twitter_cards';
        }
        
        return $result;
    }
    
    /**
     * Check keyword optimization
     */
    private function check_keyword_optimization(\DOMDocument $dom, string $body): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        // This is a simplified check - production would use NLP
        $text_content = wp_strip_all_tags($body);
        $word_count = str_word_count($text_content);
        
        if ($word_count > 0) {
            // Check keyword density (simplified)
            $result['passed'][] = 'has_content_for_keywords';
        }
        
        return $result;
    }
    
    /**
     * Check readability
     */
    private function check_readability(string $body): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        $text_content = wp_strip_all_tags($body);
        $sentences = preg_split('/[.!?]+/', $text_content, -1, PREG_SPLIT_NO_EMPTY);
        $avg_sentence_length = count($sentences) > 0 ? str_word_count($text_content) / count($sentences) : 0;
        
        if ($avg_sentence_length > 25) {
            $result['warnings'][] = [
                'type' => 'long_sentences',
                'severity' => 'low',
                'message' => sprintf('Average sentence length is high (%.1f words)', $avg_sentence_length),
                'data' => ['avg_length' => $avg_sentence_length, 'fix' => 'Use shorter sentences for better readability'],
            ];
        } else {
            $result['passed'][] = 'good_sentence_length';
        }
        
        return $result;
    }
    
    /**
     * Check mixed content
     */
    private function check_mixed_content(\DOMDocument $dom, string $base_url): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        if (strpos($base_url, 'https://') === 0) {
            // Check for HTTP resources on HTTPS page
            $xpath = new \DOMXPath($dom);
            $http_resources = $xpath->query('//*[@src[starts-with(., "http://")]] | //*[@href[starts-with(., "http://")]]');
            
            if ($http_resources->length > 0) {
                $result['issues'][] = [
                    'type' => 'mixed_content',
                    'severity' => 'high',
                    'message' => sprintf('Found %d HTTP resources on HTTPS page', $http_resources->length),
                    'data' => ['count' => $http_resources->length, 'fix' => 'Update all resources to use HTTPS'],
                ];
            } else {
                $result['passed'][] = 'no_mixed_content';
            }
        }
        
        return $result;
    }
    
    /**
     * Check mobile optimization
     */
    private function check_mobile_optimization(\DOMDocument $dom): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        $xpath = new \DOMXPath($dom);
        
        // Check viewport meta tag
        $viewport = $xpath->query('//meta[@name="viewport"]');
        if ($viewport->length === 0) {
            $result['issues'][] = [
                'type' => 'missing_viewport',
                'severity' => 'high',
                'message' => 'Missing viewport meta tag',
                'data' => ['fix' => 'Add <meta name="viewport" content="width=device-width, initial-scale=1">'],
            ];
        } else {
            $result['passed'][] = 'has_viewport_meta';
        }
        
        return $result;
    }
    
    /**
     * Check international SEO
     */
    private function check_international_seo(\DOMDocument $dom, $headers): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];
        
        // Check language declaration
        $html = $dom->getElementsByTagName('html')->item(0);
        if ($html && !$html->hasAttribute('lang')) {
            $result['warnings'][] = [
                'type' => 'missing_lang_attribute',
                'severity' => 'medium',
                'message' => 'Missing lang attribute on HTML element',
                'data' => ['fix' => 'Add lang attribute to specify page language'],
            ];
        } else {
            $result['passed'][] = 'has_lang_attribute';
        }
        
        return $result;
    }
    private function is_disallowed_by_robots(string $url): bool {
        $robots_url = trailingslashit(home_url()) . 'robots.txt';
        $robots = get_transient('prorank_audit_robots');

        if ($robots === false) {
            $resp = wp_remote_get($robots_url, ['timeout' => 4, 'sslverify' => true]);
            if (is_wp_error($resp) || wp_remote_retrieve_response_code($resp) >= 400) {
                set_transient('prorank_audit_robots', '', 15 * MINUTE_IN_SECONDS);
                return false;
            }
            $robots = wp_remote_retrieve_body($resp);
            set_transient('prorank_audit_robots', $robots, 15 * MINUTE_IN_SECONDS);
        }

        if (empty($robots)) {
            return false;
        }

        $lines = explode("
", (string) $robots);
        $apply = false;
        $disallows = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line === '' || strpos($line, '#') === 0) {
                continue;
            }
            if (stripos($line, 'User-agent:') === 0) {
                $ua = trim(substr($line, 11));
                $apply = ($ua === '*' || stripos($ua, 'prorank') !== false);
                continue;
            }
            if ($apply && stripos($line, 'Disallow:') === 0) {
                $path = trim(substr($line, 9));
                if ($path !== '') {
                    $disallows[] = $path;
                }
            }
        }

        $url_path = wp_parse_url($url, PHP_URL_PATH) ?: '/';
        foreach ($disallows as $path) {
            if ($path === '/') {
                return true;
            }
            if (strpos($url_path, $path) === 0) {
                return true;
            }
        }

        return false;
    }


    private function mark_url_checked(int $url_id): void {
        global $wpdb;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update(
            $this->audit_urls_table,
            ['status' => 'checked'],
            ['id' => $url_id],
            ['%s'],
            ['%d']
        );
    }

    private function rate_limit_host(string $url): void {
        $host = wp_parse_url($url, PHP_URL_HOST);
        if (!$host) {
            return;
        }
        $key = 'prorank_audit_host_' . md5($host);
        $last = get_transient($key);
        $now = microtime(true);
        if ($last && ($now - (float)$last) < 0.35) {
            usleep(200000);
        }
        set_transient($key, $now, MINUTE_IN_SECONDS);
    }

    private function maybe_record_core_web_vitals(string $audit_id, string $url, $headers): void {
        $count_key = 'prorank_audit_vitals_count_' . $audit_id;
        $count = (int) get_option($count_key, 0);
        if ($count >= 10) {
            return;
        }

        $content_type = is_array($headers) && isset($headers['content-type']) ? (string) $headers['content-type'] : '';
        if ($content_type && stripos($content_type, 'text/html') === false) {
            return;
        }

        if (!class_exists('\ProRank\SEO\Core\Services\CoreWebVitalsService')) {
            return;
        }

        $service = new \ProRank\SEO\Core\Services\CoreWebVitalsService();
        $vitals = $service->get_vitals($url, 'mobile');
        if (is_wp_error($vitals)) {
            return;
        }

        $data_source = !empty($vitals['field_data']) ? $vitals['field_data'] : ($vitals['lab_data'] ?? []);
        if (empty($data_source)) {
            return;
        }

        $metrics = [];
        foreach (['lcp', 'cls', 'inp', 'ttfb'] as $metric) {
            if (isset($data_source[$metric])) {
                $value = $data_source[$metric]['value'] ?? 0;
                $category = strtolower($data_source[$metric]['category'] ?? $this->get_metric_rating($metric, $value));
                $metrics[$metric] = [
                    'value' => $value,
                    'rating' => $category,
                ];
            }
        }

        if (!empty($metrics)) {
            foreach ($metrics as $metric => $meta) {
                if ($meta['rating'] === 'poor') {
                    $this->add_issue($audit_id, $url, 'core_web_vitals_' . $metric, strtoupper($metric) . ' is poor', 'high');
                } elseif ($meta['rating'] === 'needs-improvement') {
                    $this->add_issue($audit_id, $url, 'core_web_vitals_' . $metric, strtoupper($metric) . ' needs improvement', 'medium');
                }
            }
        }

        update_option($count_key, $count + 1, false);
    }

    private function get_metric_rating(string $metric, float $value): string {
        $thresholds = [
            'lcp' => ['good' => 2500, 'poor' => 4000],
            'cls' => ['good' => 0.1, 'poor' => 0.25],
            'inp' => ['good' => 200, 'poor' => 500],
            'ttfb' => ['good' => 800, 'poor' => 1800],
        ];

        if (!isset($thresholds[$metric])) {
            return 'unknown';
        }

        if ($value <= $thresholds[$metric]['good']) {
            return 'good';
        }
        if ($value <= $thresholds[$metric]['poor']) {
            return 'needs-improvement';
        }
        return 'poor';
    }

    /**
     * Check XML Sitemap - validates sitemap exists and is properly formatted
     */
    private function check_xml_sitemap(): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];

        // Check for sitemap.xml at root
        $sitemap_url = trailingslashit(home_url()) . 'sitemap.xml';
        $response = wp_remote_get($sitemap_url, ['timeout' => 5, 'sslverify' => true]);

        if (is_wp_error($response)) {
            $result['issues'][] = [
                'type' => 'sitemap_unreachable',
                'severity' => 'high',
                'message' => 'XML sitemap is not accessible',
                'data' => ['url' => $sitemap_url, 'fix' => 'Create and submit an XML sitemap'],
            ];
            return $result;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 404) {
            $result['issues'][] = [
                'type' => 'sitemap_missing',
                'severity' => 'high',
                'message' => 'No XML sitemap found at /sitemap.xml',
                'data' => ['url' => $sitemap_url, 'fix' => 'Create an XML sitemap or check if it uses a different URL'],
            ];
            return $result;
        }

        if ($status_code !== 200) {
            $result['warnings'][] = [
                'type' => 'sitemap_error',
                'severity' => 'medium',
                'message' => sprintf('Sitemap returned HTTP %d', $status_code),
                'data' => ['status' => $status_code],
            ];
            return $result;
        }

        // Validate XML structure
        $body = wp_remote_retrieve_body($response);
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);

        if ($xml === false) {
            $errors = libxml_get_errors();
            libxml_clear_errors();
            $result['issues'][] = [
                'type' => 'sitemap_invalid_xml',
                'severity' => 'high',
                'message' => 'XML sitemap contains invalid XML',
                'data' => ['errors' => count($errors), 'fix' => 'Fix XML syntax errors in sitemap'],
            ];
            return $result;
        }

        // Check for URLs in sitemap
        $namespaces = $xml->getNamespaces(true);
        $ns = isset($namespaces['']) ? $namespaces[''] : '';

        if ($ns) {
            $xml->registerXPathNamespace('sm', $ns);
            $urls = $xml->xpath('//sm:url');
        } else {
            $urls = $xml->xpath('//url');
        }

        $url_count = is_array($urls) ? count($urls) : 0;

        if ($url_count === 0) {
            // Check if it's a sitemap index
            if ($ns) {
                $sitemaps = $xml->xpath('//sm:sitemap');
            } else {
                $sitemaps = $xml->xpath('//sitemap');
            }

            if (is_array($sitemaps) && count($sitemaps) > 0) {
                $result['passed'][] = 'has_sitemap_index';
                $result['passed'][] = sprintf('sitemap_index_%d_sitemaps', count($sitemaps));
            } else {
                $result['warnings'][] = [
                    'type' => 'sitemap_empty',
                    'severity' => 'medium',
                    'message' => 'XML sitemap exists but contains no URLs',
                    'data' => ['fix' => 'Add URLs to your sitemap'],
                ];
            }
        } else {
            $result['passed'][] = 'has_valid_sitemap';
            $result['passed'][] = sprintf('sitemap_%d_urls', $url_count);
        }

        return $result;
    }

    /**
     * Check for thin content - pages with very little content
     */
    private function check_thin_content(\DOMDocument $dom, string $body): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];

        // Remove script, style, and nav elements for accurate word count
        $xpath = new \DOMXPath($dom);

        // Get main content area if exists
        $main_content = $xpath->query('//main | //article | //*[@id="content"] | //*[@class="content"]');

        $text_content = '';
        if ($main_content->length > 0) {
            $text_content = $main_content->item(0)->textContent;
        } else {
            // Fallback to body content minus scripts/styles
            $text_content = wp_strip_all_tags($body);
        }

        // Clean up whitespace
        $text_content = preg_replace('/\s+/', ' ', trim($text_content));
        $word_count = str_word_count($text_content);

        // Calculate content-to-code ratio
        $content_length = strlen($text_content);
        $total_length = strlen($body);
        $content_ratio = $total_length > 0 ? ($content_length / $total_length) * 100 : 0;

        // Thin content thresholds
        if ($word_count < 100) {
            $result['issues'][] = [
                'type' => 'very_thin_content',
                'severity' => 'high',
                'message' => sprintf('Very thin content: only %d words (minimum 300 recommended)', $word_count),
                'data' => [
                    'word_count' => $word_count,
                    'content_ratio' => round($content_ratio, 1),
                    'fix' => 'Add more valuable content to this page',
                ],
            ];
        } elseif ($word_count < 300) {
            $result['warnings'][] = [
                'type' => 'thin_content',
                'severity' => 'medium',
                'message' => sprintf('Thin content: %d words (300+ recommended for SEO)', $word_count),
                'data' => [
                    'word_count' => $word_count,
                    'content_ratio' => round($content_ratio, 1),
                ],
            ];
        } else {
            $result['passed'][] = 'adequate_content_length';
        }

        // Check content-to-code ratio
        if ($content_ratio < 10 && $word_count < 500) {
            $result['warnings'][] = [
                'type' => 'low_content_ratio',
                'severity' => 'low',
                'message' => sprintf('Low content-to-code ratio (%.1f%%)', $content_ratio),
                'data' => ['ratio' => round($content_ratio, 1)],
            ];
        }

        return $result;
    }

    /**
     * Check for duplicate content - uses content hashing
     * Stores hash in audit meta for cross-page comparison
     */
    private function check_duplicate_content(string $audit_id, string $url, \DOMDocument $dom, string $body): array {
        global $wpdb;
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];

        // Extract main content for hashing
        $xpath = new \DOMXPath($dom);
        $main_content = $xpath->query('//main | //article | //*[@id="content"]');

        $text_content = '';
        if ($main_content->length > 0) {
            $text_content = $main_content->item(0)->textContent;
        } else {
            $text_content = wp_strip_all_tags($body);
        }

        // Normalize content for comparison
        $normalized = preg_replace('/\s+/', ' ', strtolower(trim($text_content)));
        $normalized = preg_replace('/[^a-z0-9\s]/', '', $normalized);

        // Skip very short content
        if (strlen($normalized) < 100) {
            return $result;
        }

        // Create content hash
        $content_hash = md5($normalized);

        // Check for existing pages with same hash
        $meta_table = $wpdb->prefix . 'prorank_audit_meta';

        // Ensure meta table exists
        $this->maybe_create_meta_table();

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT meta_value FROM {$meta_table}
             WHERE audit_id = %s AND meta_key = 'content_hash' AND meta_value = %s",
            $audit_id,
            $content_hash
        ));

        if ($existing) {
            // Find which URL has the duplicate
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $duplicate_url = $wpdb->get_var($wpdb->prepare(
                "SELECT m2.meta_value FROM {$meta_table} m1
                 JOIN {$meta_table} m2 ON m1.audit_id = m2.audit_id AND m1.url = m2.url
                 WHERE m1.audit_id = %s AND m1.meta_key = 'content_hash' AND m1.meta_value = %s
                 AND m2.meta_key = 'url' AND m2.meta_value != %s
                 LIMIT 1",
                $audit_id,
                $content_hash,
                $url
            ));

            $result['issues'][] = [
                'type' => 'duplicate_content',
                'severity' => 'high',
                'message' => 'Duplicate content detected',
                'data' => [
                    'duplicate_of' => $duplicate_url ?: 'another page',
                    'fix' => 'Use canonical tags or consolidate content',
                ],
            ];
        } else {
            // Store hash for future comparisons
            $wpdb->insert($meta_table, [
                'audit_id' => $audit_id,
                'url' => $url,
                'meta_key' => 'content_hash',
                'meta_value' => $content_hash,
            ]);
            $wpdb->insert($meta_table, [
                'audit_id' => $audit_id,
                'url' => $url,
                'meta_key' => 'url',
                'meta_value' => $url,
            ]);

            $result['passed'][] = 'unique_content';
        }

        return $result;
    }

    /**
     * Check crawl depth - pages too deep in site structure
     */
    private function check_crawl_depth(string $audit_id, int $url_id): array {
        global $wpdb;
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $depth = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT depth FROM {$this->audit_urls_table} WHERE id = %d",
            $url_id
        ));

        if ($depth > 4) {
            $result['issues'][] = [
                'type' => 'deep_page',
                'severity' => 'medium',
                'message' => sprintf('Page is %d clicks from homepage (max 3 recommended)', $depth),
                'data' => [
                    'depth' => $depth,
                    'fix' => 'Add internal links to make this page more accessible',
                ],
            ];
        } elseif ($depth > 3) {
            $result['warnings'][] = [
                'type' => 'moderately_deep_page',
                'severity' => 'low',
                'message' => sprintf('Page is %d clicks from homepage', $depth),
                'data' => ['depth' => $depth],
            ];
        } else {
            $result['passed'][] = 'good_crawl_depth';
        }

        return $result;
    }

    /**
     * Check for orphaned pages - pages with no internal links pointing to them
     * This is run at audit completion with full link data
     */
    private function check_orphaned_pages(string $audit_id): void {
        global $wpdb;

        $meta_table = $wpdb->prefix . 'prorank_audit_meta';
        $this->maybe_create_meta_table();

        // Get all internal link targets from the audit
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $link_targets = $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT meta_value FROM {$meta_table}
             WHERE audit_id = %s AND meta_key = 'internal_link_target'",
            $audit_id
        ));

        // Get all audited URLs
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $audited_urls = $wpdb->get_results($wpdb->prepare(
            "SELECT id, url FROM {$this->audit_urls_table} WHERE audit_id = %s",
            $audit_id
        ));

        $home_url = home_url();

        foreach ($audited_urls as $page) {
            // Skip homepage
            if (rtrim($page->url, '/') === rtrim($home_url, '/')) {
                continue;
            }

            // Normalize URL for comparison
            $normalized_url = rtrim($page->url, '/');

            // Check if any internal links point to this page
            $has_incoming_links = false;
            foreach ($link_targets as $target) {
                if (rtrim($target, '/') === $normalized_url) {
                    $has_incoming_links = true;
                    break;
                }
            }

            if (!$has_incoming_links) {
                $this->add_issue(
                    $audit_id,
                    $page->url,
                    'orphaned_page',
                    'Orphaned page - no internal links point here',
                    'medium',
                    ['fix' => 'Add internal links from other pages to this content']
                );
            }
        }
    }

    /**
     * Store internal link target for orphaned page detection
     */
    private function store_internal_link(string $audit_id, string $target_url): void {
        global $wpdb;
        $meta_table = $wpdb->prefix . 'prorank_audit_meta';

        $wpdb->insert($meta_table, [
            'audit_id' => $audit_id,
            'url' => '',
            'meta_key' => 'internal_link_target',
            'meta_value' => rtrim($target_url, '/'),
        ]);
    }

    /**
     * Store all internal links from a page for orphaned page detection
     */
    private function store_internal_links_for_orphan_detection(string $audit_id, \DOMDocument $dom, string $base_url): void {
        $this->maybe_create_meta_table();
        $home_url = home_url();
        $home_host = wp_parse_url($home_url, PHP_URL_HOST);

        $links = $dom->getElementsByTagName('a');
        $stored = [];

        foreach ($links as $link) {
            $href = $link->getAttribute('href');

            if (empty($href) || $href[0] === '#' || strpos($href, 'javascript:') === 0) {
                continue;
            }

            $absolute_url = $this->make_absolute_url($href, $base_url);
            $link_host = wp_parse_url($absolute_url, PHP_URL_HOST);

            // Only store internal links
            if ($link_host === $home_host) {
                $normalized = rtrim(strtok($absolute_url, '#'), '/');
                if (!isset($stored[$normalized])) {
                    $this->store_internal_link($audit_id, $normalized);
                    $stored[$normalized] = true;
                }
            }
        }
    }

    /**
     * Check WordPress health using Site Health API
     */
    private function check_wordpress_health(): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];

        // Check WordPress version
        global $wp_version;
        $latest_version = get_site_transient('update_core');

        if ($latest_version && isset($latest_version->updates[0]->version)) {
            if (version_compare($wp_version, $latest_version->updates[0]->version, '<')) {
                $result['warnings'][] = [
                    'type' => 'wordpress_outdated',
                    'severity' => 'medium',
                    'message' => sprintf('WordPress %s is outdated (latest: %s)', $wp_version, $latest_version->updates[0]->version),
                    'data' => [
                        'current' => $wp_version,
                        'latest' => $latest_version->updates[0]->version,
                        'fix' => 'Update WordPress to the latest version',
                    ],
                ];
            } else {
                $result['passed'][] = 'wordpress_up_to_date';
            }
        }

        // Check PHP version
        $php_version = phpversion();
        if (version_compare($php_version, '8.0', '<')) {
            $result['warnings'][] = [
                'type' => 'php_outdated',
                'severity' => 'medium',
                'message' => sprintf('PHP %s is outdated (8.0+ recommended)', $php_version),
                'data' => ['current' => $php_version, 'fix' => 'Upgrade PHP to 8.0 or higher'],
            ];
        } else {
            $result['passed'][] = 'php_up_to_date';
        }

        // Check debug mode
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $result['warnings'][] = [
                'type' => 'debug_enabled',
                'severity' => 'low',
                'message' => 'WP_DEBUG is enabled on production',
                'data' => ['fix' => 'Disable WP_DEBUG in wp-config.php'],
            ];
        } else {
            $result['passed'][] = 'debug_disabled';
        }

        // Check SSL
        if (!is_ssl()) {
            $result['issues'][] = [
                'type' => 'no_ssl',
                'severity' => 'high',
                'message' => 'Site is not using HTTPS',
                'data' => ['fix' => 'Install SSL certificate and enable HTTPS'],
            ];
        } else {
            $result['passed'][] = 'ssl_enabled';
        }

        // Check memory limit
        $memory_limit = wp_convert_hr_to_bytes(WP_MEMORY_LIMIT);
        if ($memory_limit < 256 * 1024 * 1024) {
            $result['warnings'][] = [
                'type' => 'low_memory_limit',
                'severity' => 'low',
                'message' => sprintf('Memory limit is %s (256M+ recommended)', WP_MEMORY_LIMIT),
                'data' => ['current' => WP_MEMORY_LIMIT, 'fix' => 'Increase WP_MEMORY_LIMIT in wp-config.php'],
            ];
        } else {
            $result['passed'][] = 'adequate_memory';
        }

        return $result;
    }

    /**
     * Check for plugin conflicts and issues
     */
    private function check_plugin_conflicts(): array {
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];

        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $all_plugins = get_plugins();
        $active_plugins = get_option('active_plugins', []);

        // Known problematic plugins or combinations
        $problematic_plugins = [
            // SEO plugins
            'wordpress-seo/wp-seo.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'wordpress-seo-premium/wp-seo-premium.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'wpseo-local/wpseo-local.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank local SEO features'],
            'wpseo-video/wpseo-video.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank video schema'],
            'wpseo-news/wpseo-news.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank news sitemaps'],
            'woocommerce-seo/woocommerce-seo.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank WooCommerce SEO'],
            'wpseo-woocommerce/wpseo-woocommerce.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank WooCommerce SEO'],
            'seo-by-rank-math/rank-math.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'seo-by-rank-math-pro/rank-math-pro.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'all-in-one-seo-pack/all_in_one_seo_pack.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'all-in-one-seo-pack-pro/all_in_one_seo_pack.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'wp-seopress/seopress.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'wp-seopress-pro/seopress-pro.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata, schema, and sitemaps'],
            'autodescription/autodescription.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata and schema'],
            'slim-seo/slim-seo.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank metadata and schema'],
            'squirrly-seo/squirrly.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank SEO features'],
            'smartcrawl-seo/wpmu-dev-seo.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank SEO features'],
            'seokey/seokey.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank SEO features'],
            'seo-ultimate/seo-ultimate.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank SEO features'],
            'wp-meta-seo/wp-meta-seo.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank SEO features'],
            'premium-seo-pack/index.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank SEO features'],
            'seo-booster/seo-booster.php' => ['severity' => 'medium', 'reason' => 'Another SEO plugin - may conflict with ProRank SEO features'],

            // Performance/caching plugins
            'w3-total-cache/w3-total-cache.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'wp-rocket/wp-rocket.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'litespeed-cache/litespeed-cache.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'wp-super-cache/wp-cache.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'wp-fastest-cache/wpFastestCache.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'cache-enabler/cache-enabler.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'autoptimize/autoptimize.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'hummingbird-performance/wp-hummingbird.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'sg-cachepress/sg-cachepress.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'nitropack/nitropack.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],
            'perfmatters/perfmatters.php' => ['severity' => 'low', 'reason' => 'Performance plugin - may overlap with ProRank optimizations'],
            'flying-press/flying-press.php' => ['severity' => 'low', 'reason' => 'Performance plugin - may overlap with ProRank optimizations'],
            'asset-cleanup/asset-cleanup.php' => ['severity' => 'low', 'reason' => 'Asset optimization plugin - may overlap with ProRank asset controls'],
            'async-javascript/async-javascript.php' => ['severity' => 'low', 'reason' => 'Asset optimization plugin - may overlap with ProRank asset controls'],
            'wp-optimize/wp-optimize.php' => ['severity' => 'low', 'reason' => 'Optimization plugin - may overlap with ProRank performance features'],
            'breeze/breeze.php' => ['severity' => 'low', 'reason' => 'Caching/optimization plugin - may overlap with ProRank performance features'],

            // Image optimization plugins
            'shortpixel-image-optimiser/wp-shortpixel.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'imagify/imagify.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'wp-smushit/wp-smush.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'ewww-image-optimizer/ewww-image-optimizer.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'optimole-wp/optimole-wp.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'tiny-compress-images/tiny-compress-images.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'webp-express/webp-express.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'converter-for-media/converter-for-media.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
            'seo-optimized-images/seo-optimized-images.php' => ['severity' => 'low', 'reason' => 'Image optimization plugin - may overlap with ProRank image optimization'],
        ];

        $conflicts_found = 0;

        foreach ($active_plugins as $plugin) {
            if (isset($problematic_plugins[$plugin])) {
                $info = $problematic_plugins[$plugin];
                $plugin_data = $all_plugins[$plugin] ?? ['Name' => $plugin];

                $result['warnings'][] = [
                    'type' => 'potential_plugin_conflict',
                    'severity' => $info['severity'],
                    'message' => sprintf('Potential conflict: %s', $plugin_data['Name']),
                    'data' => ['plugin' => $plugin, 'reason' => $info['reason']],
                ];
                $conflicts_found++;
            }
        }

        // Check for plugins needing updates
        $update_plugins = get_site_transient('update_plugins');
        $outdated_count = 0;

        if ($update_plugins && !empty($update_plugins->response)) {
            $outdated_count = count($update_plugins->response);
            if ($outdated_count > 0) {
                $result['warnings'][] = [
                    'type' => 'plugins_need_updates',
                    'severity' => 'medium',
                    'message' => sprintf('%d plugin(s) have updates available', $outdated_count),
                    'data' => ['count' => $outdated_count, 'fix' => 'Update plugins to latest versions'],
                ];
            }
        }

        // Check plugin count
        $active_count = count($active_plugins);
        if ($active_count > 30) {
            $result['warnings'][] = [
                'type' => 'too_many_plugins',
                'severity' => 'low',
                'message' => sprintf('%d active plugins (consider reducing for performance)', $active_count),
                'data' => ['count' => $active_count],
            ];
        }

        if ($conflicts_found === 0 && $outdated_count === 0) {
            $result['passed'][] = 'no_plugin_issues';
        }

        return $result;
    }

    /**
     * Check database optimization status
     */
    private function check_database_optimization(): array {
        global $wpdb;
        $result = ['issues' => [], 'warnings' => [], 'passed' => []];

        // Check autoloaded options size
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $autoload_size = $wpdb->get_var(
            "SELECT SUM(LENGTH(option_value)) FROM {$wpdb->options} WHERE autoload = 'yes'"
        );
        $autoload_mb = $autoload_size / (1024 * 1024);

        if ($autoload_mb > 2) {
            $result['issues'][] = [
                'type' => 'large_autoload',
                'severity' => 'high',
                'message' => sprintf('Autoloaded options are %.1fMB (should be under 1MB)', $autoload_mb),
                'data' => [
                    'size_mb' => round($autoload_mb, 2),
                    'fix' => 'Review and clean up autoloaded options',
                ],
            ];
        } elseif ($autoload_mb > 1) {
            $result['warnings'][] = [
                'type' => 'moderate_autoload',
                'severity' => 'medium',
                'message' => sprintf('Autoloaded options are %.1fMB', $autoload_mb),
                'data' => ['size_mb' => round($autoload_mb, 2)],
            ];
        } else {
            $result['passed'][] = 'optimal_autoload_size';
        }

        // Check transients
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $expired_transients = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->options}
             WHERE option_name LIKE '%_transient_timeout_%'
             AND option_value < UNIX_TIMESTAMP()"
        );

        if ($expired_transients > 100) {
            $result['warnings'][] = [
                'type' => 'expired_transients',
                'severity' => 'low',
                'message' => sprintf('%d expired transients in database', $expired_transients),
                'data' => ['count' => $expired_transients, 'fix' => 'Clean up expired transients'],
            ];
        } else {
            $result['passed'][] = 'transients_clean';
        }

        // Check post revisions
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $revision_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'revision'"
        );

        if ($revision_count > 1000) {
            $result['warnings'][] = [
                'type' => 'many_revisions',
                'severity' => 'low',
                'message' => sprintf('%d post revisions stored (consider limiting)', $revision_count),
                'data' => ['count' => $revision_count, 'fix' => 'Limit revisions with WPREVISIONS constant'],
            ];
        } else {
            $result['passed'][] = 'revisions_manageable';
        }

        // Check spam comments
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $spam_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = 'spam'"
        );

        if ($spam_count > 100) {
            $result['warnings'][] = [
                'type' => 'spam_comments',
                'severity' => 'low',
                'message' => sprintf('%d spam comments in database', $spam_count),
                'data' => ['count' => $spam_count, 'fix' => 'Empty spam comments'],
            ];
        }

        // Check trash posts
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $trash_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'trash'"
        );

        if ($trash_count > 50) {
            $result['warnings'][] = [
                'type' => 'trash_posts',
                'severity' => 'low',
                'message' => sprintf('%d posts in trash', $trash_count),
                'data' => ['count' => $trash_count, 'fix' => 'Empty trash'],
            ];
        }

        return $result;
    }

    /**
     * Create meta table for audit-level data storage
     */
    private function maybe_create_meta_table(): void {
        global $wpdb;
        $meta_table = $wpdb->prefix . 'prorank_audit_meta';

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        if ($wpdb->get_var("SHOW TABLES LIKE '{$meta_table}'") === $meta_table) {
            return;
        }

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$meta_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(50) NOT NULL,
            url varchar(500) DEFAULT '',
            meta_key varchar(255) NOT NULL,
            meta_value longtext,
            PRIMARY KEY (id),
            KEY audit_id (audit_id),
            KEY meta_key (meta_key(191)),
            KEY audit_meta (audit_id, meta_key(191))
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Add issue to database
     */
    private function add_issue(string $audit_id, string $url, string $type, string $message, string $severity, array $data = []): void {
        global $wpdb;

        // Get URL ID
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $url_id = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$this->audit_urls_table} WHERE audit_id = %s AND url = %s LIMIT 1",
            $audit_id,
            $url
        ));

        if (!$url_id) {
            return;
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->insert($this->audit_issues_table, [
            'audit_id' => $audit_id,
            'url_id' => $url_id,
            'type' => $type,
            'severity' => $severity,
            'message' => $message,
            'data' => json_encode($data),
            'created_at' => current_time('mysql'),
        ], ['%s', '%d', '%s', '%s', '%s', '%s', '%s']);
    }

    /**
     * Enhanced structured data validation
     */
    private function check_structured_data_enhanced(string $audit_id, string $url, string $body): void {
        // Find all JSON-LD scripts
        if (!preg_match_all('/<script[^>]+type=["\']application\/ld\+json["\'][^>]*>(.*?)<\/script>/si', $body, $matches)) {
            $this->add_issue($audit_id, $url, 'missing_structured_data', 'No JSON-LD structured data found', 'medium');
            return;
        }

        $found_types = [];
        $has_errors = false;

        foreach ($matches[1] as $json_content) {
            $data = json_decode(trim($json_content), true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                $this->add_issue($audit_id, $url, 'invalid_json_ld', 'Invalid JSON-LD: ' . json_last_error_msg(), 'high');
                $has_errors = true;
                continue;
            }

            // Handle @graph structures
            $items = isset($data['@graph']) ? $data['@graph'] : [$data];

            foreach ($items as $item) {
                if (isset($item['@type'])) {
                    $type = is_array($item['@type']) ? implode(', ', $item['@type']) : $item['@type'];
                    $found_types[] = $type;

                    // Validate required fields for common types
                    $this->validate_schema_type($audit_id, $url, $type, $item);
                }
            }
        }

        if (!$has_errors && !empty($found_types)) {
            // Structured data is present and valid
            return;
        }
    }

    /**
     * Validate specific schema types
     */
    private function validate_schema_type(string $audit_id, string $url, string $type, array $data): void {
        $required_fields = [
            'Article' => ['headline', 'author', 'datePublished'],
            'Product' => ['name', 'description'],
            'Organization' => ['name', 'url'],
            'LocalBusiness' => ['name', 'address'],
            'BreadcrumbList' => ['itemListElement'],
            'FAQPage' => ['mainEntity'],
            'HowTo' => ['name', 'step'],
            'Recipe' => ['name', 'recipeIngredient', 'recipeInstructions'],
        ];

        if (!isset($required_fields[$type])) {
            return;
        }

        $missing = [];
        foreach ($required_fields[$type] as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                $missing[] = $field;
            }
        }

        if (!empty($missing)) {
            $this->add_issue(
                $audit_id,
                $url,
                'incomplete_schema_' . strtolower($type),
                sprintf('%s schema missing required fields: %s', $type, implode(', ', $missing)),
                'medium',
                ['type' => $type, 'missing_fields' => $missing]
            );
        }
    }

    /**
     * Check security headers
     */
    private function check_security_headers(string $audit_id, string $url, $headers): void {
        $security_headers = [
            'x-frame-options' => ['severity' => 'medium', 'message' => 'Missing X-Frame-Options header (clickjacking protection)'],
            'x-content-type-options' => ['severity' => 'medium', 'message' => 'Missing X-Content-Type-Options header'],
            'x-xss-protection' => ['severity' => 'low', 'message' => 'Missing X-XSS-Protection header'],
            'strict-transport-security' => ['severity' => 'high', 'message' => 'Missing Strict-Transport-Security (HSTS) header'],
            'content-security-policy' => ['severity' => 'low', 'message' => 'Missing Content-Security-Policy header'],
            'referrer-policy' => ['severity' => 'low', 'message' => 'Missing Referrer-Policy header'],
            'permissions-policy' => ['severity' => 'low', 'message' => 'Missing Permissions-Policy header'],
        ];

        $headers_array = is_object($headers) ? $headers->getAll() : (array) $headers;
        $headers_lower = array_change_key_case($headers_array, CASE_LOWER);

        foreach ($security_headers as $header => $config) {
            if (!isset($headers_lower[$header])) {
                $this->add_issue($audit_id, $url, 'missing_security_header_' . str_replace('-', '_', $header), $config['message'], $config['severity']);
            }
        }
    }

    /**
     * Check redirect chains
     */
    private function check_redirect_chain(string $audit_id, string $original_url): void {
        $redirects = [];
        $current_url = $original_url;
        $max_redirects = 10;

        for ($i = 0; $i < $max_redirects; $i++) {
            $response = wp_remote_head($current_url, [
                'timeout' => 5,
                'redirection' => 0, // Don't follow redirects
            ]);

            if (is_wp_error($response)) {
                break;
            }

            $code = wp_remote_retrieve_response_code($response);

            if ($code >= 300 && $code < 400) {
                $location = wp_remote_retrieve_header($response, 'location');
                if ($location) {
                    $redirects[] = ['from' => $current_url, 'to' => $location, 'code' => $code];
                    $current_url = $location;
                } else {
                    break;
                }
            } else {
                break;
            }
        }

        $chain_length = count($redirects);

        if ($chain_length >= 3) {
            $this->add_issue(
                $audit_id,
                $original_url,
                'long_redirect_chain',
                sprintf('Long redirect chain detected (%d redirects)', $chain_length),
                'high',
                ['chain' => $redirects]
            );
        } elseif ($chain_length === 2) {
            $this->add_issue(
                $audit_id,
                $original_url,
                'redirect_chain',
                'Redirect chain detected (2 redirects)',
                'medium',
                ['chain' => $redirects]
            );
        }

        // Check for redirect loops
        $urls_seen = array_column($redirects, 'from');
        if (count($urls_seen) !== count(array_unique($urls_seen))) {
            $this->add_issue($audit_id, $original_url, 'redirect_loop', 'Redirect loop detected', 'critical', ['chain' => $redirects]);
        }
    }

    /**
     * Enhanced hreflang validation
     */
    private function check_hreflang_enhanced(string $audit_id, string $url, \DOMDocument $dom): void {
        $xpath = new \DOMXPath($dom);
        $hreflangs = $xpath->query('//link[@rel="alternate"][@hreflang]');

        if ($hreflangs->length === 0) {
            return; // No hreflang - not necessarily an issue
        }

        $has_self = false;
        $has_x_default = false;
        $lang_codes = [];
        $issues = [];

        foreach ($hreflangs as $tag) {
            $hreflang = strtolower($tag->getAttribute('hreflang'));
            $href = $tag->getAttribute('href');

            if ($hreflang === 'x-default') {
                $has_x_default = true;
            }

            // Check for self-referencing
            if ($this->urls_match($href, $url)) {
                $has_self = true;
            }

            // Check for duplicate language codes
            if (isset($lang_codes[$hreflang])) {
                $issues[] = "Duplicate hreflang: $hreflang";
            }
            $lang_codes[$hreflang] = $href;

            // Validate language code format
            if ($hreflang !== 'x-default' && !preg_match('/^[a-z]{2}(-[a-z]{2})?$/i', $hreflang)) {
                $issues[] = "Invalid hreflang format: $hreflang";
            }
        }

        if (!$has_self && $hreflangs->length > 0) {
            $this->add_issue($audit_id, $url, 'missing_self_hreflang', 'Hreflang tags present but missing self-referencing tag', 'medium');
        }

        if (!$has_x_default && $hreflangs->length > 1) {
            $this->add_issue($audit_id, $url, 'missing_x_default_hreflang', 'Multiple hreflang tags but no x-default', 'low');
        }

        foreach ($issues as $issue) {
            $this->add_issue($audit_id, $url, 'hreflang_error', $issue, 'medium');
        }
    }

    /**
     * Check if two URLs match (ignoring trailing slashes and protocol)
     */
    private function urls_match(string $url1, string $url2): bool {
        $normalize = function($url) {
            $url = preg_replace('/^https?:\/\//', '', $url);
            $url = rtrim($url, '/');
            return strtolower($url);
        };

        return $normalize($url1) === $normalize($url2);
    }

    /**
     * Enhanced canonical check
     */
    private function check_canonical_enhanced(string $audit_id, string $url, \DOMDocument $dom): void {
        $xpath = new \DOMXPath($dom);
        $canonicals = $xpath->query('//link[@rel="canonical"]');

        if ($canonicals->length === 0) {
            $this->add_issue($audit_id, $url, 'missing_canonical', 'Missing canonical tag', 'medium');
            return;
        }

        if ($canonicals->length > 1) {
            $this->add_issue($audit_id, $url, 'multiple_canonicals', 'Multiple canonical tags found', 'high');
            return;
        }

        $canonical_href = $canonicals->item(0)->getAttribute('href');

        if (empty($canonical_href)) {
            $this->add_issue($audit_id, $url, 'empty_canonical', 'Canonical tag has empty href', 'high');
            return;
        }

        // Check if canonical points to a different page
        if (!$this->urls_match($canonical_href, $url)) {
            // This could be intentional (for duplicate content), just note it
            $this->add_issue(
                $audit_id,
                $url,
                'canonical_different_url',
                'Canonical points to different URL: ' . $canonical_href,
                'low',
                ['canonical' => $canonical_href]
            );
        }

        // Check if canonical URL is absolute
        if (!preg_match('/^https?:\/\//', $canonical_href)) {
            $this->add_issue($audit_id, $url, 'relative_canonical', 'Canonical URL should be absolute', 'medium');
        }
    }

}
