<?php
/**
 * Competitor Detector Service
 *
 * Detects active SEO and performance plugins that may conflict with ProRank SEO
 *
 * @package ProRank\SEO\Core\Compatibility
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Compatibility;

defined( 'ABSPATH' ) || exit;

/**
 * CompetitorDetector class
 *
 * Service to detect other active SEO plugins
 */
class CompetitorDetector {
    
    /**
     * Map of known competitor plugins
     *
     * @var array<string, string>
     */
    private static array $competitor_plugins = [
        // Yoast SEO + add-ons
        'wordpress-seo/wp-seo.php' => 'Yoast SEO',
        'wordpress-seo-premium/wp-seo-premium.php' => 'Yoast SEO Premium',
        'wpseo-local/wpseo-local.php' => 'Yoast Local SEO',
        'wpseo-video/wpseo-video.php' => 'Yoast Video SEO',
        'wpseo-news/wpseo-news.php' => 'Yoast News SEO',
        'woocommerce-seo/woocommerce-seo.php' => 'Yoast WooCommerce SEO',
        'wpseo-woocommerce/wpseo-woocommerce.php' => 'Yoast WooCommerce SEO',
        
        // Rank Math
        'seo-by-rank-math/rank-math.php' => 'Rank Math SEO',
        'seo-by-rank-math-pro/rank-math-pro.php' => 'Rank Math SEO Pro',

        // SureRank (bundled with Starter Templates/Astra)
        'surerank/surerank.php' => 'SureRank SEO',

        // SureTriggers / SureCart ecosystem
        'suretriggers/suretriggers.php' => 'SureTriggers',
        
        // All in One SEO
        'all-in-one-seo-pack/all_in_one_seo_pack.php' => 'All in One SEO',
        'all-in-one-seo-pack-pro/all_in_one_seo_pack.php' => 'All in One SEO Pro',
        
        // SEOPress
        'wp-seopress/seopress.php' => 'SEOPress',
        'wp-seopress-pro/seopress-pro.php' => 'SEOPress Pro',
        
        // The SEO Framework
        'autodescription/autodescription.php' => 'The SEO Framework',

        // SEOKEY
        'seokey/seokey.php' => 'SEOKEY',
        
        // Squirrly SEO
        'squirrly-seo/squirrly.php' => 'Squirrly SEO',
        
        // SmartCrawl
        'smartcrawl-seo/wpmu-dev-seo.php' => 'SmartCrawl SEO',
        
        // SEO Ultimate
        'seo-ultimate/seo-ultimate.php' => 'SEO Ultimate',
        
        // Platinum SEO
        'platinum-seo-pack/platinum-seo-pack.php' => 'Platinum SEO Pack',
        
        // Greg's High Performance SEO
        'gregs-high-performance-seo/ghpseo.php' => 'Greg\'s High Performance SEO',
        
        // WP Meta SEO
        'wp-meta-seo/wp-meta-seo.php' => 'WP Meta SEO',
        
        // Premium SEO Pack
        'premium-seo-pack/index.php' => 'Premium SEO Pack',
        
        // SEO Booster
        'seo-booster/seo-booster.php' => 'SEO Booster',
        
        // SEO Optimized Images
        'seo-optimized-images/seo-optimized-images.php' => 'SEO Optimized Images',
        
        // Slim SEO
        'slim-seo/slim-seo.php' => 'Slim SEO',
        'slim-seo-schema/slim-seo-schema.php' => 'Slim SEO Schema',
        'slim-seo-link-manager/slim-seo-link-manager.php' => 'Slim SEO Link Manager',
        'slim-seo-redirection/slim-seo-redirection.php' => 'Slim SEO Redirection',
        
        // SEO 2
        'seo-2/seo.php' => 'SEO 2',
        
        // Simply SEO
        'simply-seo/simply-seo.php' => 'Simply SEO',
        
        // SEO Plugin by Squirrly
        'seo-plugin-by-squirrly/squirrly.php' => 'SEO Plugin by Squirrly',

        // Performance / caching / optimization
        'wp-rocket/wp-rocket.php' => 'WP Rocket',
        'litespeed-cache/litespeed-cache.php' => 'LiteSpeed Cache',
        'wp-super-cache/wp-cache.php' => 'WP Super Cache',
        'wp-fastest-cache/wpFastestCache.php' => 'WP Fastest Cache',
        'cache-enabler/cache-enabler.php' => 'Cache Enabler',
        'autoptimize/autoptimize.php' => 'Autoptimize',
        'hummingbird-performance/wp-hummingbird.php' => 'Hummingbird',
        'sg-cachepress/sg-cachepress.php' => 'SiteGround Optimizer',
        'nitropack/nitropack.php' => 'NitroPack',
        'perfmatters/perfmatters.php' => 'Perfmatters',
        'flying-press/flying-press.php' => 'FlyingPress',
        'flying-scripts/flying-scripts.php' => 'Flying Scripts',
        'asset-cleanup/asset-cleanup.php' => 'Asset CleanUp',
        'asset-cleanup-pro/asset-cleanup.php' => 'Asset CleanUp Pro',
        'async-javascript/async-javascript.php' => 'Async JavaScript',
        'wp-optimize/wp-optimize.php' => 'WP-Optimize',
        'swift-performance-lite/performance.php' => 'Swift Performance Lite',
        'comet-cache/comet-cache.php' => 'Comet Cache',
        'breeze/breeze.php' => 'Breeze',
        'speedycache/speedycache.php' => 'SpeedyCache',
        
        // W3 Total Cache (has SEO features)
        'w3-total-cache/w3-total-cache.php' => 'W3 Total Cache (SEO Features)',
        
        // Jetpack (has SEO features)
        'jetpack/jetpack.php' => 'Jetpack (SEO Features)',

        // Image optimization
        'shortpixel-image-optimiser/wp-shortpixel.php' => 'ShortPixel Image Optimizer',
        'imagify/imagify.php' => 'Imagify',
        'wp-smushit/wp-smush.php' => 'Smush',
        'ewww-image-optimizer/ewww-image-optimizer.php' => 'EWWW Image Optimizer',
        'optimole-wp/optimole-wp.php' => 'Optimole',
        'tiny-compress-images/tiny-compress-images.php' => 'TinyPNG',
        'webp-express/webp-express.php' => 'WebP Express',
        'converter-for-media/converter-for-media.php' => 'Converter for Media',
        're-smush-it/resmushit.php' => 'reSmush.it Image Optimizer',
        'robin-image-optimizer/robin-image-optimizer.php' => 'Robin Image Optimizer',
    ];
    
    /**
     * Get active competitor plugins
     *
     * @return array<string> Array of active competitor plugin names
     */
    public function get_active_competitors(): array {
        // Ensure plugin functions are available
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $active_competitors = [];
        
        foreach (self::$competitor_plugins as $plugin_file => $plugin_name) {
            if (is_plugin_active($plugin_file)) {
                $active_competitors[] = $plugin_name;
            }
        }
        
        return $active_competitors;
    }
    
    /**
     * Get active competitor plugins with details
     *
     * @return array Array of active competitor details
     */
    public function get_active_competitors_details(): array {
        // Ensure plugin functions are available
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $active_competitors = [];
        
        foreach (self::$competitor_plugins as $plugin_file => $plugin_name) {
            if (is_plugin_active($plugin_file)) {
                $active_competitors[] = [
                    'name' => $plugin_name,
                    'file' => $plugin_file,
                    'deactivate_url' => wp_nonce_url(
                        admin_url('plugins.php?action=deactivate&plugin=' . urlencode($plugin_file)),
                        'deactivate-plugin_' . $plugin_file
                    )
                ];
            }
        }
        
        return $active_competitors;
    }
    
    /**
     * Check if any competitor is active
     *
     * @return bool True if at least one competitor is active
     */
    public function has_active_competitors(): bool {
        return !empty($this->get_active_competitors());
    }
    
    /**
     * Get competitor count
     *
     * @return int Number of active competitor plugins
     */
    public function get_competitor_count(): int {
        return count($this->get_active_competitors());
    }
    
    /**
     * Check if a specific competitor is active
     *
     * @param string $plugin_slug Plugin slug (e.g., 'wordpress-seo', 'rank-math')
     * @return bool True if the competitor is active
     */
    public function is_competitor_active(string $plugin_slug): bool {
        $active_competitors = $this->get_active_competitors();
        
        // Check by plugin name
        foreach (self::$competitor_plugins as $plugin_file => $plugin_name) {
            if (strpos($plugin_file, $plugin_slug) !== false && in_array($plugin_name, $active_competitors, true)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get all known competitor plugins
     *
     * @return array<string, string> Map of plugin files to names
     */
    public static function get_all_competitors(): array {
        return self::$competitor_plugins;
    }
    
    /**
     * Get warning message for active competitors
     *
     * @return string|null Warning message or null if no competitors
     */
    public function get_warning_message(): ?string {
        $active_competitors = $this->get_active_competitors();
        
        if (empty($active_competitors)) {
            return null;
        }
        
        $count = count($active_competitors);
        
        if ($count === 1) {
            return sprintf(
                /* translators: %s: placeholder value */
                __('Warning: %s is currently active. Running multiple SEO or performance optimization plugins can cause conflicts. Consider deactivating it before using ProRank SEO.', 'prorank-seo'),
                $active_competitors[0]
            );
        }
        
        return sprintf(
            /* translators: 1: plugin count 2: plugin names */
            __('Warning: %1\$d SEO or performance optimization plugins are currently active (%2\$s). Running multiple optimization plugins can cause conflicts. Consider deactivating them before using ProRank SEO.', 'prorank-seo'),
            $count,
            implode(', ', $active_competitors)
        );
    }
    
    /**
     * Get health check data
     *
     * @return array Health check information
     */
    public function get_health_check_data(): array {
        $active_competitors = $this->get_active_competitors();
        $has_conflicts = !empty($active_competitors);
        
        return [
            'status' => $has_conflicts ? 'warning' : 'good',
            'label' => $has_conflicts 
                ? __('Potential optimization plugin conflicts detected', 'prorank-seo')
                : __('No optimization plugin conflicts', 'prorank-seo'),
            'description' => $has_conflicts
                ? $this->get_warning_message()
                : __('No other SEO or performance optimization plugins detected. ProRank SEO can run without conflicts.', 'prorank-seo'),
            'competitors' => $active_competitors,
            'count' => count($active_competitors),
            'test' => 'prorank_seo_competitor_check',
        ];
    }
}
