<?php
/**
 * Domain Configuration
 *
 * Manages all ProRank.io domain URLs and endpoints
 *
 * @package ProRank\SEO\Core\Config
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Config;

defined( 'ABSPATH' ) || exit;

/**
 * Domain Configuration Class
 */
class DomainConfig {
    /**
     * Initialize domain constants if not already defined
     */
    public static function init(): void {
        // Base URLs
        if (!defined('PRORANK_API_BASE')) {
            define('PRORANK_API_BASE', 'https://prorank.io/api');
        }

        if (!defined('PRORANK_WEB_URL')) {
            define('PRORANK_WEB_URL', 'https://prorank.io');
        }

        if (!defined('PRORANK_DOCS_URL')) {
            define('PRORANK_DOCS_URL', 'https://prorank.io/docs');
        }

        if (!defined('PRORANK_SUPPORT_URL')) {
            define('PRORANK_SUPPORT_URL', 'https://prorank.io/support');
        }

        if (!defined('PRORANK_ACCOUNT_URL')) {
            define('PRORANK_ACCOUNT_URL', 'https://prorank.io/account');
        }

        if (!defined('PRORANK_CDN_URL')) {
            define('PRORANK_CDN_URL', 'https://cdn.prorank.io');
        }

        if (!defined('PRORANK_NOTIFICATIONS_ENDPOINT')) {
            define('PRORANK_NOTIFICATIONS_ENDPOINT', PRORANK_API_BASE . '/v1/notifications');
        }

        if (!defined('PRORANK_IMAGE_OPTIMIZE_ENDPOINT')) {
            define('PRORANK_IMAGE_OPTIMIZE_ENDPOINT', PRORANK_API_BASE . '/v1/optimize');
        }

        // License API Endpoints
        if (!defined('PRORANK_LICENSE_VALIDATE_URL')) {
            define('PRORANK_LICENSE_VALIDATE_URL', PRORANK_API_BASE . '/wp/validate');
        }

        if (!defined('PRORANK_LICENSE_ACTIVATE_URL')) {
            define('PRORANK_LICENSE_ACTIVATE_URL', PRORANK_API_BASE . '/wp/activate');
        }

        if (!defined('PRORANK_LICENSE_DEACTIVATE_URL')) {
            define('PRORANK_LICENSE_DEACTIVATE_URL', PRORANK_API_BASE . '/wp/deactivate');
        }

        if (!defined('PRORANK_LICENSE_CHECK_UPDATE_URL')) {
            define('PRORANK_LICENSE_CHECK_UPDATE_URL', PRORANK_API_BASE . '/wp/check-update');
        }

        if (!defined('PRORANK_LICENSE_CHANGELOG_URL')) {
            define('PRORANK_LICENSE_CHANGELOG_URL', PRORANK_API_BASE . '/wp/changelog');
        }

        // Marketing Links
        if (!defined('PRORANK_PRICING_URL')) {
            define('PRORANK_PRICING_URL', PRORANK_WEB_URL . '/pricing');
        }

        if (!defined('PRORANK_FEATURES_URL')) {
            define('PRORANK_FEATURES_URL', PRORANK_WEB_URL . '/features');
        }

        if (!defined('PRORANK_BLOG_URL')) {
            define('PRORANK_BLOG_URL', PRORANK_WEB_URL . '/blog');
        }

        if (!defined('PRORANK_CHANGELOG_URL')) {
            define('PRORANK_CHANGELOG_URL', PRORANK_WEB_URL . '/changelog');
        }

        // Legal Pages
        if (!defined('PRORANK_PRIVACY_URL')) {
            define('PRORANK_PRIVACY_URL', PRORANK_WEB_URL . '/privacy');
        }

        if (!defined('PRORANK_TERMS_URL')) {
            define('PRORANK_TERMS_URL', PRORANK_WEB_URL . '/terms');
        }

        if (!defined('PRORANK_CONTACT_URL')) {
            define('PRORANK_CONTACT_URL', PRORANK_WEB_URL . '/contact');
        }

        // User Account Links
        if (!defined('PRORANK_ACCOUNT_LICENSES_URL')) {
            define('PRORANK_ACCOUNT_LICENSES_URL', PRORANK_ACCOUNT_URL . '/licenses');
        }

        if (!defined('PRORANK_ACCOUNT_BILLING_URL')) {
            define('PRORANK_ACCOUNT_BILLING_URL', PRORANK_ACCOUNT_URL . '/billing');
        }

        if (!defined('PRORANK_ACCOUNT_SUPPORT_URL')) {
            define('PRORANK_ACCOUNT_SUPPORT_URL', PRORANK_ACCOUNT_URL . '/support');
        }
        
        // Load Google API configuration if exists
        $google_config = PRORANK_PLUGIN_DIR . 'prorank-config-google.php';
        if (file_exists($google_config)) {
            require_once $google_config;
        }
    }

    /**
     * Get API endpoint URL
     *
     * @param string $endpoint Endpoint path
     * @return string Full URL
     */
    public static function getApiUrl(string $endpoint): string {
        return PRORANK_API_BASE . '/' . ltrim($endpoint, '/');
    }

    /**
     * Get web URL
     *
     * @param string $path Path
     * @return string Full URL
     */
    public static function getWebUrl(string $path = ''): string {
        return PRORANK_WEB_URL . ($path ? '/' . ltrim($path, '/') : '');
    }

    /**
     * Get all URLs for JavaScript
     *
     * @return array
     */
    public static function getAllUrls(): array {
        return [
            'api_base' => PRORANK_API_BASE,
            'web_url' => PRORANK_WEB_URL,
            'docs_url' => PRORANK_DOCS_URL,
            'support_url' => PRORANK_SUPPORT_URL,
            'account_url' => PRORANK_ACCOUNT_URL,
            'cdn_url' => PRORANK_CDN_URL,
            'pricing_url' => PRORANK_PRICING_URL,
            'features_url' => PRORANK_FEATURES_URL,
            'changelog_url' => PRORANK_CHANGELOG_URL,
            'account_licenses_url' => PRORANK_ACCOUNT_LICENSES_URL,
            'upgrade_url' => PRORANK_PRICING_URL,
        ];
    }
}
