<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude

declare(strict_types=1);

/**
 * AI Transparency Provider
 *
 * Handles AI content disclosure, templates, and transparency features.
 *
 * @package ProRank\SEO\Core\DataProviders
 */

namespace ProRank\SEO\Core\DataProviders;

defined( 'ABSPATH' ) || exit;

/**
 * Provides AI content transparency and disclosure functionality
 */
class AITransparencyProvider {
    /**
     * Meta key for AI disclosure data
     */
    const META_KEY = '_prorank_ai_disclosure';

    /**
     * AI usage types
     */
    const USAGE_TYPES = [
        'none'       => 'No AI Used',
        'full'       => 'Fully AI Generated',
        'partial'    => 'Partially AI Generated',
        'assisted'   => 'AI Assisted (Research/Editing)',
        'translated' => 'AI Translated',
        'summarized' => 'AI Summarized',
    ];

    /**
     * Known AI tools
     */
    const AI_TOOLS = [
        'chatgpt'       => 'ChatGPT (OpenAI)',
        'claude'        => 'Claude (Anthropic)',
        'gemini'        => 'Gemini (Google)',
        'copilot'       => 'Microsoft Copilot',
        'jasper'        => 'Jasper AI',
        'writesonic'    => 'Writesonic',
        'copy_ai'       => 'Copy.ai',
        'grammarly'     => 'Grammarly AI',
        'quillbot'      => 'QuillBot',
        'notion_ai'     => 'Notion AI',
        'midjourney'    => 'Midjourney (Images)',
        'dalle'         => 'DALL-E (Images)',
        'stable_diff'   => 'Stable Diffusion (Images)',
        'other'         => 'Other AI Tool',
    ];

    /**
     * Disclosure templates
     */
    const TEMPLATES = [
        'full_ai' => [
            'id'       => 'full_ai',
            'name'     => 'Fully AI Generated',
            'template' => 'This content was generated using {tools}. It has been reviewed by {author} for accuracy and quality.',
            'short'    => 'AI-generated content, human reviewed.',
        ],
        'partial_ai' => [
            'id'       => 'partial_ai',
            'name'     => 'Partially AI Generated',
            'template' => 'Portions of this content were generated with AI assistance using {tools}. The content has been edited and verified by {author}.',
            'short'    => 'AI-assisted content, human edited.',
        ],
        'ai_assisted' => [
            'id'       => 'ai_assisted',
            'name'     => 'AI Assisted',
            'template' => 'AI tools were used to assist in {assistance_type} this content. Final content written and verified by {author}.',
            'short'    => 'Human-written with AI assistance.',
        ],
        'research_only' => [
            'id'       => 'research_only',
            'name'     => 'AI Research Only',
            'template' => 'AI was used for initial research and information gathering. All writing is original content by {author}.',
            'short'    => 'Human-written, AI-researched.',
        ],
        'translated' => [
            'id'       => 'translated',
            'name'     => 'AI Translated',
            'template' => 'This content was translated from {source_language} using {tools}. Translation reviewed by {author}.',
            'short'    => 'AI-translated content.',
        ],
        'images_only' => [
            'id'       => 'images_only',
            'name'     => 'AI Images Only',
            'template' => 'Images in this content were generated using {tools}. All written content is by {author}.',
            'short'    => 'Human-written, AI-generated images.',
        ],
        'custom' => [
            'id'       => 'custom',
            'name'     => 'Custom Disclosure',
            'template' => '{custom_text}',
            'short'    => '{custom_short}',
        ],
    ];

    /**
     * Platform-specific label formats
     */
    const PLATFORM_LABELS = [
        'youtube' => [
            'name'   => 'YouTube',
            'format' => 'Description Label',
            'prefix' => 'AI Disclosure: ',
        ],
        'meta' => [
            'name'   => 'Meta (Facebook/Instagram)',
            'format' => 'Post Label',
            'prefix' => 'Made with AI | ',
        ],
        'tiktok' => [
            'name'   => 'TikTok',
            'format' => 'AI Label',
            'prefix' => '#AIGenerated ',
        ],
        'linkedin' => [
            'name'   => 'LinkedIn',
            'format' => 'Post Footer',
            'prefix' => 'Created with AI assistance: ',
        ],
        'google_ads' => [
            'name'   => 'Google Ads',
            'format' => 'Ad Disclaimer',
            'prefix' => 'AI-assisted content. ',
        ],
    ];

    /**
     * Get AI disclosure for a post
     *
     * @param int $post_id Post ID
     * @return array
     */
    public function get_disclosure( int $post_id ) {
        $data = get_post_meta( $post_id, self::META_KEY, true );

        if ( empty( $data ) || ! is_array( $data ) ) {
            return $this->get_default_disclosure();
        }

        return wp_parse_args( $data, $this->get_default_disclosure() );
    }

    /**
     * Get default disclosure structure
     *
     * @return array
     */
    public function get_default_disclosure() {
        return [
            'uses_ai'         => false,
            'ai_type'         => 'none',
            'ai_tools'        => [],
            'human_review'    => true,
            'disclosure_text' => '',
            'template_id'     => '',
            'custom_text'     => '',
            'show_on_post'    => false,
            'position'        => 'after_content',
            'last_updated'    => '',
        ];
    }

    /**
     * Save AI disclosure for a post
     *
     * @param int   $post_id Post ID
     * @param array $data    Disclosure data
     * @return bool
     */
    public function save_disclosure( int $post_id, array $data ) {
        $sanitized = [
            'uses_ai'         => ! empty( $data['uses_ai'] ),
            'ai_type'         => sanitize_key( $data['ai_type'] ?? 'none' ),
            'ai_tools'        => array_map( 'sanitize_key', (array) ( $data['ai_tools'] ?? [] ) ),
            'human_review'    => ! empty( $data['human_review'] ),
            'disclosure_text' => wp_kses_post( $data['disclosure_text'] ?? '' ),
            'template_id'     => sanitize_key( $data['template_id'] ?? '' ),
            'custom_text'     => wp_kses_post( $data['custom_text'] ?? '' ),
            'show_on_post'    => ! empty( $data['show_on_post'] ),
            'position'        => sanitize_key( $data['position'] ?? 'after_content' ),
            'last_updated'    => current_time( 'mysql' ),
        ];

        return (bool) update_post_meta( $post_id, self::META_KEY, $sanitized );
    }

    /**
     * Generate disclosure text from template
     *
     * @param int   $post_id     Post ID
     * @param array $disclosure  Disclosure data
     * @param array $replacements Additional replacements
     * @return string
     */
    public function generate_disclosure_text( int $post_id, array $disclosure, array $replacements = [] ) {
        $template_id = $disclosure['template_id'] ?? 'ai_assisted';
        $template = self::TEMPLATES[ $template_id ] ?? self::TEMPLATES['ai_assisted'];

        $post = get_post( $post_id );
        $author = get_the_author_meta( 'display_name', $post->post_author ?? 0 );

        // Build tools list
        $tools = [];
        foreach ( $disclosure['ai_tools'] ?? [] as $tool_key ) {
            if ( isset( self::AI_TOOLS[ $tool_key ] ) ) {
                $tools[] = self::AI_TOOLS[ $tool_key ];
            }
        }
        $tools_text = empty( $tools ) ? 'AI tools' : implode( ', ', $tools );

        // Default replacements
        $defaults = [
            '{author}'          => $author ?: 'our team',
            '{tools}'           => $tools_text,
            '{assistance_type}' => $this->get_assistance_type_text( $disclosure['ai_type'] ?? 'assisted' ),
            '{source_language}' => $replacements['{source_language}'] ?? 'the original language',
            '{custom_text}'     => $disclosure['custom_text'] ?? '',
            '{custom_short}'    => $disclosure['custom_text'] ?? '',
            '{site_name}'       => get_bloginfo( 'name' ),
            '{date}'            => get_the_date( '', $post_id ),
        ];

        $all_replacements = array_merge( $defaults, $replacements );

        return str_replace(
            array_keys( $all_replacements ),
            array_values( $all_replacements ),
            $template['template']
        );
    }

    /**
     * Get assistance type text
     *
     * @param string $type AI usage type
     * @return string
     */
    private function get_assistance_type_text( string $type ) {
        return match ( $type ) {
            'full'       => 'generating',
            'partial'    => 'partially creating',
            'assisted'   => 'researching and drafting',
            'translated' => 'translating',
            'summarized' => 'summarizing',
            default      => 'creating',
        };
    }

    /**
     * Generate schema.org markup for AI disclosure
     *
     * @param int   $post_id    Post ID
     * @param array $disclosure Disclosure data
     * @return array Schema data to merge with existing
     */
    public function generate_schema( int $post_id, array $disclosure ) {
        if ( empty( $disclosure['uses_ai'] ) ) {
            return [];
        }

        $schema = [];

        // Add creativeWorkStatus for AI content
        $status = match ( $disclosure['ai_type'] ?? 'none' ) {
            'full'    => 'ai-generated',
            'partial' => 'ai-assisted',
            default   => 'human-created-ai-assisted',
        };

        $schema['creativeWorkStatus'] = $status;

        // Add AI disclosure as abstract/description
        if ( ! empty( $disclosure['disclosure_text'] ) ) {
            $schema['abstract'] = 'AI Disclosure: ' . $disclosure['disclosure_text'];
        }

        // Add contributor for AI tools used
        if ( ! empty( $disclosure['ai_tools'] ) ) {
            $schema['contributor'] = [];
            foreach ( $disclosure['ai_tools'] as $tool_key ) {
                if ( isset( self::AI_TOOLS[ $tool_key ] ) ) {
                    $schema['contributor'][] = [
                        '@type' => 'SoftwareApplication',
                        'name'  => self::AI_TOOLS[ $tool_key ],
                        'applicationCategory' => 'AI Content Generation',
                    ];
                }
            }
        }

        return $schema;
    }

    /**
     * Generate platform-specific label
     *
     * @param string $platform    Platform key
     * @param array  $disclosure  Disclosure data
     * @return string
     */
    public function generate_platform_label( string $platform, array $disclosure ) {
        if ( ! isset( self::PLATFORM_LABELS[ $platform ] ) ) {
            return '';
        }

        $config = self::PLATFORM_LABELS[ $platform ];
        $prefix = $config['prefix'];

        // Build tools list
        $tools = [];
        foreach ( $disclosure['ai_tools'] ?? [] as $tool_key ) {
            if ( isset( self::AI_TOOLS[ $tool_key ] ) ) {
                // Use short names for platforms
                $tools[] = explode( ' ', self::AI_TOOLS[ $tool_key ] )[0];
            }
        }
        $tools_text = empty( $tools ) ? 'AI' : implode( ', ', $tools );

        $type_text = match ( $disclosure['ai_type'] ?? 'assisted' ) {
            'full'       => 'Generated with',
            'partial'    => 'Partially created with',
            'assisted'   => 'Created with assistance from',
            'translated' => 'Translated using',
            default      => 'Made with',
        };

        return $prefix . $type_text . ' ' . $tools_text;
    }

    /**
     * Get all available templates
     *
     * @return array
     */
    public function get_templates() {
        return self::TEMPLATES;
    }

    /**
     * Get all AI tools
     *
     * @return array
     */
    public function get_ai_tools() {
        return self::AI_TOOLS;
    }

    /**
     * Get all usage types
     *
     * @return array
     */
    public function get_usage_types() {
        return self::USAGE_TYPES;
    }

    /**
     * Get all platform labels
     *
     * @return array
     */
    public function get_platform_labels() {
        return self::PLATFORM_LABELS;
    }

    /**
     * Render disclosure HTML for frontend
     *
     * @param int   $post_id    Post ID
     * @param array $disclosure Disclosure data
     * @return string HTML
     */
    public function render_disclosure_html( int $post_id, array $disclosure ) {
        if ( empty( $disclosure['uses_ai'] ) || empty( $disclosure['show_on_post'] ) ) {
            return '';
        }

        $text = ! empty( $disclosure['disclosure_text'] )
            ? $disclosure['disclosure_text']
            : $this->generate_disclosure_text( $post_id, $disclosure );

        $classes = [
            'prorank-ai-disclosure',
            'prorank-ai-disclosure--' . esc_attr( $disclosure['ai_type'] ?? 'assisted' ),
        ];

        if ( ! empty( $disclosure['human_review'] ) ) {
            $classes[] = 'prorank-ai-disclosure--reviewed';
        }

        ob_start();
        ?>
        <div class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>">
            <span class="prorank-ai-disclosure__icon" aria-hidden="true">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-1 17.93c-3.95-.49-7-3.85-7-7.93 0-.62.08-1.21.21-1.79L9 15v1c0 1.1.9 2 2 2v1.93zm6.9-2.54c-.26-.81-1-1.39-1.9-1.39h-1v-3c0-.55-.45-1-1-1H8v-2h2c.55 0 1-.45 1-1V7h2c1.1 0 2-.9 2-2v-.41c2.93 1.19 5 4.06 5 7.41 0 2.08-.8 3.97-2.1 5.39z"/>
                </svg>
            </span>
            <span class="prorank-ai-disclosure__text">
                <?php echo wp_kses_post( $text ); ?>
            </span>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Get posts with AI disclosure
     *
     * @param array $args Query args
     * @return array
     */
    public function get_posts_with_ai( array $args = [] ) {
        $defaults = [
            'post_type'      => 'any',
            'posts_per_page' => 50,
            'meta_query'     => [
                [
                    'key'     => self::META_KEY,
                    'compare' => 'EXISTS',
                ],
            ],
        ];

        $query = new \WP_Query( array_merge( $defaults, $args ) );
        $results = [];

        foreach ( $query->posts as $post ) {
            $disclosure = $this->get_disclosure( $post->ID );
            if ( ! empty( $disclosure['uses_ai'] ) ) {
                $results[] = [
                    'post_id'    => $post->ID,
                    'title'      => $post->post_title,
                    'post_type'  => $post->post_type,
                    'disclosure' => $disclosure,
                ];
            }
        }

        return $results;
    }

    /**
     * Get AI usage statistics
     *
     * @return array
     */
    public function get_statistics() {
        global $wpdb;

        $meta_key = self::META_KEY;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $total_with_meta = $wpdb->get_var(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $meta_key
            )
        );

        $stats = [
            'total_posts_with_disclosure' => (int) $total_with_meta,
            'by_type'                     => [],
            'by_tool'                     => [],
        ];

        // Get all disclosures
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $disclosures = $wpdb->get_col(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s",
                $meta_key
            )
        );

        foreach ( $disclosures as $disclosure_raw ) {
            $disclosure = maybe_unserialize( $disclosure_raw );
            if ( ! is_array( $disclosure ) || empty( $disclosure['uses_ai'] ) ) {
                continue;
            }

            $type = $disclosure['ai_type'] ?? 'none';
            $stats['by_type'][ $type ] = ( $stats['by_type'][ $type ] ?? 0 ) + 1;

            foreach ( $disclosure['ai_tools'] ?? [] as $tool ) {
                $stats['by_tool'][ $tool ] = ( $stats['by_tool'][ $tool ] ?? 0 ) + 1;
            }
        }

        return $stats;
    }
}
