<?php
/**
 * Advanced Image Optimization Engine
 *
 * Next-generation image optimization with multiple backends and advanced features
 *
 * @package ProRank\SEO\Core\ImageOptimization
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\ImageOptimization;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;

/**
 * OptimizationEngine class
 */
class OptimizationEngine {
	
	/**
	 * Available optimization tools
	 *
	 * @var array
	 */
	private array $available_tools = [];
	
	/**
	 * Plugin instance
	 *
	 * @var Plugin
	 */
	private Plugin $plugin;
	
	/**
	 * Constructor
	 */
	public function __construct() {
		$this->plugin = Plugin::get_instance();
		$this->detect_available_tools();
	}
	
	/**
	 * Detect available optimization tools on the system
	 */
	private function detect_available_tools(): void {
		// PHP Extensions
		if ( extension_loaded( 'gd' ) ) {
			$this->available_tools['gd'] = [
				'name' => 'GD Library',
				'priority' => 10,
				'formats' => [ 'jpeg', 'jpg', 'png', 'gif', 'webp' ],
			];
		}
		
		if ( extension_loaded( 'imagick' ) ) {
			$imagick = new \Imagick();
			$formats = $imagick->queryFormats();
			$this->available_tools['imagick'] = [
				'name' => 'ImageMagick',
				'priority' => 50,
				'formats' => array_map( 'strtolower', $formats ),
			];
		}
		
		// Binary tools detection
		$binaries = [
			'cwebp' => [
				'name' => 'WebP Converter',
				'priority' => 80,
				'formats' => [ 'jpeg', 'jpg', 'png' ],
				'output' => 'webp',
			],
			'avifenc' => [
				'name' => 'AVIF Encoder',
				'priority' => 85,
				'formats' => [ 'jpeg', 'jpg', 'png' ],
				'output' => 'avif',
			],
			'cjxl' => [
				'name' => 'JPEG XL Encoder (libjxl)',
				'priority' => 95,
				'formats' => [ 'jpeg', 'jpg', 'png', 'gif' ],
				'output' => 'jxl',
			],
			'cjpegli' => [
				'name' => 'Jpegli Encoder (libjxl)',
				'priority' => 92,
				'formats' => [ 'jpeg', 'jpg', 'png' ],
				'output' => 'jpg',
			],
			'jpegoptim' => [
				'name' => 'JPEGOptim',
				'priority' => 90,
				'formats' => [ 'jpeg', 'jpg' ],
			],
			'optipng' => [
				'name' => 'OptiPNG',
				'priority' => 90,
				'formats' => [ 'png' ],
			],
			'pngquant' => [
				'name' => 'PNGQuant',
				'priority' => 95,
				'formats' => [ 'png' ],
			],
			'oxipng' => [
				'name' => 'Oxipng',
				'priority' => 96,
				'formats' => [ 'png' ],
			],
			'gifsicle' => [
				'name' => 'Gifsicle',
				'priority' => 85,
				'formats' => [ 'gif' ],
			],
		];
		
		foreach ( $binaries as $binary => $info ) {
			if ( $this->is_binary_available( $binary ) ) {
				$this->available_tools[ $binary ] = $info;
			}
		}
	}
	
	/**
	 * Check if a binary is available
	 *
	 * @param string $binary Binary name.
	 * @return bool
	 */
	private function is_binary_available( string $binary ): bool {
		if ( ! function_exists( 'exec' ) ) {
			return false;
		}
		
		$output = [];
		$return_var = 0;
		exec( "which $binary 2>/dev/null", $output, $return_var );
		
		return $return_var === 0;
	}
	
	/**
	 * Optimize image using the best available method
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array
	 */
	public function optimize_image( int $attachment_id ): array {
		$file_path = get_attached_file( $attachment_id );
		if ( ! $file_path || ! file_exists( $file_path ) ) {
			return [
				'success' => false,
				'error' => __( 'File not found', 'prorank-seo' ),
			];
		}
		
		// Get settings
		$settings = $this->plugin->settings();
		$compression_type = $settings->get( 'images_compression_type', 'glossy' );
		$smart_compression = $settings->get( 'images_smart_compression', true );
		$backup_originals = $settings->get( 'images_backup_originals', true );
		
		// Backup original if enabled
		if ( $backup_originals ) {
			$this->backup_original( $file_path );
		}
		
		// Get file info
		$file_info = pathinfo( $file_path );
		$mime_type = wp_check_filetype( $file_path )['type'];
		$original_size = filesize( $file_path );
		
		// Choose optimization method based on settings
		// Read from image optimization settings option
		$image_settings = get_option( 'prorank_image_optimization_settings', [] );
		$optimization_method = $image_settings['optimization_method'] ?? 'local';
		
		// Use cloud optimization if enabled
		if ( $optimization_method === 'prorank' || $optimization_method === 'cloud' ) {
			$format = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
			if ( $format === 'jpg' ) {
				$format = 'jpeg';
			}
			$quality = $compression_type === 'lossless' ? 100 : ( $compression_type === 'glossy' ? 90 : 85 );
			$result = $this->convert_via_cloud( $file_path, $format, $quality );
		} else {
			// Use local tools
			$result = $this->optimize_with_best_tool( $file_path, $mime_type, [
				'compression_type' => $compression_type,
				'smart_compression' => $smart_compression,
			] );
		}
		
		if ( $result['success'] ) {
			// Update metadata
			clearstatcache( true, $file_path );
			$new_size = filesize( $file_path );
			$saved_bytes = $original_size - $new_size;
			$saved_percent = ( $saved_bytes / $original_size ) * 100;
			
			// Store optimization data
			update_post_meta( $attachment_id, '_prorank_optimized', true );
			update_post_meta( $attachment_id, '_prorank_optimization_savings', $saved_bytes );
			update_post_meta( $attachment_id, '_prorank_optimization_percent', round( $saved_percent, 2 ) );
			update_post_meta( $attachment_id, '_prorank_original_size', $original_size );
			update_post_meta( $attachment_id, '_prorank_optimized_size', $new_size );
			update_post_meta( $attachment_id, '_prorank_optimization_date', current_time( 'mysql' ) );
			update_post_meta( $attachment_id, '_prorank_optimization_method', $result['method'] );
			
			// Handle WebP/AVIF conversion if enabled
			$this->handle_modern_formats( $attachment_id, $file_path );
			
			return [
				'success' => true,
				'original_size' => size_format( $original_size ),
				'new_size' => size_format( $new_size ),
				'saved_size' => size_format( $saved_bytes ),
				'saved_percent' => round( $saved_percent, 2 ),
				'method' => $result['method'],
			];
		}
		
		return $result;
	}
	
	/**
	 * Optimize with the best available tool
	 *
	 * @param string $file_path File path.
	 * @param string $mime_type MIME type.
	 * @param array  $options   Options.
	 * @return array
	 */
	private function optimize_with_best_tool( string $file_path, string $mime_type, array $options ): array {
		$extension = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
		
		// Get tools that support this format, sorted by priority
		$compatible_tools = [];
		foreach ( $this->available_tools as $tool => $info ) {
			if ( in_array( $extension, $info['formats'], true ) ) {
				$compatible_tools[ $tool ] = $info['priority'];
			}
		}
		arsort( $compatible_tools );
		
		// Try each tool in order of priority
		foreach ( array_keys( $compatible_tools ) as $tool ) {
			$result = $this->optimize_with_tool( $tool, $file_path, $mime_type, $options );
			if ( $result['success'] ) {
				return $result;
			}
		}
		
		return [
			'success' => false,
			'error' => __( 'No optimization tool available', 'prorank-seo' ),
		];
	}
	
	/**
	 * Optimize with specific tool
	 *
	 * @param string $tool      Tool name.
	 * @param string $file_path File path.
	 * @param string $mime_type MIME type.
	 * @param array  $options   Options.
	 * @return array
	 */
	private function optimize_with_tool( string $tool, string $file_path, string $mime_type, array $options ): array {
		switch ( $tool ) {
			case 'pngquant':
				return $this->optimize_with_pngquant( $file_path, $options );
				
			case 'jpegoptim':
				return $this->optimize_with_jpegoptim( $file_path, $options );
				
			case 'optipng':
				return $this->optimize_with_optipng( $file_path, $options );
				
			case 'imagick':
				return $this->optimize_with_imagick( $file_path, $mime_type, $options );
				
			case 'gd':
				return $this->optimize_with_gd( $file_path, $mime_type, $options );
				
			default:
				return [
					'success' => false,
					'error' => __( 'Unknown optimization tool', 'prorank-seo' ),
				];
		}
	}
	
	/**
	 * Optimize PNG with pngquant
	 *
	 * @param string $file_path File path.
	 * @param array  $options   Options.
	 * @return array
	 */
	private function optimize_with_pngquant( string $file_path, array $options ): array {
		$quality = $this->get_quality_for_compression_type( $options['compression_type'], 'png' );
		
		$command = sprintf(
			'pngquant --quality=%d-%d --speed=1 --force --output="%s" "%s" 2>&1',
			max( 0, $quality - 10 ),
			$quality,
			$file_path,
			$file_path
		);
		
		exec( $command, $output, $return_var );
		
		return [
			'success' => $return_var === 0,
			'method' => 'pngquant',
			'error' => $return_var !== 0 ? implode( "\n", $output ) : null,
		];
	}
	
	/**
	 * Optimize JPEG with jpegoptim
	 *
	 * @param string $file_path File path.
	 * @param array  $options   Options.
	 * @return array
	 */
	private function optimize_with_jpegoptim( string $file_path, array $options ): array {
		$quality = $this->get_quality_for_compression_type( $options['compression_type'], 'jpeg' );
		
		$args = [ '--strip-all' ]; // Remove metadata
		
		if ( $options['compression_type'] !== 'lossless' ) {
			$args[] = "--max=$quality";
		}
		
		$command = sprintf(
			'jpegoptim %s "%s" 2>&1',
			implode( ' ', $args ),
			$file_path
		);
		
		exec( $command, $output, $return_var );
		
		return [
			'success' => $return_var === 0,
			'method' => 'jpegoptim',
			'error' => $return_var !== 0 ? implode( "\n", $output ) : null,
		];
	}
	
	/**
	 * Optimize PNG with optipng
	 *
	 * @param string $file_path File path.
	 * @param array  $options   Options.
	 * @return array
	 */
	private function optimize_with_optipng( string $file_path, array $options ): array {
		$level = $options['compression_type'] === 'lossless' ? 2 : 5;
		
		$command = sprintf(
			'optipng -o%d -strip all "%s" 2>&1',
			$level,
			$file_path
		);
		
		exec( $command, $output, $return_var );
		
		return [
			'success' => $return_var === 0,
			'method' => 'optipng',
			'error' => $return_var !== 0 ? implode( "\n", $output ) : null,
		];
	}
	
	/**
	 * Optimize with ImageMagick
	 *
	 * @param string $file_path File path.
	 * @param string $mime_type MIME type.
	 * @param array  $options   Options.
	 * @return array
	 */
	private function optimize_with_imagick( string $file_path, string $mime_type, array $options ): array {
		try {
			$imagick = new \Imagick( $file_path );
			
			// Strip metadata
			$imagick->stripImage();
			
			// Set compression based on type
			if ( strpos( $mime_type, 'jpeg' ) !== false ) {
				$quality = $this->get_quality_for_compression_type( $options['compression_type'], 'jpeg' );
				$imagick->setImageCompression( \Imagick::COMPRESSION_JPEG );
				$imagick->setImageCompressionQuality( $quality );
				
				// Progressive JPEG
				$imagick->setInterlaceScheme( \Imagick::INTERLACE_PLANE );
			} elseif ( strpos( $mime_type, 'png' ) !== false ) {
				$imagick->setImageCompression( \Imagick::COMPRESSION_ZIP );
				$imagick->setOption( 'png:compression-level', '9' );
			}
			
			// Smart compression adjustments
			if ( $options['smart_compression'] ) {
				$this->apply_smart_compression( $imagick );
			}
			
			$imagick->writeImage( $file_path );
			$imagick->destroy();
			
			return [
				'success' => true,
				'method' => 'imagick',
			];
		} catch ( \Exception $e ) {
			return [
				'success' => false,
				'method' => 'imagick',
				'error' => esc_html($e->getMessage()),
			];
		}
	}
	
	/**
	 * Optimize with GD
	 *
	 * @param string $file_path File path.
	 * @param string $mime_type MIME type.
	 * @param array  $options   Options.
	 * @return array
	 */
	private function optimize_with_gd( string $file_path, string $mime_type, array $options ): array {
		$image = null;
		
		// Load image
		switch ( $mime_type ) {
			case 'image/jpeg':
				$image = imagecreatefromjpeg( $file_path );
				break;
			case 'image/png':
				$image = imagecreatefrompng( $file_path );
				break;
			case 'image/gif':
				$image = imagecreatefromgif( $file_path );
				break;
			case 'image/webp':
				if ( function_exists( 'imagecreatefromwebp' ) ) {
					$image = imagecreatefromwebp( $file_path );
				}
				break;
		}
		
		if ( ! $image ) {
			return [
				'success' => false,
				'method' => 'gd',
				'error' => __( 'Failed to load image', 'prorank-seo' ),
			];
		}
		
		// Save optimized image
		$success = false;
		switch ( $mime_type ) {
			case 'image/jpeg':
				$quality = $this->get_quality_for_compression_type( $options['compression_type'], 'jpeg' );
				imageinterlace( $image, true ); // Progressive JPEG
				$success = imagejpeg( $image, $file_path, $quality );
				break;
			case 'image/png':
				imagesavealpha( $image, true );
				$success = imagepng( $image, $file_path, 9 );
				break;
			case 'image/gif':
				$success = imagegif( $image, $file_path );
				break;
			case 'image/webp':
				if ( function_exists( 'imagewebp' ) ) {
					$quality = $this->get_quality_for_compression_type( $options['compression_type'], 'webp' );
					$success = imagewebp( $image, $file_path, $quality );
				}
				break;
		}
		
		imagedestroy( $image );
		
		return [
			'success' => $success,
			'method' => 'gd',
			'error' => ! $success ? __( 'Failed to save image', 'prorank-seo' ) : null,
		];
	}
	
	/**
	 * Get quality setting for compression type
	 *
	 * @param string $compression_type Compression type.
	 * @param string $format           Image format.
	 * @return int
	 */
	private function get_quality_for_compression_type( string $compression_type, string $format ): int {
		$settings = $this->plugin->settings();
		
		// Get base quality from settings
		$base_quality = (int) $settings->get( 'images_compression_level', 85 );
		
		// Adjust based on compression type
		switch ( $compression_type ) {
			case 'lossy':
				return max( 60, $base_quality - 15 );
			case 'glossy':
				return $base_quality;
			case 'lossless':
				return 100;
			default:
				return $base_quality;
		}
	}
	
	/**
	 * Apply smart compression adjustments
	 *
	 * @param \Imagick $imagick Imagick instance.
	 */
	private function apply_smart_compression( \Imagick $imagick ): void {
		// Analyze image characteristics
		$width = $imagick->getImageWidth();
		$height = $imagick->getImageHeight();
		$colors = $imagick->getImageColors();
		
		// High resolution images can use more compression
		if ( $width > 2000 || $height > 2000 ) {
			$current_quality = $imagick->getImageCompressionQuality();
			$imagick->setImageCompressionQuality( max( 70, $current_quality - 10 ) );
		}
		
		// Images with few colors (logos, graphics) need less compression
		if ( $colors < 256 ) {
			$current_quality = $imagick->getImageCompressionQuality();
			$imagick->setImageCompressionQuality( min( 95, $current_quality + 10 ) );
		}
	}
	
	/**
	 * Handle modern format conversion
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $file_path     Original file path.
	 */
	private function handle_modern_formats( int $attachment_id, string $file_path ): void {
		$settings = $this->plugin->settings();

		if ( $settings->get( 'images_convert_to_webp', false ) ) {
			$this->convert_to_webp( $attachment_id, $file_path );
		}

		if ( $settings->get( 'images_convert_to_avif', false ) ) {
			$this->convert_to_avif( $attachment_id, $file_path );
		}

		if ( $settings->get( 'images_convert_to_jxl', false ) ) {
			$this->convert_to_jxl( $attachment_id, $file_path );
		}
	}

	/**
	 * Convert image to JPEG XL
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $source_path   Source file path.
	 * @return array
	 */
	public function convert_to_jxl( int $attachment_id, string $source_path ): array {
		$jxl_path = preg_replace( '/\.(jpe?g|png|gif)$/i', '.jxl', $source_path );
		$settings = $this->plugin->settings();
		$quality = (int) $settings->get( 'images_jxl_quality', 85 );

		// Try cjxl from libjxl (best quality)
		if ( isset( $this->available_tools['cjxl'] ) ) {
			// JXL uses distance: 0 = lossless, 1 = visually lossless, higher = more compression
			// Convert quality (0-100) to distance (0-15)
			$distance = ( 100 - $quality ) / 100 * 3; // 0-3 range for high quality

			$command = sprintf(
				'cjxl -d %.1f --effort=7 "%s" "%s" 2>&1',
				$distance,
				$source_path,
				$jxl_path
			);

			exec( $command, $output, $return_var );

			if ( $return_var === 0 ) {
				$this->register_jxl_file( $attachment_id, $jxl_path );
				return [
					'success' => true,
					'jxl_path' => $jxl_path,
					'method' => 'cjxl',
				];
			}
		}

		// Try cloud API if available
		// Read from image optimization settings option
		$image_settings = get_option( 'prorank_image_optimization_settings', [] );
		$optimization_method = $image_settings['optimization_method'] ?? 'local';
		if ( $optimization_method === 'cloud' || $optimization_method === 'prorank' || $optimization_method === 'smart' ) {
			$cloud_result = $this->convert_via_cloud( $source_path, 'jxl', $quality );
			if ( $cloud_result['success'] ) {
				$this->register_jxl_file( $attachment_id, $cloud_result['output_path'] );
				return [
					'success' => true,
					'jxl_path' => $cloud_result['output_path'],
					'method' => 'cloud',
				];
			}
		}

		return [
			'success' => false,
			'error' => __( 'JPEG XL conversion not available. Enable ProRank Cloud optimization for JXL support.', 'prorank-seo' ),
		];
	}

	/**
	 * Register JXL file
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $jxl_path      JXL file path.
	 */
	private function register_jxl_file( int $attachment_id, string $jxl_path ): void {
		update_post_meta( $attachment_id, '_prorank_jxl_path', $jxl_path );
		update_post_meta( $attachment_id, '_prorank_jxl_size', filesize( $jxl_path ) );
	}

	/**
	 * Convert image via ProRank Cloud API
	 *
	 * @param string $source_path Source file path.
	 * @param string $format      Target format (webp, avif, jxl, jpeg).
	 * @param int    $quality     Quality level.
	 * @return array
	 */
	public function convert_via_cloud( string $source_path, string $format, int $quality = 85 ): array {
		$settings = $this->plugin->settings();
		$api_url = defined( 'PRORANK_IMAGE_OPTIMIZE_ENDPOINT' )
			? PRORANK_IMAGE_OPTIMIZE_ENDPOINT
			: 'https://prorank.io/api/v1/optimize';

		if ( $api_url === '#disabled-for-wporg#' || $api_url === '' ) {
			return [
				'success' => false,
				'error' => __( 'Cloud image optimization is not available in this build.', 'prorank-seo' ),
			];
		}

		// Read the file
		$file_data = file_get_contents( $source_path );
		if ( ! $file_data ) {
			return [
				'success' => false,
				'error' => __( 'Failed to read source file', 'prorank-seo' ),
			];
		}

		// Get license key
		$license_key = $settings->get( 'general_license_key', '' );
		if ( empty( $license_key ) ) {
			return [
				'success' => false,
				'error' => __( 'No license key configured', 'prorank-seo' ),
			];
		}

		// Check cloud credit quota before making the API request
		try {
			$module = $this->plugin->modules()->get_module( 'image_optimization' );
			if ( $module && method_exists( $module, 'get_quota_status' ) ) {
				$quota_status = $module->get_quota_status();
				if ( ( $quota_status['status'] ?? '' ) !== 'active' ) {
					return [
						'success' => false,
						'error' => $quota_status['message'] ?? __( 'Cloud credits are not available', 'prorank-seo' ),
					];
				}
			}
		} catch ( \Throwable $e ) {
			// Ignore quota check failures and fall back to API response
		}

		// Build request
		$boundary = wp_generate_password( 24, false );
		$body = '';

		// Add file
		$body .= "--{$boundary}\r\n";
		$body .= "Content-Disposition: form-data; name=\"image\"; filename=\"" . basename( $source_path ) . "\"\r\n";
		$body .= "Content-Type: " . wp_check_filetype( $source_path )['type'] . "\r\n\r\n";
		$body .= $file_data . "\r\n";

		// Add format
		$body .= "--{$boundary}\r\n";
		$body .= "Content-Disposition: form-data; name=\"format\"\r\n\r\n";
		$body .= $format . "\r\n";

		// Add quality
		$body .= "--{$boundary}\r\n";
		$body .= "Content-Disposition: form-data; name=\"quality\"\r\n\r\n";
		$body .= $quality . "\r\n";

		$body .= "--{$boundary}--\r\n";

		// Make request
		$response = wp_remote_post( $api_url, [
			'timeout' => 60,
			'headers' => [
				'Authorization' => 'Bearer ' . $license_key,
				'X-Site-Url' => home_url(),
				'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
			],
			'body' => $body,
		] );

		if ( is_wp_error( $response ) ) {
			return [
				'success' => false,
				'error' => $response->get_error_message(),
			];
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body = wp_remote_retrieve_body( $response );

		if ( $status_code !== 200 ) {
			$error_message = __( 'Cloud optimization failed', 'prorank-seo' );
			$decoded = json_decode( $body, true );
			if ( is_array( $decoded ) && ! empty( $decoded['message'] ) ) {
				$error_message = $decoded['message'];
			}

			return [
				'success' => false,
				'error' => $error_message,
			];
		}

		$response_data = json_decode( $body, true );
		if ( ! is_array( $response_data ) || empty( $response_data['data']['optimized'] ) ) {
			return [
				'success' => false,
				'error' => __( 'Unexpected response from cloud optimizer', 'prorank-seo' ),
			];
		}

		$optimized_data = base64_decode( $response_data['data']['optimized'] );
		if ( ! $optimized_data ) {
			return [
				'success' => false,
				'error' => __( 'Failed to decode optimized image', 'prorank-seo' ),
			];
		}

		$output_path = $this->get_optimized_path( $source_path, $format );
		if ( ! $output_path ) {
			return [
				'success' => false,
				'error' => __( 'Could not determine output path', 'prorank-seo' ),
			];
		}

		$write_result = file_put_contents( $output_path, $optimized_data );
		if ( ! $write_result ) {
			return [
				'success' => false,
				'error' => __( 'Failed to write optimized image', 'prorank-seo' ),
			];
		}

		return [
			'success' => true,
			'output_path' => $output_path,
			'method' => 'cloud',
		];
	}
	
	/**
	 * Convert image to WebP
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $source_path   Source file path.
	 * @return array
	 */
	public function convert_to_webp( int $attachment_id, string $source_path ): array {
		$webp_path = preg_replace( '/\.(jpe?g|png)$/i', '.webp', $source_path );
		
		// Try cwebp first (best quality)
		if ( isset( $this->available_tools['cwebp'] ) ) {
			$quality = $this->get_quality_for_compression_type( 'glossy', 'webp' );
			$command = sprintf(
				'cwebp -q %d "%s" -o "%s" 2>&1',
				$quality,
				$source_path,
				$webp_path
			);
			
			exec( $command, $output, $return_var );
			
			if ( $return_var === 0 ) {
				$this->register_webp_file( $attachment_id, $webp_path );
				return [
					'success' => true,
					'webp_path' => $webp_path,
					'method' => 'cwebp',
				];
			}
		}
		
		// Fallback to ImageMagick
		if ( isset( $this->available_tools['imagick'] ) ) {
			try {
				$imagick = new \Imagick( $source_path );
				$imagick->setImageFormat( 'webp' );
				$imagick->setImageCompressionQuality( $this->get_quality_for_compression_type( 'glossy', 'webp' ) );
				$imagick->writeImage( $webp_path );
				$imagick->destroy();
				
				$this->register_webp_file( $attachment_id, $webp_path );
				return [
					'success' => true,
					'webp_path' => $webp_path,
					'method' => 'imagick',
				];
			} catch ( \Exception $e ) {
				// Continue to next method
			}
		}
		
		// Fallback to GD
		if ( function_exists( 'imagewebp' ) ) {
			$image = null;
			$mime_type = wp_check_filetype( $source_path )['type'];
			
			switch ( $mime_type ) {
				case 'image/jpeg':
					$image = imagecreatefromjpeg( $source_path );
					break;
				case 'image/png':
					$image = imagecreatefrompng( $source_path );
					imagesavealpha( $image, true );
					break;
			}
			
			if ( $image ) {
				$quality = $this->get_quality_for_compression_type( 'glossy', 'webp' );
				if ( imagewebp( $image, $webp_path, $quality ) ) {
					imagedestroy( $image );
					$this->register_webp_file( $attachment_id, $webp_path );
					return [
						'success' => true,
						'webp_path' => $webp_path,
						'method' => 'gd',
					];
				}
				imagedestroy( $image );
			}
		}
		
		return [
			'success' => false,
			'error' => __( 'WebP conversion failed', 'prorank-seo' ),
		];
	}
	
	/**
	 * Convert image to AVIF
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $source_path   Source file path.
	 * @return array
	 */
	public function convert_to_avif( int $attachment_id, string $source_path ): array {
		$avif_path = preg_replace( '/\.(jpe?g|png)$/i', '.avif', $source_path );
		
		// Try avifenc
		if ( isset( $this->available_tools['avifenc'] ) ) {
			$quality = $this->get_quality_for_compression_type( 'glossy', 'avif' );
			$command = sprintf(
				'avifenc --min 0 --max 63 --minalpha 0 --maxalpha 63 -a end-usage=q -a cq-level=%d "%s" "%s" 2>&1',
				100 - $quality,
				$source_path,
				$avif_path
			);
			
			exec( $command, $output, $return_var );
			
			if ( $return_var === 0 ) {
				$this->register_avif_file( $attachment_id, $avif_path );
				return [
					'success' => true,
					'avif_path' => $avif_path,
					'method' => 'avifenc',
				];
			}
		}
		
		// Try ImageMagick (if it supports AVIF)
		if ( isset( $this->available_tools['imagick'] ) && in_array( 'avif', $this->available_tools['imagick']['formats'], true ) ) {
			try {
				$imagick = new \Imagick( $source_path );
				$imagick->setImageFormat( 'avif' );
				$imagick->setImageCompressionQuality( $this->get_quality_for_compression_type( 'glossy', 'avif' ) );
				$imagick->writeImage( $avif_path );
				$imagick->destroy();
				
				$this->register_avif_file( $attachment_id, $avif_path );
				return [
					'success' => true,
					'avif_path' => $avif_path,
					'method' => 'imagick',
				];
			} catch ( \Exception $e ) {
				// AVIF not supported
			}
		}
		
		return [
			'success' => false,
			'error' => __( 'AVIF conversion not available', 'prorank-seo' ),
		];
	}
	
	/**
	 * Backup original file
	 *
	 * @param string $file_path File path.
	 */
	private function backup_original( string $file_path ): void {
		$backup_dir = wp_upload_dir()['basedir'] . '/prorank-seo-backups';
		if ( ! file_exists( $backup_dir ) ) {
			wp_mkdir_p( $backup_dir );
		}
		
		$backup_path = $backup_dir . '/' . basename( $file_path ) . '.original';
		if ( ! file_exists( $backup_path ) ) {
			copy( $file_path, $backup_path );
		}
	}
	
	/**
	 * Register WebP file
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $webp_path     WebP file path.
	 */
	private function register_webp_file( int $attachment_id, string $webp_path ): void {
		update_post_meta( $attachment_id, '_prorank_webp_path', $webp_path );
		update_post_meta( $attachment_id, '_prorank_webp_size', filesize( $webp_path ) );
	}
	
	/**
	 * Register AVIF file
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $avif_path     AVIF file path.
	 */
	private function register_avif_file( int $attachment_id, string $avif_path ): void {
		update_post_meta( $attachment_id, '_prorank_avif_path', $avif_path );
		update_post_meta( $attachment_id, '_prorank_avif_size', filesize( $avif_path ) );
	}
	
	/**
	 * Get optimization capabilities
	 *
	 * @return array
	 */
	public function get_capabilities(): array {
		$settings = $this->plugin->settings();
		// Read from image optimization settings option
		$image_settings = get_option( 'prorank_image_optimization_settings', [] );
		$optimization_method = $image_settings['optimization_method'] ?? 'local';
		$cloud_available = $optimization_method === 'cloud' || $optimization_method === 'prorank' || $optimization_method === 'smart';

		return [
			'tools' => $this->available_tools,
			'formats' => [
				'webp' => isset( $this->available_tools['cwebp'] ) || isset( $this->available_tools['imagick'] ) || function_exists( 'imagewebp' ) || $cloud_available,
				'avif' => isset( $this->available_tools['avifenc'] ) || ( isset( $this->available_tools['imagick'] ) && in_array( 'avif', $this->available_tools['imagick']['formats'], true ) ) || $cloud_available,
				'jxl' => isset( $this->available_tools['cjxl'] ) || $cloud_available,
				'jpegli' => isset( $this->available_tools['cjpegli'] ) || $cloud_available,
			],
			'features' => [
				'smart_compression' => true,
				'batch_processing' => true,
				'background_processing' => true,
				'auto_webp' => true,
				'auto_avif' => true,
				'auto_jxl' => isset( $this->available_tools['cjxl'] ) || $cloud_available,
				'jpegli_encoding' => isset( $this->available_tools['cjpegli'] ) || $cloud_available,
				'progressive_jpeg' => true,
				'strip_metadata' => true,
				'backup_originals' => true,
				'cloud_optimization' => $cloud_available,
			],
			'cloud' => [
				'available' => $cloud_available,
				'formats' => [ 'webp', 'avif', 'jxl', 'jpeg' ],
				'version' => 'Dec 2025 Stack',
			],
		];
	}
}
