<?php
/**
 * Base Importer Class
 *
 * @package ProRank\SEO\Core\Importers
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Importers;

defined( 'ABSPATH' ) || exit;

/**
 * Abstract BaseImporter class
 * 
 * Base class for all competitor importers
 */
abstract class BaseImporter {
    
    /**
     * Import statistics
     *
     * @var array
     */
    protected array $stats = [
        'posts' => 0,
        'pages' => 0,
        'terms' => 0,
        'redirects' => 0,
        'settings' => 0,
        'errors' => [],
    ];
    
    /**
     * Import types to process
     *
     * @var array
     */
    protected array $import_types = ['meta', 'settings', 'redirects', 'schema'];
    
    /**
     * Constructor
     *
     * @param array $types Import types to process
     */
    public function __construct(array $types = []) {
        if (!empty($types)) {
            $this->import_types = $types;
        }
    }
    
    /**
     * Run the import
     *
     * @return array Import results
     */
    abstract public function import(): array;
    
    /**
     * Check if the competitor plugin is installed
     *
     * @return bool
     */
    abstract public function is_available(): bool;
    
    /**
     * Import post meta data
     *
     * @return void
     */
    protected function import_post_meta(): void {
        if (!in_array('meta', $this->import_types, true)) {
            return;
        }
        
        $post_types = get_post_types(['public' => true]);
        
        foreach ($post_types as $post_type) {
            $posts = get_posts([
                'post_type' => $post_type,
                'posts_per_page' => -1,
                'post_status' => 'any',
            ]);
            
            foreach ($posts as $post) {
                $this->import_single_post_meta($post->ID);
                $this->stats['posts']++;
            }
        }
    }
    
    /**
     * Import single post meta
     *
     * @param int $post_id Post ID
     * @return void
     */
    abstract protected function import_single_post_meta(int $post_id): void;
    
    /**
     * Import term meta data
     *
     * @return void
     */
    protected function import_term_meta(): void {
        if (!in_array('meta', $this->import_types, true)) {
            return;
        }
        
        $taxonomies = get_taxonomies(['public' => true]);
        
        foreach ($taxonomies as $taxonomy) {
            $terms = get_terms([
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
            ]);
            
            if (is_wp_error($terms)) {
                continue;
            }
            
            foreach ($terms as $term) {
                $this->import_single_term_meta($term->term_id);
                $this->stats['terms']++;
            }
        }
    }
    
    /**
     * Import single term meta
     *
     * @param int $term_id Term ID
     * @return void
     */
    abstract protected function import_single_term_meta(int $term_id): void;
    
    /**
     * Map competitor meta key to ProRank meta key
     *
     * @param string $competitor_key Competitor meta key
     * @return string ProRank meta key
     */
    abstract protected function map_meta_key(string $competitor_key): string;
    
    /**
     * Import settings
     *
     * @return void
     */
    protected function import_settings(): void {
        if (!in_array('settings', $this->import_types, true)) {
            return;
        }
        
        // Override in child classes
    }
    
    /**
     * Import redirects
     *
     * @return void
     */
    protected function import_redirects(): void {
        if (!in_array('redirects', $this->import_types, true)) {
            return;
        }
        
        // Override in child classes
    }
    
    /**
     * Import schema settings
     *
     * @return void
     */
    protected function import_schema(): void {
        if (!in_array('schema', $this->import_types, true)) {
            return;
        }
        
        // Override in child classes
    }
    
    /**
     * Get import statistics
     *
     * @return array
     */
    public function get_stats(): array {
        return $this->stats;
    }
    
    /**
     * Log import error
     *
     * @param string $message Error message
     * @param array  $context Additional context
     * @return void
     */
    protected function log_error(string $message, array $context = []): void {
        $this->stats['errors'][] = [
            'message' => $message,
            'context' => $context,
            'timestamp' => current_time('mysql'),
        ];
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('ProRank SEO Import Error: ' . $message . ' ' . wp_json_encode($context));
        }
    }
    
    /**
     * Clean and sanitize meta value
     *
     * @param mixed $value Meta value
     * @return mixed Sanitized value
     */
    protected function sanitize_meta_value($value) {
        if (is_string($value)) {
            return sanitize_text_field($value);
        }
        
        if (is_array($value)) {
            return array_map([$this, 'sanitize_meta_value'], $value);
        }
        
        return $value;
    }
}