<?php
/**
 * Notifications Feed
 *
 * @package ProRank\SEO\Core\Notifications
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Notifications;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\Config\Settings;
use ProRank\SEO\Core\Managers\LicenseManager;
use ProRank\SEO\Core\ModuleManager;

/**
 * Class NotificationsFeed
 *
 * Manages dashboard notifications feed with local and remote notifications
 */
class NotificationsFeed {
    /**
     * Notification types
     */
    const TYPE_INFO = 'info';
    const TYPE_WARNING = 'warning';
    const TYPE_ERROR = 'error';
    const TYPE_SUCCESS = 'success';
    const TYPE_TIP = 'tip';
    const TYPE_UPDATE = 'update';
    const TYPE_PROMO = 'promo';

    /**
     * Notification priorities
     */
    const PRIORITY_LOW = 'low';
    const PRIORITY_MEDIUM = 'medium';
    const PRIORITY_HIGH = 'high';
    const PRIORITY_CRITICAL = 'critical';

    /**
     * License manager instance
     *
     * @var LicenseManager
     */
    private LicenseManager $license_manager;

    /**
     * Module manager instance
     *
     * @var ModuleManager
     */
    private ModuleManager $module_manager;

    /**
     * Transient key for cached notifications
     *
     * @var string
     */
    private string $cache_key = 'prorank_notifications_feed';

    /**
     * Get remote feed URL
     *
     * @return string
     */
    private function getFeedUrl(): string {
        return defined('PRORANK_NOTIFICATIONS_ENDPOINT') ? PRORANK_NOTIFICATIONS_ENDPOINT : 'https://prorank.io/api/v1/notifications';
    }

    /**
     * Constructor
     *
     * @param LicenseManager $license_manager License manager instance
     * @param ModuleManager  $module_manager  Module manager instance
     */
    public function __construct(LicenseManager $license_manager, ModuleManager $module_manager) {
        $this->license_manager = $license_manager;
        $this->module_manager = $module_manager;
    }

    /**
     * Get all notifications
     *
     * @param array $filters Optional filters
     * @return array
     */
    public function get_notifications(array $filters = []): array {
        $notifications = [];

        // Get local notifications
        $local_notifications = $this->get_local_notifications();
        $notifications = array_merge($notifications, $local_notifications);

        // Get remote notifications
        $remote_notifications = $this->get_remote_notifications();
        $notifications = array_merge($notifications, $remote_notifications);

        // Get dismissed notifications
        $dismissed = $this->get_dismissed_notifications();

        // Filter out dismissed notifications
        $notifications = array_filter($notifications, function($notification) use ($dismissed) {
            return !isset($notification['id']) || !in_array($notification['id'], $dismissed, true);
        });

        // Apply filters
        if (!empty($filters['type'])) {
            $notifications = array_filter($notifications, function($notification) use ($filters) {
                return $notification['type'] === $filters['type'];
            });
        }

        if (!empty($filters['priority'])) {
            $notifications = array_filter($notifications, function($notification) use ($filters) {
                return $notification['priority'] === $filters['priority'];
            });
        }

        // Sort by priority and timestamp
        usort($notifications, function($a, $b) {
            $priority_order = [
                self::PRIORITY_CRITICAL => 0,
                self::PRIORITY_HIGH => 1,
                self::PRIORITY_MEDIUM => 2,
                self::PRIORITY_LOW => 3,
            ];

            $a_priority = $priority_order[$a['priority']] ?? 4;
            $b_priority = $priority_order[$b['priority']] ?? 4;

            if ($a_priority !== $b_priority) {
                return $a_priority - $b_priority;
            }

            // Sort by timestamp (newer first)
            return ($b['timestamp'] ?? 0) - ($a['timestamp'] ?? 0);
        });

        return array_values($notifications);
    }

    /**
     * Get local notifications based on system state
     *
     * @return array
     */
    private function get_local_notifications(): array {
        $notifications = [];
        $current_time = time();

        // Check for onboarding
        if (!Settings::get('onboarding_completed', false)) {
            $notifications[] = [
                'id' => 'onboarding-pending',
                'type' => self::TYPE_INFO,
                'priority' => self::PRIORITY_HIGH,
                'title' => __('Complete ProRank SEO Setup', 'prorank-seo'),
                'message' => __('Get started with ProRank SEO by completing the quick setup wizard.', 'prorank-seo'),
                'action' => [
                    'label' => __('Start Setup', 'prorank-seo'),
                    'url' => admin_url('admin.php?page=prorank-seo-onboarding'),
                ],
                'dismissible' => false,
                'timestamp' => $current_time,
            ];
        }

        // Check for license expiration
        $license_data = $this->license_manager->get_license_data();
        if ($license_data && !empty($license_data['expires_at'])) {
            $expires_timestamp = strtotime($license_data['expires_at']);
            $days_until_expiry = floor(($expires_timestamp - $current_time) / DAY_IN_SECONDS);

            if ($days_until_expiry <= 7 && $days_until_expiry > 0) {
                $notifications[] = [
                    'id' => 'license-expiring-soon',
                    'type' => self::TYPE_WARNING,
                    'priority' => self::PRIORITY_HIGH,
                    'title' => __('License Expiring Soon', 'prorank-seo'),
                    'message' => sprintf(
                        /* translators: %s: placeholder value */
                        __('Your ProRank SEO license expires in %d days. Renew now to continue receiving updates and support.', 'prorank-seo'),
                        $days_until_expiry
                    ),
                    'action' => [
                        'label' => __('Renew License', 'prorank-seo'),
                        'url' => 'https://prorank.io/account/licenses',
                    ],
                    'dismissible' => true,
                    'timestamp' => $current_time,
                ];
            } elseif ($days_until_expiry <= 0) {
                $notifications[] = [
                    'id' => 'license-expired',
                    'type' => self::TYPE_ERROR,
                    'priority' => self::PRIORITY_CRITICAL,
                    'title' => __('License Expired', 'prorank-seo'),
                    'message' => __('Your ProRank SEO license has expired. Renew now to continue receiving updates and support.', 'prorank-seo'),
                    'action' => [
                        'label' => __('Renew License', 'prorank-seo'),
                        'url' => 'https://prorank.io/account/licenses',
                    ],
                    'dismissible' => false,
                    'timestamp' => $current_time,
                ];
            }
        }

        // Check for missing sitemaps
        if ($this->module_manager->is_module_enabled('sitemaps')) {
            $sitemap_path = ABSPATH . 'sitemap_index.xml';
            if (!file_exists($sitemap_path)) {
                $notifications[] = [
                    'id' => 'sitemap-missing',
                    'type' => self::TYPE_WARNING,
                    'priority' => self::PRIORITY_MEDIUM,
                    'title' => __('XML Sitemap Not Generated', 'prorank-seo'),
                    'message' => __('Your XML sitemap needs to be generated. This helps search engines discover your content.', 'prorank-seo'),
                    'action' => [
                        'label' => __('Generate Sitemap', 'prorank-seo'),
                        'url' => admin_url('admin.php?page=prorank-seo-technical#sitemaps'),
                    ],
                    'dismissible' => true,
                    'timestamp' => $current_time,
                ];
            }
        }

        // Check for missing robots.txt
        if (!file_exists(ABSPATH . 'robots.txt')) {
            $notifications[] = [
                'id' => 'robots-txt-missing',
                'type' => self::TYPE_TIP,
                'priority' => self::PRIORITY_LOW,
                'title' => __('No Robots.txt File', 'prorank-seo'),
                'message' => __('Consider creating a robots.txt file to control how search engines crawl your site.', 'prorank-seo'),
                'action' => [
                    'label' => __('Configure Robots.txt', 'prorank-seo'),
                        'url' => admin_url('admin.php?page=prorank-seo-technical#robots-indexing'),
                    ],
                    'dismissible' => true,
                    'timestamp' => $current_time,
                ];
        }

        // Check for plugin updates
        $update_data = get_site_transient('update_plugins');
        if ($update_data && isset($update_data->response['prorank-seo/prorank-seo.php'])) {
            $notifications[] = [
                'id' => 'plugin-update-available',
                'type' => self::TYPE_UPDATE,
                'priority' => self::PRIORITY_MEDIUM,
                'title' => __('ProRank SEO Update Available', 'prorank-seo'),
                'message' => sprintf(
                    /* translators: %s: placeholder value */
                    __('Version %s is available. Update now to get the latest features and improvements.', 'prorank-seo'),
                    $update_data->response['prorank-seo/prorank-seo.php']->new_version
                ),
                'action' => [
                    'label' => __('Update Now', 'prorank-seo'),
                    'url' => admin_url('plugins.php'),
                ],
                'dismissible' => true,
                'timestamp' => $current_time,
            ];
        }

        // Check for disabled critical modules
        $critical_modules = ['meta', 'sitemaps'];
        foreach ($critical_modules as $module_slug) {
            if ($this->module_manager->has_module($module_slug) && !$this->module_manager->is_module_enabled($module_slug)) {
                $module = $this->module_manager->get_module($module_slug);
                if ($module) {
                    $notifications[] = [
                        'id' => 'module-disabled-' . $module_slug,
                        'type' => self::TYPE_TIP,
                        'priority' => self::PRIORITY_MEDIUM,
                        'title' => sprintf(
                            /* translators: %s: module name */
                            __('%s Module Disabled', 'prorank-seo'), $module->get_name()),
                        'message' => sprintf(
                            /* translators: %s: placeholder value */
                            __('The %s module is currently disabled. Enable it to improve your SEO.', 'prorank-seo'),
                            $module->get_name()
                        ),
                        'action' => [
                            'label' => __('Enable Module', 'prorank-seo'),
                            'url' => admin_url('admin.php?page=prorank-seo-modules'),
                        ],
                        'dismissible' => true,
                        'timestamp' => $current_time,
                    ];
                }
            }
        }

        return $notifications;
    }

    /**
     * Get remote notifications from API
     *
     * @return array
     */
    private function get_remote_notifications(): array {
        // Check cache first
        $cached = get_transient($this->cache_key);
        if ($cached !== false) {
            return $cached;
        }

        // Fetch from API
        $response = wp_remote_get($this->getFeedUrl(), [
            'timeout' => 10,
            'headers' => [
                'X-ProRank-Version' => defined('PRORANK_SEO_VERSION') ? PRORANK_SEO_VERSION : '0.1.0',
                'X-ProRank-License' => $this->license_manager->get_license_key() ?: 'free',
                'X-ProRank-Tier' => $this->license_manager->get_tier(),
            ],
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!is_array($data) || empty($data['notifications'])) {
            return [];
        }

        $notifications = $data['notifications'];

        // Filter by tier
        $current_tier = $this->license_manager->get_tier();
        $notifications = array_filter($notifications, function($notification) use ($current_tier) {
            if (!isset($notification['tiers'])) {
                return true; // Show to all if no tier restriction
            }
            return in_array($current_tier, $notification['tiers'], true);
        });

        // Cache for 6 hours
        set_transient($this->cache_key, $notifications, 6 * HOUR_IN_SECONDS);

        return $notifications;
    }

    /**
     * Dismiss a notification
     *
     * @param string $notification_id Notification ID
     * @return bool
     */
    public function dismiss_notification(string $notification_id): bool {
        $dismissed = $this->get_dismissed_notifications();
        
        if (!in_array($notification_id, $dismissed, true)) {
            $dismissed[] = $notification_id;
            return update_user_meta(get_current_user_id(), 'prorank_dismissed_notifications', $dismissed);
        }

        return true;
    }

    /**
     * Get dismissed notifications for current user
     *
     * @return array
     */
    private function get_dismissed_notifications(): array {
        $dismissed = get_user_meta(get_current_user_id(), 'prorank_dismissed_notifications', true);
        return is_array($dismissed) ? $dismissed : [];
    }

    /**
     * Clear notifications cache
     *
     * @return void
     */
    public function clear_cache(): void {
        delete_transient($this->cache_key);
    }

    /**
     * Get notification counts by type
     *
     * @return array
     */
    public function get_notification_counts(): array {
        $notifications = $this->get_notifications();
        $counts = [
            'total' => count($notifications),
            'by_type' => [],
            'by_priority' => [],
        ];

        foreach ($notifications as $notification) {
            $type = $notification['type'] ?? 'info';
            $priority = $notification['priority'] ?? 'low';

            if (!isset($counts['by_type'][$type])) {
                $counts['by_type'][$type] = 0;
            }
            $counts['by_type'][$type]++;

            if (!isset($counts['by_priority'][$priority])) {
                $counts['by_priority'][$priority] = 0;
            }
            $counts['by_priority'][$priority]++;
        }

        return $counts;
    }
}
