<?php
/**
 * Audit Export REST Controller
 *
 * Handles PDF/CSV export and AI fix suggestions via server API
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\ExternalReportClient;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * AuditExportController class
 */
class AuditExportController {

    private const NAMESPACE = 'prorank-seo/v1';

    /**
     * Register routes
     */
    public function register_routes(): void {
        register_rest_route(self::NAMESPACE, '/audit/export/pdf/(?P<audit_id>[a-zA-Z0-9-]+)', [
            'methods' => 'POST',
            'callback' => [$this, 'export_pdf'],
            'permission_callback' => [$this, 'check_permission'],
            'args' => [
                'audit_id' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
                'include_ignored' => [
                    'type' => 'boolean',
                    'default' => false,
                ],
            ],
        ]);

        register_rest_route(self::NAMESPACE, '/audit/export/csv/(?P<audit_id>[a-zA-Z0-9-]+)', [
            'methods' => 'POST',
            'callback' => [$this, 'export_csv'],
            'permission_callback' => [$this, 'check_permission'],
            'args' => [
                'audit_id' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
                'export_type' => [
                    'type' => 'string',
                    'default' => 'issues',
                    'enum' => ['issues', 'urls'],
                ],
            ],
        ]);

        register_rest_route(self::NAMESPACE, '/audit/ai-fix', [
            'methods' => 'POST',
            'callback' => [$this, 'get_ai_suggestions'],
            'permission_callback' => [$this, 'check_permission'],
            'args' => [
                'issues' => [
                    'required' => true,
                    'type' => 'array',
                ],
            ],
        ]);
    }

    /**
     * Check permission
     */
    public function check_permission(): bool {
        return current_user_can('manage_options');
    }

    /**
     * Export PDF report
     */
    public function export_pdf(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $audit_id = $request->get_param('audit_id');
        $include_ignored = $request->get_param('include_ignored');

        $client = new ExternalReportClient();
        $result = $client->request_pdf_report($audit_id, [
            'include_ignored' => $include_ignored,
        ]);

        if (!$result['success']) {
            return new WP_Error(
                'export_failed',
                $result['error'] ?? 'Failed to generate PDF report',
                ['status' => 500]
            );
        }

        return new WP_REST_Response($result, 200);
    }

    /**
     * Export CSV
     */
    public function export_csv(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $audit_id = $request->get_param('audit_id');
        $export_type = $request->get_param('export_type');

        $client = new ExternalReportClient();
        $result = $client->request_csv_export($audit_id, $export_type);

        if (!$result['success']) {
            return new WP_Error(
                'export_failed',
                $result['error'] ?? 'Failed to generate CSV export',
                ['status' => 500]
            );
        }

        return new WP_REST_Response($result, 200);
    }

    /**
     * Get AI fix suggestions
     */
    public function get_ai_suggestions(WP_REST_Request $request): WP_REST_Response|WP_Error {
        $issues = $request->get_param('issues');

        if (empty($issues)) {
            return new WP_Error(
                'no_issues',
                'No issues provided for analysis',
                ['status' => 400]
            );
        }

        $client = new ExternalReportClient();
        $result = $client->get_ai_fix_suggestions($issues);

        if (!$result['success']) {
            $status = isset($result['upgrade_url']) ? 403 : 500;
            return new WP_Error(
                'ai_failed',
                $result['error'] ?? 'Failed to generate AI suggestions',
                ['status' => $status]
            );
        }

        return new WP_REST_Response($result, 200);
    }
}
