<?php
/**
 * Changelog REST API Controller
 *
 * Handles changelog data retrieval
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   3.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Changelog controller class
 */
class ChangelogController extends BaseController {
    
    /**
     * Rest base
     *
     * @var string
     */
    protected $rest_base = 'changelog';
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Get changelog
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_changelog'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
    }
    
    /**
     * Get changelog data
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_changelog(WP_REST_Request $request) {
        // Try to get cached changelog
        $cached_changelog = get_transient('prorank_seo_changelog');
        
        if ($cached_changelog !== false) {
            return new WP_REST_Response($cached_changelog, 200);
        }
        
        // Fetch from remote API or local file
        $changelog = $this->fetch_changelog();
        
        if (is_wp_error($changelog)) {
            // Return hardcoded recent releases as fallback
            $changelog = $this->get_fallback_changelog();
        }
        
        // Cache for 24 hours
        set_transient('prorank_seo_changelog', $changelog, DAY_IN_SECONDS);
        
        return new WP_REST_Response($changelog, 200);
    }
    
    /**
     * Fetch changelog from remote API
     *
     * @return array|WP_Error
     */
    private function fetch_changelog() {
        $response = wp_remote_get('https://prorank.io/api/changelog', [
            'timeout' => 10,
            'headers' => [
                'Accept' => 'application/json',
            ],
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
            return new WP_Error('invalid_response', 'Invalid changelog data received');
        }
        
        return [
            'success'         => true,
            'current_version' => PRORANK_SEO_VERSION,
            'releases'        => $data['releases'] ?? $this->get_fallback_releases(),
        ];
    }
    
    /**
     * Get fallback changelog data
     *
     * @return array
     */
    private function get_fallback_changelog(): array {
        return [
            'success'         => true,
            'current_version' => PRORANK_SEO_VERSION,
            'releases'        => $this->get_fallback_releases(),
        ];
    }
    
    /**
     * Get fallback releases
     *
     * @return array
     */
    private function get_fallback_releases(): array {
        return [
            [
                'version'    => '3.0.0',
                'date'       => '2025-01-10',
                'type'       => 'major',
                'breaking'   => true,
                'highlights' => [
                    'Complete UI redesign with modern React components',
                    'Enhanced performance monitoring and Core Web Vitals',
                    'AI-powered content optimization',
                    'Advanced local SEO features',
                ],
                'changes'    => [
                    'added' => [
                        'New modular CSS architecture for better maintainability',
                        'React Query for optimized data fetching',
                        'TypeScript support for improved type safety',
                        'Zustand state management for better performance',
                        'Advanced feedback system with database storage',
                        'Real-time license validation and management',
                    ],
                    'improved' => [
                        'Completely rebuilt admin interface with modern UI',
                        'Enhanced API response times with caching',
                        'Better error handling and user feedback',
                        'Optimized database queries for faster load times',
                    ],
                    'fixed' => [
                        'Memory leaks in long-running processes',
                        'Race conditions in concurrent API calls',
                        'CSS conflicts with other plugins',
                        'JavaScript errors in older browsers',
                    ],
                    'removed' => [
                        'Deprecated jQuery dependencies',
                        'Legacy code from version 1.x',
                        'Unused database tables',
                    ],
                ],
            ],
            [
                'version'    => '2.9.8',
                'date'       => '2024-12-15',
                'type'       => 'patch',
                'breaking'   => false,
                'highlights' => [
                    'Security updates and bug fixes',
                ],
                'changes'    => [
                    'fixed' => [
                        'XSS vulnerability in search forms',
                        'SQL injection in custom queries',
                        'CSRF token validation issues',
                    ],
                    'improved' => [
                        'Security headers implementation',
                        'Input sanitization across all forms',
                    ],
                ],
            ],
            [
                'version'    => '2.9.7',
                'date'       => '2024-11-20',
                'type'       => 'patch',
                'breaking'   => false,
                'highlights' => [
                    'Performance improvements and compatibility fixes',
                ],
                'changes'    => [
                    'improved' => [
                        'Compatibility with WordPress 6.7',
                        'Database query optimization',
                        'Admin page load times',
                    ],
                    'fixed' => [
                        'PHP 8.3 compatibility issues',
                        'Memory usage in large sites',
                        'Cache invalidation problems',
                    ],
                ],
            ],
            [
                'version'    => '2.9.0',
                'date'       => '2024-10-01',
                'type'       => 'minor',
                'breaking'   => false,
                'highlights' => [
                    'New AI-powered features',
                    'Enhanced schema markup options',
                    'Improved Core Web Vitals monitoring',
                ],
                'changes'    => [
                    'added' => [
                        'AI content suggestions',
                        'Advanced schema types',
                        'Real-time CWV monitoring',
                        'Bulk content optimization',
                    ],
                    'improved' => [
                        'Schema validation accuracy',
                        'Meta description generation',
                        'XML sitemap generation speed',
                    ],
                ],
            ],
            [
                'version'    => '2.8.0',
                'date'       => '2024-08-15',
                'type'       => 'minor',
                'breaking'   => false,
                'highlights' => [
                    'Local SEO enhancements',
                    'E-commerce SEO tools',
                ],
                'changes'    => [
                    'added' => [
                        'Google My Business integration',
                        'Local business schema',
                        'Product schema for WooCommerce',
                        'Review management system',
                    ],
                    'improved' => [
                        'Location-based keyword tracking',
                        'Multi-location support',
                    ],
                ],
            ],
        ];
    }
}
