<?php
/**
 * Command Palette REST API Controller.
 *
 * @package ProRank\SEO\Core\RestApi
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\MenuRegistry;
use ProRank\SEO\Core\LicenseManager;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Command Palette controller class.
 */
class CommandPaletteController extends BaseController {
    /**
     * Menu registry instance.
     *
     * @var MenuRegistry
     */
    private MenuRegistry $menu_registry;

    /**
     * License manager instance.
     *
     * @var LicenseManager
     */
    private LicenseManager $license_manager;

    /**
     * Constructor.
     *
     * @param MenuRegistry   $menu_registry   Menu registry instance.
     * @param LicenseManager $license_manager License manager instance.
     */
    public function __construct(MenuRegistry $menu_registry, LicenseManager $license_manager) {
        $this->menu_registry   = $menu_registry;
        $this->license_manager = $license_manager;
    }

    /**
     * Register routes.
     *
     * @return void
     */
    public function register_routes(): void {
        register_rest_route(
            $this->namespace,
            '/command-palette/items',
            [
                [
                    'methods'             => 'GET',
                    'callback'            => [$this, 'get_items'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [],
                ],
                'schema' => [$this, 'get_items_schema'],
            ]
        );
    }

    /**
     * Get command palette items.
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error Response object or error.
     */
    public function get_items($request) {
        try {
            $items = [];
            
            // Get current user tier
            $current_tier = $this->license_manager->get_tier();
            
            // Get all registered menu items from MenuRegistry
            $menu_items = $this->menu_registry->get_all_items();
            
            // Process menu items
            foreach ($menu_items as $item) {
                // Skip if user doesn't have capability
                if (isset($item['capability']) && !current_user_can($item['capability'])) {
                    continue;
                }
                
                // Prepare command item
                $command_item = [
                    'label'       => $item['menu_title'] ?? $item['page_title'] ?? '',
                    'description' => $item['description'] ?? null,
                    'url'         => $this->build_admin_url($item),
                    'type'        => 'navigation',
                    'locked'      => false,
                    'requiredTier' => null,
                ];
                
                // Check if item is tier-locked
                if (isset($item['required_tier']) && $item['required_tier'] !== 'free') {
                    $is_locked = !$this->license_manager->has_tier_access($item['required_tier']);
                    if ($is_locked) {
                        $command_item['locked'] = true;
                        $command_item['requiredTier'] = $item['required_tier'];
                    }
                }
                
                $items[] = $command_item;
            }
            
            // Add additional command items (non-menu items)
            $additional_items = $this->get_additional_command_items();
            foreach ($additional_items as $additional_item) {
                // Check capability
                if (isset($additional_item['capability']) && !current_user_can($additional_item['capability'])) {
                    continue;
                }
                
                // Check tier
                if (isset($additional_item['required_tier']) && $additional_item['required_tier'] !== 'free') {
                    $is_locked = !$this->license_manager->has_tier_access($additional_item['required_tier']);
                    if ($is_locked) {
                        $additional_item['locked'] = true;
                        $additional_item['requiredTier'] = $additional_item['required_tier'];
                    }
                }
                
                unset($additional_item['capability'], $additional_item['required_tier']);
                $items[] = $additional_item;
            }
            
            return $this->get_success_response([
                'items' => $items,
                'total' => count($items),
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                __('Failed to fetch command palette items.', 'prorank-seo'),
                500,
                'fetch_failed'
            );
        }
    }

    /**
     * Build admin URL for menu item.
     *
     * @param array $item Menu item data.
     * @return string Admin URL.
     */
    private function build_admin_url(array $item): string {
        if (isset($item['menu_slug'])) {
            return admin_url('admin.php?page=' . $item['menu_slug']);
        }
        
        if (isset($item['parent_slug']) && isset($item['menu_slug'])) {
            return admin_url('admin.php?page=' . $item['menu_slug']);
        }
        
        return admin_url();
    }

    /**
     * Get additional command items not in menu.
     *
     * @return array Additional command items.
     */
    private function get_additional_command_items(): array {
        return [
            [
                'label'         => __('SEO Settings', 'prorank-seo'),
                'description'   => __('Configure general SEO settings', 'prorank-seo'),
                'url'           => admin_url('admin.php?page=prorank-seo-dashboard#settings'),
                'type'          => 'navigation',
                'locked'        => false,
                'capability'    => 'manage_options',
                'required_tier' => 'free',
            ],
            [
                'label'         => __('Import/Export Settings', 'prorank-seo'),
                'description'   => __('Import or export your SEO settings', 'prorank-seo'),
                'url'           => admin_url('admin.php?page=prorank-seo-tools#import-export'),
                'type'          => 'navigation',
                'locked'        => false,
                'capability'    => 'manage_options',
                'required_tier' => 'business',
            ],
            [
                'label'         => __('View Documentation', 'prorank-seo'),
                'description'   => __('Open ProRank SEO documentation', 'prorank-seo'),
                'url'           => 'https://docs.prorank.com/',
                'type'          => 'external',
                'locked'        => false,
                'capability'    => 'read',
                'required_tier' => 'free',
            ],
            [
                'label'         => __('Contact Support', 'prorank-seo'),
                'description'   => __('Get help from ProRank support team', 'prorank-seo'),
                'url'           => admin_url('admin.php?page=prorank-seo-support'),
                'type'          => 'navigation',
                'locked'        => false,
                'capability'    => 'manage_options',
                'required_tier' => 'free',
            ],
        ];
    }

    /**
     * Get items schema.
     *
     * @return array Schema definition.
     */
    public function get_items_schema(): array {
        return [
            '$schema'    => 'http://json-schema.org/draft-04/schema#',
            'title'      => 'command-palette-items',
            'type'       => 'object',
            'properties' => [
                'items' => [
                    'description' => __('List of command palette items.', 'prorank-seo'),
                    'type'        => 'array',
                    'items'       => [
                        'type'       => 'object',
                        'properties' => [
                            'label' => [
                                'description' => __('Item label.', 'prorank-seo'),
                                'type'        => 'string',
                            ],
                            'description' => [
                                'description' => __('Item description.', 'prorank-seo'),
                                'type'        => ['string', 'null'],
                            ],
                            'url' => [
                                'description' => __('Navigation URL.', 'prorank-seo'),
                                'type'        => 'string',
                                'format'      => 'uri',
                            ],
                            'action' => [
                                'description' => __('Action identifier.', 'prorank-seo'),
                                'type'        => ['string', 'null'],
                            ],
                            'type' => [
                                'description' => __('Item type.', 'prorank-seo'),
                                'type'        => 'string',
                                'enum'        => ['navigation', 'action', 'external'],
                            ],
                            'locked' => [
                                'description' => __('Whether item is locked.', 'prorank-seo'),
                                'type'        => 'boolean',
                            ],
                            'requiredTier' => [
                                'description' => __('Required tier for access.', 'prorank-seo'),
                                'type'        => ['string', 'null'],
                            ],
                        ],
                    ],
                ],
                'total' => [
                    'description' => __('Total number of items.', 'prorank-seo'),
                    'type'        => 'integer',
                ],
            ],
        ];
    }
}