<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Content Safeguard REST API Controller
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\Settings;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Class ContentSafeguardController
 */
class ContentSafeguardController extends BaseController {
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Settings endpoints
        register_rest_route($this->namespace, '/settings/content-safeguard', [
            [
                'methods'             => 'GET',
                'callback'            => [$this, 'get_settings'],
                'permission_callback' => [$this, 'check_permission'],
            ],
            [
                'methods'             => 'POST',
                'callback'            => [$this, 'update_settings'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'settings' => [
                        'required' => true,
                        'type'     => 'object',
                    ],
                ],
            ],
        ]);

        // Stats endpoint
        register_rest_route($this->namespace, '/content-safeguard/stats', [
            'methods'             => 'GET',
            'callback'            => [$this, 'get_stats'],
            'permission_callback' => [$this, 'check_permission'],
        ]);

        // Recent changes endpoint
        register_rest_route($this->namespace, '/content-safeguard/recent-changes', [
            'methods'             => 'GET',
            'callback'            => [$this, 'get_recent_changes'],
            'permission_callback' => [$this, 'check_permission'],
            'args'                => [
                'limit' => [
                    'default' => 20,
                    'validate_callback' => function($param) {
                        return is_numeric($param) && $param > 0 && $param <= 100;
                    },
                ],
            ],
        ]);

        // Clear log endpoint
        register_rest_route($this->namespace, '/content-safeguard/clear-log', [
            'methods'             => 'POST',
            'callback'            => [$this, 'clear_change_log'],
            'permission_callback' => [$this, 'check_permission'],
        ]);

        // Protect/unprotect post endpoints
        register_rest_route($this->namespace, '/content-safeguard/protect/(?P<id>\d+)', [
            'methods'             => 'POST',
            'callback'            => [$this, 'protect_post'],
            'permission_callback' => [$this, 'check_permission'],
            'args'                => [
                'id' => [
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    },
                ],
            ],
        ]);

        register_rest_route($this->namespace, '/content-safeguard/unprotect/(?P<id>\d+)', [
            'methods'             => 'POST',
            'callback'            => [$this, 'unprotect_post'],
            'permission_callback' => [$this, 'check_permission'],
            'args'                => [
                'id' => [
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    },
                ],
            ],
        ]);
    }

    /**
     * Get content safeguard settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_settings(WP_REST_Request $request) {
        $defaults = [
            'enable_safeguard' => true,
            'protect_all' => false,
            'post_types' => ['post', 'page'],
            'monitor_title' => true,
            'monitor_slug' => true,
            'monitor_content' => false,
            'content_similarity_threshold' => 80,
            'prevent_slug_change' => true,
            'email_notifications' => true,
            'notification_emails' => '',
            'protected_meta_keys' => [
                '_prorank_seo_title',
                '_prorank_seo_description',
                '_prorank_seo_canonical',
                '_prorank_seo_keywords',
            ],
            'require_approval' => true,
            'approval_roles' => ['administrator', 'editor'],
            'log_changes' => true,
            'log_retention_days' => 30,
            'auto_backup' => true,
            'backup_retention_days' => 7,
        ];

        $settings = [];
        foreach ($defaults as $key => $default) {
            $settings[$key] = Settings::get('modules.content_safeguard.' . $key, $default);
        }

        return new WP_REST_Response([
            'settings' => $settings,
        ]);
    }

    /**
     * Update content safeguard settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_settings(WP_REST_Request $request) {
        $settings = $request->get_param('settings');

        if (!is_array($settings)) {
            return new WP_Error(
                'invalid_settings',
                __('Invalid settings format', 'prorank-seo'),
                ['status' => 400]
            );
        }

        // Save each setting
        foreach ($settings as $key => $value) {
            Settings::set('modules.content_safeguard.' . $key, $value);
        }

        // Clear caches
        wp_cache_delete('prorank_protected_posts', 'prorank_seo');

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Settings saved successfully', 'prorank-seo'),
        ]);
    }

    /**
     * Get content safeguard statistics
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_stats(WP_REST_Request $request) {
        global $wpdb;

        // Get protected posts count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $protected_posts = $wpdb->get_var(
            "SELECT COUNT(DISTINCT post_id) 
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             FROM {$wpdb->postmeta} 
             WHERE meta_key = '_prorank_content_protected' 
             AND meta_value = '1'"
        );

        // Get post type breakdown
        $post_types = Settings::get('modules.content_safeguard.post_types', ['post', 'page']);
        $post_type_stats = [];

        foreach ($post_types as $post_type) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(p.ID) 
                 // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                 FROM {$wpdb->posts} p
                 JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
                 WHERE p.post_type = %s 
                 AND p.post_status = 'publish'
                 AND pm.meta_key = '_prorank_content_protected' 
                 AND pm.meta_value = '1'",
                $post_type
            ));
            
            $post_type_stats[$post_type] = (int) $count;
        }

        return new WP_REST_Response([
            'protected_posts' => [
                'total' => (int) $protected_posts,
                'by_type' => $post_type_stats,
            ],
        ]);
    }

    /**
     * Get recent changes
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_recent_changes(WP_REST_Request $request) {
        global $wpdb;
        $limit = $request->get_param('limit');

        // Get recent change logs
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $changes = $wpdb->get_results($wpdb->prepare(
            "SELECT p.ID, p.post_title, pm.meta_value as change_log
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             FROM {$wpdb->posts} p
             JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
             WHERE pm.meta_key = '_prorank_change_log'
             AND p.post_status != 'trash'
             ORDER BY p.post_modified DESC
             LIMIT %d",
            $limit
        ));

        $formatted_changes = [];

        foreach ($changes as $change) {
            $log = maybe_unserialize($change->change_log);
            if (!is_array($log)) {
                continue;
            }

            // Get the most recent log entries
            $recent_entries = array_slice($log, -5);
            
            foreach ($recent_entries as $entry) {
                $user = get_user_by('id', $entry['user_id']);
                
                $formatted_entry = [
                    'post_id' => $change->ID,
                    'post_title' => $change->post_title,
                    'date' => date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $entry['timestamp']),
                    'user' => $user ? $user->display_name : __('Unknown', 'prorank-seo'),
                    'changes' => $entry['changes'],
                ];

                // Format change type
                $change_types = array_keys($entry['changes']);
                $formatted_entry['change_type'] = $this->format_change_type($change_types);
                
                // Add details if available
                $formatted_entry['details'] = $this->format_change_details($entry['changes']);
                
                $formatted_changes[] = $formatted_entry;
            }
        }

        // Sort by date (most recent first)
        usort($formatted_changes, function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });

        // Limit results
        $formatted_changes = array_slice($formatted_changes, 0, $limit);

        return new WP_REST_Response([
            'changes' => $formatted_changes,
        ]);
    }

    /**
     * Clear change log
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function clear_change_log(WP_REST_Request $request) {
        global $wpdb;

        // Delete all change logs
        $deleted = $wpdb->delete(
            $wpdb->postmeta,
            ['meta_key' => '_prorank_change_log']
        );

        return new WP_REST_Response([
            'success' => true,
            'message' => sprintf(
                /* translators: %s: placeholder value */
                __('Cleared %d change log entries', 'prorank-seo'),
                $deleted
            ),
        ]);
    }

    /**
     * Protect a post
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function protect_post(WP_REST_Request $request) {
        $post_id = $request->get_param('id');

        // Check if post exists
        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error(
                'post_not_found',
                __('Post not found', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Enable protection
        update_post_meta($post_id, '_prorank_content_protected', '1');

        // Save current meta values as protected
        $meta_keys = Settings::get('modules.content_safeguard.protected_meta_keys', []);
        $protected_values = [];

        foreach ($meta_keys as $meta_key) {
            $value = get_post_meta($post_id, $meta_key, true);
            if ($value !== '') {
                $protected_values[$meta_key] = $value;
            }
        }

        if (!empty($protected_values)) {
            update_post_meta($post_id, '_prorank_protected_meta', $protected_values);
        }

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Post protected successfully', 'prorank-seo'),
        ]);
    }

    /**
     * Unprotect a post
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function unprotect_post(WP_REST_Request $request) {
        $post_id = $request->get_param('id');

        // Check if post exists
        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error(
                'post_not_found',
                __('Post not found', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Remove protection
        delete_post_meta($post_id, '_prorank_content_protected');
        delete_post_meta($post_id, '_prorank_protected_meta');

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Post unprotected successfully', 'prorank-seo'),
        ]);
    }

    /**
     * Format change type
     *
     * @param array $change_types Change types
     * @return string
     */
    private function format_change_type(array $change_types): string {
        $formatted = [];

        foreach ($change_types as $type) {
            switch ($type) {
                case 'title':
                    $formatted[] = __('Title changed', 'prorank-seo');
                    break;
                case 'slug':
                    $formatted[] = __('URL changed', 'prorank-seo');
                    break;
                case 'content':
                    $formatted[] = __('Content modified', 'prorank-seo');
                    break;
                case 'status':
                    $formatted[] = __('Status changed', 'prorank-seo');
                    break;
                default:
                    $formatted[] = ucfirst($type);
            }
        }

        return implode(', ', $formatted);
    }

    /**
     * Format change details
     *
     * @param array $changes Changes array
     * @return string
     */
    private function format_change_details(array $changes): string {
        $details = [];

        foreach ($changes as $type => $change) {
            switch ($type) {
                case 'title':
                    $details[] = sprintf(
                        __('Title: "%1\$s" → "%2\$s"', 'prorank-seo'),
                        $change['before'],
                        $change['after']
                    );
                    break;

                case 'slug':
                    $details[] = sprintf(
                        __('URL: /%1\$s/ → /%2\$s/', 'prorank-seo'),
                        $change['before'],
                        $change['after']
                    );
                    break;

                case 'content':
                    $details[] = sprintf(
                        __('Content similarity: %1\$.1f%% (Word count: %2\$d → %3\$d)', 'prorank-seo'),
                        $change['similarity'],
                        $change['word_count_before'],
                        $change['word_count_after']
                    );
                    break;

                case 'status':
                    $details[] = sprintf(
                        __('Status: %1\$s → %2\$s', 'prorank-seo'),
                        $change['before'],
                        $change['after']
                    );
                    break;
            }
        }

        return implode(' | ', $details);
    }
}