<?php
/**
 * Data REST API Controller
 *
 * Handles general data endpoints for the plugin
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Data controller class
 */
class DataController extends BaseController {
    
    /**
     * Rest base
     *
     * @var string
     */
    protected $rest_base = 'data';
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // User roles endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/user-roles',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_user_roles'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Post types endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/post-types',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_post_types'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Taxonomies endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/taxonomies',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_taxonomies'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
    }
    
    /**
     * Get user roles
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_user_roles(WP_REST_Request $request) {
        global $wp_roles;
        
        if (!isset($wp_roles)) {
            return $this->get_error_response(
                __('User roles not available', 'prorank-seo'),
                500
            );
        }
        
        $roles = [];
        foreach ($wp_roles->roles as $role_slug => $role_data) {
            $roles[] = [
                'value' => $role_slug,
                'label' => translate_user_role($role_data['name']),
                'capabilities' => array_keys(array_filter($role_data['capabilities'])),
            ];
        }
        
        return rest_ensure_response([
            'roles' => $roles,
        ]);
    }
    
    /**
     * Get public post types
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_post_types(WP_REST_Request $request) {
        $post_types = get_post_types(
            [
                'public' => true,
            ],
            'objects'
        );
        
        $types = [];
        foreach ($post_types as $post_type) {
            // Skip attachment post type
            if ($post_type->name === 'attachment') {
                continue;
            }
            
            $types[] = [
                'value' => $post_type->name,
                'label' => $post_type->labels->name,
                'singular' => $post_type->labels->singular_name,
                'has_archive' => $post_type->has_archive,
                'hierarchical' => $post_type->hierarchical,
                'rewrite' => $post_type->rewrite,
            ];
        }
        
        return rest_ensure_response([
            'post_types' => $types,
        ]);
    }
    
    /**
     * Get public taxonomies
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_taxonomies(WP_REST_Request $request) {
        $taxonomies = get_taxonomies(
            [
                'public' => true,
            ],
            'objects'
        );
        
        $tax_list = [];
        foreach ($taxonomies as $taxonomy) {
            // Skip post format taxonomy
            if ($taxonomy->name === 'post_format') {
                continue;
            }
            
            $tax_list[] = [
                'value' => $taxonomy->name,
                'label' => $taxonomy->labels->name,
                'singular' => $taxonomy->labels->singular_name,
                'hierarchical' => $taxonomy->hierarchical,
                'post_types' => $taxonomy->object_type,
                'rewrite' => $taxonomy->rewrite,
            ];
        }
        
        return rest_ensure_response([
            'taxonomies' => $tax_list,
        ]);
    }
}