<?php
/**
 * Enhanced Audit REST API Controller (Free)
 *
 * Local-only audit endpoints for the free plugin.
 *
 * @package ProRank\SEO\Core\RestApi
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use ProRank\SEO\Core\Audits\BasicAuditEngine;

class EnhancedAuditController extends BaseController {
    /**
     * Register routes
     */
    public function register_routes(): void {
        // Audit status
        register_rest_route($this->namespace, '/audit/status', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_enhanced_status'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);

        // Start audit
        register_rest_route($this->namespace, '/audit/start', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'start_deep_audit'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'type' => [
                        'required'          => false,
                        'default'           => 'basic',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Audit issues
        register_rest_route($this->namespace, '/audit/issues', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_audit_issues'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);

        // Audit history
        register_rest_route($this->namespace, '/audit/history', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_audit_history'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'limit' => [
                        'required'          => false,
                        'default'           => 10,
                        'sanitize_callback' => 'absint',
                    ],
                ],
            ],
        ]);

        // Audit report details
        register_rest_route($this->namespace, '/audit/report/(?P<id>[\w-]+)', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_audit_report'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'id' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Stop audit (local audits run synchronously, but keep endpoint for UI)
        register_rest_route($this->namespace, '/audit/stop', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'stop_audit'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);

        // Quick fix (not available in basic audit, but keep endpoint for UI)
        register_rest_route($this->namespace, '/audit/quick-fix', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'apply_quick_fix'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'fix_id' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'fix_type' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Fix broken link (not available in basic audit, but keep endpoint for UI)
        register_rest_route($this->namespace, '/audit/fix-broken-link', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'fix_broken_link'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'link_id' => [
                        'required'          => true,
                        'sanitize_callback' => 'absint',
                    ],
                    'action' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'new_url' => [
                        'required'          => false,
                        'sanitize_callback' => 'esc_url_raw',
                    ],
                ],
            ],
        ]);

        // Ping endpoint for UI connection checks
        register_rest_route($this->namespace, '/audit/ping', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'ping_audit_service'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);
    }

    /**
     * Get audit status (local basic)
     */
    public function get_enhanced_status(WP_REST_Request $request): WP_REST_Response {
        $engine = new BasicAuditEngine();
        $status = $engine->get_latest_status();

        if (empty($status)) {
            return new WP_REST_Response([
                'state' => 'idle',
                'message' => __('No active audit', 'prorank-seo'),
                'can_start' => true,
            ], 200);
        }

        $status['score'] = $status['overallScore'] ?? $status['overall_score'] ?? null;
        $status['stats'] = $this->build_stats_from_status($status);
        $status['quick_fixes'] = [];

        return new WP_REST_Response($status, 200);
    }

    /**
     * Start a local basic audit
     */
    public function start_deep_audit(WP_REST_Request $request): WP_REST_Response {
        // Get settings from request - frontend sends { mode: 'local', settings: {...} }
        $settings = $request->get_param('settings');

        // If settings param not found, try extracting from JSON body
        if (!is_array($settings)) {
            $json_params = $request->get_json_params();
            if (is_array($json_params) && isset($json_params['settings']) && is_array($json_params['settings'])) {
                $settings = $json_params['settings'];
            } else {
                $settings = $json_params;
            }
        }

        // Fallback to saved settings if still not an array
        if (!is_array($settings)) {
            $settings = get_option('prorank_audit_settings', []);
        }

        $engine = new BasicAuditEngine();
        $result = $engine->run($settings);

        if (is_wp_error($result)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $result->get_error_message(),
                'error' => $result->get_error_code(),
            ], 400);
        }

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Basic audit completed locally', 'prorank-seo'),
            'audit_id' => $result['audit_id'] ?? null,
            'status' => 'completed',
            'mode' => 'local_basic',
        ], 200);
    }

    /**
     * Get audit history
     */
    public function get_audit_history(WP_REST_Request $request): WP_REST_Response {
        $limit = $request->get_param('limit');
        $limit = is_numeric($limit) ? (int) $limit : 10;
        $limit = max(1, min(50, $limit));

        $engine = new BasicAuditEngine();
        $audits = $engine->get_history($limit);

        $normalized = array_map(function ($audit) {
            $audit_id = $audit['audit_id'] ?? $audit['id'] ?? '';
            $score = $audit['overallScore'] ?? $audit['overall_score'] ?? 0;
            $counts = $audit['issues'] ?? [];

            return [
                'id' => $audit_id,
                'audit_id' => $audit_id,
                'date' => $audit['completed_at'] ?? $audit['started_at'] ?? $audit['created_at'] ?? current_time('mysql'),
                'score' => (int) $score,
                'totalUrls' => (int) ($audit['total_urls'] ?? $audit['totalUrls'] ?? 0),
                'total_issues' => (int) ($counts['total'] ?? 0),
                'critical' => (int) ($counts['critical'] ?? 0),
                'duration' => null,
                'stats' => [
                    'total_urls' => (int) ($audit['total_urls'] ?? $audit['totalUrls'] ?? 0),
                    'total_issues' => (int) ($counts['total'] ?? 0),
                    'critical_issues' => (int) ($counts['critical'] ?? 0),
                    'high_issues' => (int) ($counts['high'] ?? 0),
                    'medium_issues' => (int) ($counts['medium'] ?? 0),
                    'low_issues' => (int) ($counts['low'] ?? 0),
                ],
                'completed_at' => $audit['completed_at'] ?? null,
                'started_at' => $audit['started_at'] ?? null,
            ];
        }, $audits);

        return new WP_REST_Response([
            'audits' => $normalized,
            'total' => count($normalized),
        ], 200);
    }

    /**
     * Get audit issues
     */
    public function get_audit_issues(WP_REST_Request $request): WP_REST_Response {
        $engine = new BasicAuditEngine();
        $status = $engine->get_latest_status();

        if (empty($status['audit_id'])) {
            return new WP_REST_Response(['issues' => []], 200);
        }

        $issues = $engine->get_issues((string) $status['audit_id']);

        return new WP_REST_Response([
            'issues' => $issues,
        ], 200);
    }

    /**
     * Get audit report details
     */
    public function get_audit_report(WP_REST_Request $request): WP_REST_Response {
        $report_id = (string) $request->get_param('id');

        if ($report_id === '') {
            return new WP_REST_Response([
                'error' => __('Report ID is required', 'prorank-seo'),
            ], 400);
        }

        $engine = new BasicAuditEngine();
        $issues = $engine->get_issues($report_id);

        if (empty($issues)) {
            return new WP_REST_Response([
                'error' => __('Report not found', 'prorank-seo'),
            ], 404);
        }

        $counts = $this->count_issues($issues);
        $issues_by_page = $this->group_issues_by_page($issues);
        $category_breakdown = $this->build_category_breakdown($issues);

        $report = [
            'id' => $report_id,
            'date' => current_time('mysql'),
            'score' => $this->calculate_score($counts),
            'stats' => [
                'critical_issues' => $counts['critical'],
                'high_issues' => $counts['high'],
                'medium_issues' => $counts['medium'],
                'low_issues' => $counts['low'],
                'total_issues' => $counts['total'],
                'total_urls' => count($issues_by_page),
            ],
            'issues_by_page' => $issues_by_page,
            'category_breakdown' => $category_breakdown,
        ];

        return new WP_REST_Response($report, 200);
    }

    /**
     * Stop audit (no-op for synchronous local audits)
     */
    public function stop_audit(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response([
            'success' => true,
            'message' => __('No active audit to stop', 'prorank-seo'),
        ], 200);
    }

    /**
     * Quick fixes are not available in the free local audit
     */
    public function apply_quick_fix(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('Quick fixes are available in the premium audit.', 'prorank-seo'),
        ], 200);
    }

    /**
     * Broken link fixes are not available in the free local audit
     */
    public function fix_broken_link(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response([
            'success' => false,
            'message' => __('Broken link fixes are available in the premium audit.', 'prorank-seo'),
        ], 200);
    }

    /**
     * Ping endpoint
     */
    public function ping_audit_service(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response([
            'success' => true,
            'message' => 'pong',
            'timestamp' => time(),
        ], 200);
    }

    private function build_stats_from_status(array $status): array {
        $critical = (int) ($status['issuesCritical'] ?? 0);
        $high = (int) ($status['issuesHigh'] ?? 0);
        $medium = (int) ($status['issuesMedium'] ?? 0);
        $low = (int) ($status['issuesLow'] ?? 0);
        $total = $critical + $high + $medium + $low;

        return [
            'critical_issues' => $critical,
            'high_issues' => $high,
            'medium_issues' => $medium,
            'low_issues' => $low,
            'total_issues' => $total,
            'total_urls' => (int) ($status['total_urls'] ?? $status['totalUrls'] ?? 0),
        ];
    }

    private function count_issues(array $issues): array {
        $counts = [
            'critical' => 0,
            'high' => 0,
            'medium' => 0,
            'low' => 0,
            'total' => 0,
        ];

        foreach ($issues as $issue) {
            $severity = (string) ($issue['severity'] ?? 'low');
            if (!isset($counts[$severity])) {
                $severity = 'low';
            }
            $counts[$severity]++;
            $counts['total']++;
        }

        return $counts;
    }

    private function calculate_score(array $counts): int {
        $score = 100;
        $score -= min(60, $counts['critical'] * 15);
        $score -= min(40, $counts['high'] * 8);
        $score -= min(25, $counts['medium'] * 4);
        $score -= min(15, $counts['low'] * 2);

        return max(0, min(100, $score));
    }

    private function group_issues_by_page(array $issues): array {
        $grouped = [];

        foreach ($issues as $issue) {
            $url = (string) ($issue['url'] ?? '');
            if ($url === '') {
                $url = home_url('/');
            }

            if (!isset($grouped[$url])) {
                $grouped[$url] = [];
            }

            $grouped[$url][] = [
                'severity' => $issue['severity'] ?? 'low',
                'check_type' => $issue['type'] ?? $issue['issue_type'] ?? __('Issue', 'prorank-seo'),
                'message' => $issue['message'] ?? $issue['description'] ?? '',
            ];
        }

        return $grouped;
    }

    private function build_category_breakdown(array $issues): array {
        $categories = [];

        foreach ($issues as $issue) {
            $category = (string) ($issue['display_category'] ?? $issue['category'] ?? 'Other');
            if ($category === '') {
                $category = 'Other';
            }

            if (!isset($categories[$category])) {
                $categories[$category] = [
                    'issues' => 0,
                    'score' => 100,
                ];
            }

            $categories[$category]['issues']++;
        }

        foreach ($categories as $key => $data) {
            $score = max(0, 100 - ($data['issues'] * 10));
            $categories[$key]['score'] = $score;
        }

        return $categories;
    }
}
