<?php
/**
 * Feedback REST API Controller
 *
 * Handles feedback submission via REST API.
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

class FeedbackController extends BaseController {
    /**
     * Constructor
     */
    public function __construct() {
        $this->rest_base = 'feedback';
    }

    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Simple feedback (positive/negative)
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => \WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'submit_feedback'],
                    'permission_callback' => [$this, 'submit_feedback_permissions_check'],
                    'args'                => $this->get_feedback_schema(),
                ],
            ]
        );
        
        // Comprehensive feedback submission
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/submit',
            [
                [
                    'methods'             => \WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'submit_comprehensive_feedback'],
                    'permission_callback' => [$this, 'submit_feedback_permissions_check'],
                    'args'                => $this->get_comprehensive_feedback_schema(),
                ],
            ]
        );
        
        // Get feedback list
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/list',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_feedback_list'],
                    'permission_callback' => [$this, 'submit_feedback_permissions_check'],
                    'args'                => [
                        'type' => [
                            'description' => __('Filter by feedback type', 'prorank-seo'),
                            'type'        => 'string',
                            'enum'        => ['all', 'feature', 'bug', 'general', 'support'],
                            'default'     => 'all',
                        ],
                        'limit' => [
                            'description' => __('Number of items to return', 'prorank-seo'),
                            'type'        => 'integer',
                            'default'     => 10,
                            'minimum'     => 1,
                            'maximum'     => 100,
                        ],
                    ],
                ],
            ]
        );
    }

    /**
     * Check permissions for submitting feedback
     *
     * @param WP_REST_Request $request Request object.
     * @return bool|WP_Error
     */
    public function submit_feedback_permissions_check(WP_REST_Request $request) {
        if (!current_user_can('read')) {
            return new WP_Error(
                'rest_forbidden',
                __('You do not have permission to submit feedback.', 'prorank-seo'),
                ['status' => 403]
            );
        }

        return true;
    }

    /**
     * Submit feedback
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function submit_feedback(WP_REST_Request $request) {
        return $this->try_callback(function() use ($request) {
            $type = $request->get_param('type');
            $screen_id = $request->get_param('screen_id');

            // Log feedback (placeholder for Phase 0)
            prorank_log(sprintf(
                '[ProRank SEO Feedback] Type: %s, Screen: %s, User: %d',
                $type,
                $screen_id,
                get_current_user_id()
            ));

            return $this->get_success_response(
                ['submitted' => true],
                __('Thank you for your feedback!', 'prorank-seo')
            );
        });
    }

    /**
     * Submit comprehensive feedback
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function submit_comprehensive_feedback(WP_REST_Request $request) {
        return $this->try_callback(function() use ($request) {
            $type = $request->get_param('type');
            $subject = $request->get_param('subject');
            $message = $request->get_param('message');
            $priority = $request->get_param('priority');
            $email = $request->get_param('email');

            // Get current user info
            $current_user = wp_get_current_user();
            $user_email = $email ?: $current_user->user_email;
            $user_name = $current_user->display_name ?: __('Anonymous', 'prorank-seo');

            // Get license key
            $license_key = get_option('prorank_license_key', '');

            // Get system info for better debugging
            global $wp_version;
            $system_info = [
                'php_version' => PHP_VERSION,
                'wp_version' => $wp_version,
                'plugin_version' => defined('PRORANK_VERSION') ? PRORANK_VERSION : '1.0.0',
                'active_theme' => get_template(),
                'is_multisite' => is_multisite(),
                'memory_limit' => ini_get('memory_limit'),
            ];

            // Prepare feedback data
            $feedback_data = [
                'type' => $type,
                'subject' => $subject,
                'message' => $message,
                'priority' => $priority,
                'user_email' => $user_email,
                'user_name' => $user_name,
                'site_url' => get_site_url(),
                'license_key' => $license_key,
                'timestamp' => current_time('mysql'),
                'system_info' => $system_info,
            ];

            // Try to send to prorank.io server
            $sent_to_server = $this->send_feedback_to_server($feedback_data);

            if (!$sent_to_server) {
                // Store locally as fallback
                $this->store_comprehensive_feedback($feedback_data);
            }

            return $this->get_success_response(
                [
                    'success' => true,
                    'synced' => $sent_to_server,
                ],
                $sent_to_server
                    ? __('Thank you for your feedback! We\'ll review it soon.', 'prorank-seo')
                    : __('Thank you for your feedback! It has been saved and will be synced when connection is available.', 'prorank-seo')
            );
        });
    }
    
    /**
     * Get feedback list
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function get_feedback_list(WP_REST_Request $request) {
        return $this->try_callback(function() use ($request) {
            $type = $request->get_param('type');
            $limit = $request->get_param('limit');
            
            // Fallback to local data if server is unavailable
            $feedback_queue = get_option('prorank_feedback_queue', []);
            
            // Filter by type if needed
            if ($type && $type !== 'all') {
                $feedback_queue = array_filter($feedback_queue, function($item) use ($type) {
                    return $item['type'] === $type;
                });
            }
            
            // Sort by timestamp (newest first)
            usort($feedback_queue, function($a, $b) {
                $time_a = $a['submitted_at'] ?? $a['timestamp'] ?? '1970-01-01 00:00:00';
                $time_b = $b['submitted_at'] ?? $b['timestamp'] ?? '1970-01-01 00:00:00';
                return strtotime($time_b) - strtotime($time_a);
            });
            
            // Limit results
            $feedback_queue = array_slice($feedback_queue, 0, $limit);
            
            return $this->get_success_response([
                'feedback' => $feedback_queue,
                'stats' => [],
                'source' => 'local',
            ]);
        });
    }
    
    /**
     * Send feedback to prorank.io server
     *
     * @param array $feedback_data Feedback data
     * @return bool Success status
     */
    private function send_feedback_to_server(array $feedback_data): bool {
        // Get the feedback endpoint
        $endpoint = defined('PRORANK_FEEDBACK_ENDPOINT')
            ? PRORANK_FEEDBACK_ENDPOINT
            : 'https://prorank.io/api/v1/feedback';

        // Prepare the request
        $response = wp_remote_post($endpoint, [
            'timeout' => 15,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-ProRank-Plugin-Version' => defined('PRORANK_VERSION') ? PRORANK_VERSION : '1.0.0',
            ],
            'body' => wp_json_encode([
                'type' => $feedback_data['type'],
                'subject' => $feedback_data['subject'],
                'message' => $feedback_data['message'],
                'priority' => $feedback_data['priority'] ?? 'medium',
                'email' => $feedback_data['user_email'],
                'name' => $feedback_data['user_name'],
                'site_url' => $feedback_data['site_url'],
                'license_key' => $feedback_data['license_key'],
                'system_info' => $feedback_data['system_info'] ?? [],
                'submitted_at' => $feedback_data['timestamp'],
            ]),
        ]);

        // Check for errors
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                prorank_log('[ProRank SEO] Feedback API error: ' . $response->get_error_message());
            }
            return false;
        }

        // Check response code
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code < 200 || $response_code >= 300) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                prorank_log('[ProRank SEO] Feedback API returned status: ' . $response_code);
            }
            return false;
        }

        // Parse response body
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        return isset($data['success']) && $data['success'] === true;
    }
    
    /**
     * Store comprehensive feedback
     *
     * @param array $feedback_data Feedback data
     * @return bool Success status
     */
    private function store_comprehensive_feedback(array $feedback_data): bool {
        $feedback_queue = get_option('prorank_feedback_queue', []);

        // Add unique ID
        $feedback_data['id'] = wp_generate_uuid4();
        if (empty($feedback_data['submitted_at'])) {
            $feedback_data['submitted_at'] = current_time('mysql');
        }
        if (empty($feedback_data['timestamp'])) {
            $feedback_data['timestamp'] = $feedback_data['submitted_at'];
        }

        // Add status for frontend display
        $feedback_data['status'] = 'pending';

        // Add createdAt for frontend (formatted date)
        $feedback_data['createdAt'] = current_time('F j, Y');

        // Add to queue
        $feedback_queue[] = $feedback_data;

        // Keep only last 100 items
        if (count($feedback_queue) > 100) {
            $feedback_queue = array_slice($feedback_queue, -100);
        }

        return update_option('prorank_feedback_queue', $feedback_queue);
    }

    /**
     * Get feedback schema
     *
     * @return array
     */
    private function get_feedback_schema(): array {
        return [
            'type' => [
                'description'       => __('Feedback type (positive or negative).', 'prorank-seo'),
                'type'              => 'string',
                'enum'              => ['positive', 'negative'],
                'required'          => true,
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'screen_id' => [
                'description'       => __('Current admin screen ID.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => true,
                'sanitize_callback' => 'sanitize_text_field',
            ],
        ];
    }
    
    /**
     * Get comprehensive feedback schema
     *
     * @return array
     */
    private function get_comprehensive_feedback_schema(): array {
        return [
            'type' => [
                'description'       => __('Type of feedback', 'prorank-seo'),
                'type'              => 'string',
                'enum'              => ['feature', 'bug', 'general', 'support'],
                'required'          => true,
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'subject' => [
                'description'       => __('Feedback subject', 'prorank-seo'),
                'type'              => 'string',
                'required'          => true,
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'message' => [
                'description'       => __('Feedback message', 'prorank-seo'),
                'type'              => 'string',
                'required'          => true,
                'sanitize_callback' => 'sanitize_textarea_field',
            ],
            'priority' => [
                'description'       => __('Priority level', 'prorank-seo'),
                'type'              => 'string',
                'enum'              => ['low', 'medium', 'high'],
                'default'           => 'medium',
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'email' => [
                'description'       => __('Contact email', 'prorank-seo'),
                'type'              => 'string',
                'format'            => 'email',
                'sanitize_callback' => 'sanitize_email',
            ],
        ];
    }
}
