<?php
/**
 * Headline Analyzer REST API Controller
 *
 * REST API endpoints for Google-aligned headline analysis.
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\Content\HeadlineAnalyzer;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WP_Error;

/**
 * HeadlineAnalyzerController class
 *
 * Provides REST endpoints for headline analysis:
 * - POST /headline-analyzer/analyze - Analyze a title
 * - GET /headline-analyzer/duplicates/{post_id} - Get duplicate titles
 */
class HeadlineAnalyzerController extends BaseController {
    /**
     * Headline score meta keys.
     */
    private const HEADLINE_SCORE_META = '_prorank_headline_score';
    private const HEADLINE_TITLE_HASH_META = '_prorank_headline_title_hash';
    private const HEADLINE_UPDATED_META = '_prorank_headline_updated';

    /**
     * REST base
     *
     * @var string
     */
    protected $rest_base = 'headline-analyzer';

    /**
     * Register REST routes
     *
     * @return void
     */
    public function register_routes(): void {
        // POST /prorank-seo/v1/headline-analyzer/analyze
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/analyze',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [ $this, 'analyze_headline' ],
                    'permission_callback' => [ $this, 'check_permission_edit_posts' ],
                    'args'                => [
                        'title'   => [
                            'description'       => __( 'Title to analyze', 'prorank-seo' ),
                            'type'              => 'string',
                            'required'          => true,
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                        'post_id' => [
                            'description'       => __( 'Post ID for context', 'prorank-seo' ),
                            'type'              => 'integer',
                            'required'          => false,
                            'default'           => 0,
                            'sanitize_callback' => 'absint',
                        ],
                    ],
                ],
            ]
        );

        // GET /prorank-seo/v1/headline-analyzer/duplicates/{post_id}
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/duplicates/(?P<post_id>\d+)',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [ $this, 'get_duplicates' ],
                    'permission_callback' => [ $this, 'check_permission_edit_posts' ],
                    'args'                => [
                        'post_id' => [
                            'description'       => __( 'Post ID', 'prorank-seo' ),
                            'type'              => 'integer',
                            'required'          => true,
                            'minimum'           => 1,
                            'sanitize_callback' => 'absint',
                        ],
                    ],
                ],
            ]
        );

        // GET /prorank-seo/v1/headline-analyzer/{post_id}
        // Convenience endpoint to analyze current post's title
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<post_id>\d+)',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [ $this, 'analyze_post' ],
                    'permission_callback' => [ $this, 'check_permission_edit_posts' ],
                    'args'                => [
                        'post_id' => [
                            'description'       => __( 'Post ID', 'prorank-seo' ),
                            'type'              => 'integer',
                            'required'          => true,
                            'minimum'           => 1,
                            'sanitize_callback' => 'absint',
                        ],
                    ],
                ],
            ]
        );
    }

    /**
     * Analyze a headline
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error Response or error.
     */
    public function analyze_headline( WP_REST_Request $request ) {
        return $this->try_callback( function () use ( $request ) {
            $title = $request->get_param( 'title' );
            $post_id = (int) $request->get_param( 'post_id' );

            if ( empty( $title ) ) {
                return $this->get_error_response(
                    __( 'Title is required', 'prorank-seo' ),
                    400,
                    'missing_title'
                );
            }

            $analysis = HeadlineAnalyzer::analyze( $title, $post_id );
            $this->maybe_update_headline_score( $post_id, $title, $analysis );

            return $this->get_success_response( $analysis );
        } );
    }

    /**
     * Analyze a post's current title
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error Response or error.
     */
    public function analyze_post( WP_REST_Request $request ) {
        return $this->try_callback( function () use ( $request ) {
            $post_id = (int) $request->get_param( 'post_id' );

            if ( $post_id <= 0 ) {
                return $this->get_error_response(
                    __( 'Invalid post ID', 'prorank-seo' ),
                    400,
                    'invalid_post_id'
                );
            }

            $post = get_post( $post_id );
            if ( ! $post ) {
                return $this->get_error_response(
                    __( 'Post not found', 'prorank-seo' ),
                    404,
                    'post_not_found'
                );
            }

            // Get SEO title if set, otherwise use post title
            $seo_title = get_post_meta( $post_id, '_prorank_title', true );
            $title = ! empty( $seo_title ) ? $seo_title : $post->post_title;

            $analysis = HeadlineAnalyzer::analyze( $title, $post_id );
            $this->maybe_update_headline_score( $post_id, $title, $analysis );

            // Add source info
            $analysis['title_source'] = ! empty( $seo_title ) ? 'seo_title' : 'post_title';

            return $this->get_success_response( $analysis );
        } );
    }

    /**
     * Get duplicate titles for a post
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error Response or error.
     */
    public function get_duplicates( WP_REST_Request $request ) {
        return $this->try_callback( function () use ( $request ) {
            $post_id = (int) $request->get_param( 'post_id' );

            if ( $post_id <= 0 ) {
                return $this->get_error_response(
                    __( 'Invalid post ID', 'prorank-seo' ),
                    400,
                    'invalid_post_id'
                );
            }

            $post = get_post( $post_id );
            if ( ! $post ) {
                return $this->get_error_response(
                    __( 'Post not found', 'prorank-seo' ),
                    404,
                    'post_not_found'
                );
            }

            // Get SEO title if set, otherwise use post title
            $seo_title = get_post_meta( $post_id, '_prorank_title', true );
            $title = ! empty( $seo_title ) ? $seo_title : $post->post_title;

            // Run uniqueness analysis
            $analysis = HeadlineAnalyzer::analyze( $title, $post_id );

            return $this->get_success_response( [
                'title'       => $title,
                'is_unique'   => $analysis['uniqueness']['is_unique'] ?? true,
                'duplicates'  => $analysis['uniqueness']['duplicates'] ?? [],
                'is_template' => $analysis['uniqueness']['is_template'] ?? false,
                'score'       => $analysis['uniqueness']['score'] ?? 100,
            ] );
        } );
    }

    /**
     * Store headline score for list views.
     *
     * @param int    $post_id Post ID.
     * @param string $title   Title used for analysis.
     * @param array  $analysis Analysis results.
     * @return void
     */
    private function maybe_update_headline_score( int $post_id, string $title, array $analysis ): void {
        if ( $post_id <= 0 ) {
            return;
        }

        if ( ! isset( $analysis['overall_score'] ) ) {
            return;
        }

        $score = (int) $analysis['overall_score'];
        $title_hash = md5( $title );

        $stored_hash = get_post_meta( $post_id, self::HEADLINE_TITLE_HASH_META, true );
        $stored_score = get_post_meta( $post_id, self::HEADLINE_SCORE_META, true );

        if ( $stored_hash === $title_hash && $stored_score !== '' && (int) $stored_score === $score ) {
            return;
        }

        update_post_meta( $post_id, self::HEADLINE_SCORE_META, $score );
        update_post_meta( $post_id, self::HEADLINE_TITLE_HASH_META, $title_hash );
        update_post_meta( $post_id, self::HEADLINE_UPDATED_META, current_time( 'mysql' ) );
    }
}
