<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Image REST API Controller
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\RestApi\BaseController;
use ProRank\SEO\Plugin;
use ProRank\SEO\Core\ImageOptimization\OptimizationEngine;

/**
 * ImageController class
 */
class ImageController extends BaseController {
	
	/**
	 * Register routes
	 */
	public function register_routes(): void {
		// Image optimization endpoint
		register_rest_route(
			$this->namespace,
			'/images/optimize',
			[
				[
					'methods'             => 'POST',
					'callback'            => [ $this, 'optimize_image' ],
					'permission_callback' => [ $this, 'check_image_permission' ],
					'args'                => [
						'attachment_id' => [
							'required'          => true,
							'type'              => 'integer',
							'sanitize_callback' => 'absint',
						],
					],
				],
			]
		);
		
		// Get optimization stats for specific image
		register_rest_route(
			$this->namespace,
			'/images/(?P<id>\d+)/optimization-stats',
			[
				[
					'methods'             => 'GET',
					'callback'            => [ $this, 'get_image_optimization_stats' ],
					'permission_callback' => [ $this, 'check_image_permission' ],
					'args'                => [
						'id' => [
							'required'          => true,
							'type'              => 'integer',
							'sanitize_callback' => 'absint',
						],
					],
				],
			]
		);
		
		// Bulk optimization status endpoint
		register_rest_route(
			$this->namespace,
			'/images/optimization-status',
			[
				[
					'methods'             => 'GET',
					'callback'            => [ $this, 'get_optimization_status' ],
					'permission_callback' => [ $this, 'check_image_permission' ],
				],
			]
		);
		
		// Get optimization capabilities
		register_rest_route(
			$this->namespace,
			'/images/capabilities',
			[
				[
					'methods'             => 'GET',
					'callback'            => [ $this, 'get_capabilities' ],
					'permission_callback' => [ $this, 'check_image_permission' ],
				],
			]
		);
		
		// Convert image to WebP
		register_rest_route(
			$this->namespace,
			'/images/convert-webp',
			[
				[
					'methods'             => 'POST',
					'callback'            => [ $this, 'convert_to_webp' ],
					'permission_callback' => [ $this, 'check_image_permission' ],
					'args'                => [
						'attachment_id' => [
							'required'          => true,
							'type'              => 'integer',
							'sanitize_callback' => 'absint',
						],
					],
				],
			]
		);
		
		// Get unoptimized images
		register_rest_route(
			$this->namespace,
			'/images/unoptimized',
			[
				[
					'methods'             => 'GET',
					'callback'            => [ $this, 'get_unoptimized_images' ],
					'permission_callback' => [ $this, 'check_image_permission' ],
				],
			]
		);
	}
	
	/**
	 * Check permission for image operations
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return bool|\WP_Error
	 */
	public function check_image_permission( \WP_REST_Request $request ): bool {
		if ( $request->get_route() === '/prorank-seo/v1/images/optimize' ) {
			$attachment_id = $request->get_param( 'attachment_id' );
			
			// Check if user can edit this attachment
			if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
				return false;
			}
			
			// Verify it's an image
			if ( ! wp_attachment_is_image( $attachment_id ) ) {
				return false;
			}
		}
		
		return current_user_can( 'upload_files' );
	}
	
	/**
	 * Optimize image
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function optimize_image( \WP_REST_Request $request ) {
		$attachment_id = $request->get_param( 'attachment_id' );
		
		// Use the new optimization engine
		$engine = new OptimizationEngine();
		$result = $engine->optimize_image( $attachment_id );
		
		if ( $result['success'] ) {
			// Clear image cache
			clean_attachment_cache( $attachment_id );
			
			// Regenerate thumbnails
			$metadata = wp_get_attachment_metadata( $attachment_id );
			if ( $metadata ) {
				wp_update_attachment_metadata( $attachment_id, $metadata );
			}
			
			return rest_ensure_response( $result );
		}
		
		return new \WP_Error(
			'optimization_failed',
			$result['error'] ?? __( 'Failed to optimize image.', 'prorank-seo' ),
			[ 'status' => 500 ]
		);
	}
	
	/**
	 * Get image optimization stats
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function get_image_optimization_stats( \WP_REST_Request $request ) {
		$attachment_id = $request->get_param( 'id' );
		
		// Verify it's an image
		if ( ! wp_attachment_is_image( $attachment_id ) ) {
			return new \WP_Error(
				'not_an_image',
				__( 'Attachment is not an image.', 'prorank-seo' ),
				[ 'status' => 400 ]
			);
		}
		
		// Get optimization data
		$is_optimized = get_post_meta( $attachment_id, '_prorank_optimized', true );
		
		if ( ! $is_optimized ) {
			return rest_ensure_response( [
				'optimized' => false,
				'stats' => null,
			] );
		}
		
		// Get all optimization metadata
		$original_size = get_post_meta( $attachment_id, '_prorank_original_size', true );
		$optimized_size = get_post_meta( $attachment_id, '_prorank_optimized_size', true );
		$saved_bytes = get_post_meta( $attachment_id, '_prorank_optimization_savings', true );
		$saved_percent = get_post_meta( $attachment_id, '_prorank_optimization_percent', true );
		$optimization_date = get_post_meta( $attachment_id, '_prorank_optimization_date', true );
		$optimization_method = get_post_meta( $attachment_id, '_prorank_optimization_method', true );
		
		return rest_ensure_response( [
			'optimized' => true,
			'stats' => [
				'original_size' => size_format( (int) $original_size ),
				'optimized_size' => size_format( (int) $optimized_size ),
				'saved_size' => size_format( (int) $saved_bytes ),
				'saved_percent' => (float) $saved_percent,
				'optimization_date' => $optimization_date,
				'optimization_method' => $optimization_method ?: 'Unknown',
			],
		] );
	}
	
	/**
	 * Get optimization status
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response
	 */
	public function get_optimization_status( \WP_REST_Request $request ) {
		// Get total images
		$total_images = wp_count_attachments();
		$image_count = ( $total_images->{'image/jpeg'} ?? 0 ) + 
		               ( $total_images->{'image/png'} ?? 0 ) + 
		               ( $total_images->{'image/webp'} ?? 0 );
		
		// Get optimized images count
		$optimized_count = (int) get_posts( [
			'post_type'      => 'attachment',
			'post_mime_type' => 'image',
			'meta_key'       => '_prorank_optimized',
			'meta_value'     => '1',
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'count_total'    => true,
		] );
		
		// Calculate total savings
		global $wpdb;
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
		$total_savings = $wpdb->get_var(
			"SELECT SUM(meta_value) FROM {$wpdb->postmeta} 
			WHERE meta_key = '_prorank_optimization_savings'"
		);
		
		return rest_ensure_response( [
			'total_images' => $image_count,
			'optimized_images' => $optimized_count,
			'unoptimized_images' => $image_count - $optimized_count,
			'total_savings' => size_format( (int) $total_savings ),
			'optimization_percentage' => $image_count > 0 ? round( ( $optimized_count / $image_count ) * 100, 2 ) : 0,
		] );
	}
	
	/**
	 * Get optimization capabilities
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response
	 */
	public function get_capabilities( \WP_REST_Request $request ) {
		$engine = new OptimizationEngine();
		return rest_ensure_response( $engine->get_capabilities() );
	}
	
	/**
	 * Convert image to WebP
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function convert_to_webp( \WP_REST_Request $request ) {
		$attachment_id = $request->get_param( 'attachment_id' );
		
		// Get file path
		$file_path = get_attached_file( $attachment_id );
		if ( ! $file_path || ! file_exists( $file_path ) ) {
			return new \WP_Error(
				'file_not_found',
				__( 'Image file not found.', 'prorank-seo' ),
				[ 'status' => 404 ]
			);
		}
		
		// Check if already WebP
		$mime_type = wp_check_filetype( $file_path )['type'];
		if ( $mime_type === 'image/webp' ) {
			return new \WP_Error(
				'already_webp',
				__( 'Image is already in WebP format.', 'prorank-seo' ),
				[ 'status' => 400 ]
			);
		}
		
		// Convert to WebP
		$engine = new OptimizationEngine();
		$result = $engine->convert_to_webp( $attachment_id, $file_path );
		
		if ( $result['success'] ) {
			return rest_ensure_response( $result );
		}
		
		return new \WP_Error(
			'conversion_failed',
			$result['error'] ?? __( 'Failed to convert image to WebP.', 'prorank-seo' ),
			[ 'status' => 500 ]
		);
	}
	
	/**
	 * Get unoptimized images
	 *
	 * @param \WP_REST_Request $request Request object.
	 * @return \WP_REST_Response
	 */
	public function get_unoptimized_images( \WP_REST_Request $request ) {
		// Get all image attachments that are not optimized
		$args = [
			'post_type'      => 'attachment',
			'post_mime_type' => 'image',
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'meta_query'     => [
				'relation' => 'OR',
				[
					'key'     => '_prorank_optimized',
					'compare' => 'NOT EXISTS',
				],
				[
					'key'     => '_prorank_optimized',
					'value'   => '1',
					'compare' => '!=',
				],
			],
		];
		
		$query = new \WP_Query( $args );
		$attachment_ids = $query->posts;
		
		return rest_ensure_response( [
			'attachment_ids' => $attachment_ids,
			'count' => count( $attachment_ids ),
		] );
	}
}