<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Image Optimization REST API Controller
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * ImageOptimizationController class
 */
class ImageOptimizationController extends \WP_REST_Controller {
    
    /**
     * Module instance
     *
     * @var object|null
     */
    private $module = null;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->namespace = 'prorank-seo/v1';
        $this->rest_base = 'image-optimization';
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Get optimization statistics
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/stats',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_stats'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Start bulk conversion
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/bulk-convert',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'bulk_convert'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                ],
            ]
        );
        
        // Get conversion progress
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/progress',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_progress'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Stop optimization
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/stop',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'stop_optimization'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                ],
            ]
        );

        // Delete all backups
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/delete-backups',
            [
                [
                    'methods'             => WP_REST_Server::DELETABLE,
                    'callback'            => [$this, 'delete_backups'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                ],
            ]
        );

        // Restore all backups
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/restore-backups',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'restore_backups'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                ],
            ]
        );
        
        // Get image sizes
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/sizes',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_sizes'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Check server support for image optimization
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/check-support',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'check_support'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Get quota status
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/quota',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_quota'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Check API quota (force refresh)
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/check-quota',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'check_quota'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                ],
            ]
        );

        // Single image optimization
        register_rest_route(
            $this->namespace,
            '/images/(?P<id>\d+)/optimize',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'optimize_single_image'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args'                => [
                        'id' => [
                            'description' => __('Attachment ID', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                        ],
                    ],
                ],
            ]
        );

        // Get single image optimization stats
        register_rest_route(
            $this->namespace,
            '/images/(?P<id>\d+)/optimization-stats',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_single_image_stats'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                    'args'                => [
                        'id' => [
                            'description' => __('Attachment ID', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                        ],
                    ],
                ],
            ]
        );

        // Restore single image
        register_rest_route(
            $this->namespace,
            '/images/(?P<id>\d+)/restore',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'restore_single_image'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args'                => [
                        'id' => [
                            'description' => __('Attachment ID', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                        ],
                    ],
                ],
            ]
        );

        // Get unoptimized images list
        register_rest_route(
            $this->namespace,
            '/images/unoptimized',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_unoptimized_images'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );

        // Bulk optimize endpoint (single image at a time via POST)
        register_rest_route(
            $this->namespace,
            '/images/optimize',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'optimize_image_by_id'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args'                => [
                        'attachment_id' => [
                            'description' => __('Attachment ID', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                        ],
                    ],
                ],
            ]
        );
    }
    
    /**
     * Get module instance
     *
     * @return object|null
     */
    private function get_module() {
        if ($this->module !== null) {
            return $this->module;
        }

        if (!class_exists('\\ProRank\\SEO\\Plugin')) {
            return null;
        }

        $plugin = \ProRank\SEO\Plugin::get_instance();
        if (!$plugin || !$plugin->is_initialized()) {
            return null;
        }

        $module_manager = $plugin->modules();
        if (!$module_manager) {
            return null;
        }

        $module = $module_manager->get_module('image_optimization');
        if (!is_object($module)) {
            return null;
        }

        foreach (['is_enabled', 'get_stats', 'get_quota_status'] as $method) {
            if (!method_exists($module, $method)) {
                return null;
            }
        }

        $this->module = $module;

        return $this->module;
    }
    
    /**
     * Check permissions for reading
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error
     */
    public function get_items_permissions_check($request) {
        if (!current_user_can('manage_options') && !current_user_can('upload_files')) {
            return new WP_Error(
                'rest_forbidden',
                __('Sorry, you are not allowed to view image optimization data.', 'prorank-seo'),
                ['status' => 403]
            );
        }
        
        return true;
    }
    
    /**
     * Check permissions for creating
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error
     */
    public function create_item_permissions_check($request) {
        if (!current_user_can('manage_options') && !current_user_can('upload_files')) {
            return new WP_Error(
                'rest_forbidden',
                __('Sorry, you are not allowed to manage image optimization.', 'prorank-seo'),
                ['status' => 403]
            );
        }

        // Image optimization is a FREE feature - no license check required
        return true;
    }
    
    /**
     * Get optimization statistics
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_stats(WP_REST_Request $request) {
        $module = $this->get_module();

        // Always return basic stats even if module is disabled
        if (!$module || !$module->is_enabled()) {
            global $wpdb;

            // Get basic counts directly from database
            $like_image = $wpdb->esc_like('image/') . '%';
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $total_count = (int) $wpdb->get_var(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->posts}
                     WHERE post_type = 'attachment'
                     AND post_mime_type LIKE %s",
                    $like_image
                )
            );

            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $optimized_count = (int) $wpdb->get_var(
                "SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta}
                 WHERE meta_key = '_prorank_optimized'"
            );

            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $bytes_saved = (int) $wpdb->get_var(
                "SELECT SUM(meta_value) FROM {$wpdb->postmeta}
                 WHERE meta_key = '_prorank_bytes_saved'"
            ) ?: 0;

            return rest_ensure_response([
                'success' => true,
                'module_enabled' => false,
                'stats' => [
                    'total' => $total_count,
                    'optimized' => $optimized_count,
                    'bytes_saved' => $bytes_saved,
                    'optimization_percentage' => $total_count > 0
                        ? round(($optimized_count / $total_count) * 100, 2)
                        : 0,
                ],
            ]);
        }

        $stats = $module->get_stats();
        
        // Add human-readable sizes
        if (isset($stats['total_savings'])) {
            $stats['total_savings_formatted'] = size_format($stats['total_savings']);
        }
        
        return rest_ensure_response([
            'success' => true,
            'stats' => $stats,
        ]);
    }
    
    /**
     * Start bulk conversion - IMPROVED VERSION
     *
     * This version schedules ALL unoptimized images at once using batch processing,
     * stores progress in database (not transients), and supports auto-continuation.
     *
     * FREE VERSION: Bulk optimization is LOCAL ONLY to prevent cloud cost spikes.
     * Cloud credits are reserved for single image optimization.
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function bulk_convert(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        if (!$module->is_enabled()) {
            return new WP_Error(
                'module_disabled',
                __('Image optimization module is disabled.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        // Check if there's an active job
        $active_job = $this->get_active_bulk_job();
        if ($active_job && $active_job['status'] === 'running') {
            // Check if job is stale (no update in 10 minutes)
            $last_update = $active_job['updated_at'] ?? $active_job['started_at'] ?? 0;
            if (time() - $last_update < 600) {
                return new WP_Error(
                    'already_running',
                    __('Bulk optimization is already running. Please wait or stop the current job first.', 'prorank-seo'),
                    ['status' => 409]
                );
            }
            // Job is stale, mark it as failed
            $this->update_bulk_job($active_job['job_id'], ['status' => 'failed', 'error' => 'Job timed out']);
        }

        // FREE VERSION: Force local-only bulk optimization
        // This prevents cloud cost spikes and abuse. Cloud credits are for single image ops.
        $this->force_local_bulk_optimization_settings();

        // Count ALL unoptimized images using optimized query
        $total_count = $this->count_unoptimized_images();

        if ($total_count === 0) {
            return rest_ensure_response([
                'success' => true,
                'complete' => true,
                'message' => __('All images have been optimized', 'prorank-seo'),
            ]);
        }

        // Create new job with unique ID
        $job_id = wp_generate_uuid4();
        $batch_size = apply_filters('prorank_bulk_optimization_batch_size', 50);

        $job_data = [
            'job_id' => $job_id,
            'status' => 'running',
            'total' => $total_count,
            'processed' => 0,
            'failed' => 0,
            'skipped' => 0,
            'bytes_saved' => 0,
            'started_at' => time(),
            'updated_at' => time(),
            'batch_size' => $batch_size,
            'current_batch' => 0,
            'current_image' => '',
            'log' => [],
            'failed_images' => [],
            'error' => null,
        ];

        // Store job in database (persistent, not transient!)
        update_option('prorank_bulk_job_' . $job_id, $job_data, false);
        update_option('prorank_bulk_job_active', $job_id, false);

        // Clear last completed job to avoid stale status display
        delete_option('prorank_bulk_job_last');

        // Also update transient for backward compatibility with UI polling
        $this->sync_job_to_transient($job_data);

        // Schedule the first batch of images
        $scheduled = $this->schedule_image_batch($job_id, 0, $batch_size);

        // Trigger Action Scheduler immediately
        $this->trigger_action_scheduler();

        return rest_ensure_response([
            'success' => true,
            'complete' => false,
            'job_id' => $job_id,
            'total' => $total_count,
            'scheduled' => $scheduled,
            'batch_size' => $batch_size,
            'message' => sprintf(
                /* translators: %1$d: scheduled images, %2$d: total images */
                __('Started bulk optimization: %1$d of %2$d images scheduled in first batch', 'prorank-seo'),
                $scheduled,
                $total_count
            ),
        ]);
    }

    /**
     * Count unoptimized images using optimized direct SQL query
     *
     * @return int
     */
    private function count_unoptimized_images(): int {
        global $wpdb;

        $like_image = $wpdb->esc_like('image/') . '%';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $count = $wpdb->get_var(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT COUNT(p.ID)
                 // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                 FROM {$wpdb->posts} p
                 WHERE p.post_type = 'attachment'
                 AND p.post_mime_type LIKE %s
                 AND p.post_status = 'inherit'
                 AND NOT EXISTS (
                     // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                     SELECT 1 FROM {$wpdb->postmeta} pm
                     WHERE pm.post_id = p.ID
                     AND pm.meta_key = '_prorank_optimized'
                     AND pm.meta_value = '1'
                 )",
                $like_image
            )
        );

        return (int) $count;
    }

    /**
     * Get unoptimized image IDs using optimized query with pagination
     *
     * @param int $limit  Number of images to get
     * @param int $offset Offset for pagination
     * @return array Array of attachment IDs
     */
    private function get_unoptimized_image_ids(int $limit, int $offset = 0): array {
        global $wpdb;

        // Prioritize smaller images first for faster perceived progress
        $like_image = $wpdb->esc_like('image/') . '%';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $results = $wpdb->get_col($wpdb->prepare(
            "SELECT p.ID
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm_meta ON p.ID = pm_meta.post_id
                 AND pm_meta.meta_key = '_wp_attachment_metadata'
             WHERE p.post_type = 'attachment'
             AND p.post_mime_type LIKE %s
             AND p.post_status = 'inherit'
             AND NOT EXISTS (
                 // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                 SELECT 1 FROM {$wpdb->postmeta} pm
                 WHERE pm.post_id = p.ID
                 AND pm.meta_key = '_prorank_optimized'
                 AND pm.meta_value = '1'
             )
             ORDER BY p.ID ASC
             LIMIT %d OFFSET %d",
            $like_image,
            $limit,
            $offset
        ));

        return array_map('intval', $results);
    }

    /**
     * Schedule a batch of images for optimization
     *
     * @param string $job_id    Job ID
     * @param int    $offset    Offset for getting images
     * @param int    $batch_size Batch size
     * @return int Number of images scheduled
     */
    private function schedule_image_batch(string $job_id, int $offset, int $batch_size): int {
        $image_ids = $this->get_unoptimized_image_ids($batch_size, $offset);

        if (empty($image_ids)) {
            return 0;
        }

        // Check if Action Scheduler is available
        if (function_exists('as_enqueue_async_action')) {
            foreach ($image_ids as $attachment_id) {
                // Schedule using the original prorank_optimize_image action
                as_enqueue_async_action(
                    'prorank_optimize_image',
                    ['attachment_id' => $attachment_id],
                    'prorank-image-optimization'
                );
            }

            // Schedule continuation action to process next batch
            as_schedule_single_action(
                time() + 5,
                'prorank_bulk_optimization_continue',
                ['job_id' => $job_id, 'next_offset' => $offset + $batch_size],
                'prorank-image-optimization'
            );
        } else {
            // Fallback: Use WP-Cron for continuation
            wp_schedule_single_event(
                time() + 10,
                'prorank_bulk_optimization_batch',
                [$job_id, $offset]
            );
        }

        return count($image_ids);
    }

    /**
     * Continue bulk optimization by scheduling the next batch.
     *
     * @param string $job_id Job ID.
     * @param int    $next_offset Next offset.
     * @return void
     */
    public function handle_bulk_optimization_continue(string $job_id = '', int $next_offset = 0): void {
        if ($job_id === '') {
            return;
        }

        $job_data = get_option('prorank_bulk_job_' . $job_id, []);
        if (!is_array($job_data) || empty($job_data)) {
            return;
        }

        $status = $job_data['status'] ?? 'running';
        if (in_array($status, ['completed', 'stopped', 'failed'], true)) {
            return;
        }

        $batch_size = isset($job_data['batch_size'])
            ? (int) $job_data['batch_size']
            : (int) apply_filters('prorank_bulk_optimization_batch_size', 50);

        $scheduled = $this->schedule_image_batch($job_id, $next_offset, $batch_size);

        if ($scheduled <= 0) {
            $this->update_bulk_job($job_id, [
                'status' => 'completed',
                'completed_at' => time(),
            ]);
            update_option('prorank_bulk_job_last', $job_id, false);
            delete_option('prorank_bulk_job_active');
        } else {
            $this->update_bulk_job($job_id, [
                'status' => 'running',
                'current_batch' => (int) ($job_data['current_batch'] ?? 0) + 1,
            ]);
        }

        $this->trigger_action_scheduler();
    }

    /**
     * Get active bulk job
     *
     * @return array|null Job data or null
     */
    private function get_active_bulk_job(): ?array {
        // First check for active running job
        $active_job_id = get_option('prorank_bulk_job_active', '');
        if (!empty($active_job_id)) {
            $job_data = get_option('prorank_bulk_job_' . $active_job_id, null);
            if (is_array($job_data)) {
                return $job_data;
            }
        }

        // Then check for recently completed job (to show final status)
        $last_job_id = get_option('prorank_bulk_job_last', '');
        if (!empty($last_job_id)) {
            $job_data = get_option('prorank_bulk_job_' . $last_job_id, null);
            if (is_array($job_data) && in_array($job_data['status'] ?? '', ['completed', 'stopped', 'failed'], true)) {
                // Only return completed jobs from last 5 minutes
                $completed_at = $job_data['completed_at'] ?? $job_data['updated_at'] ?? 0;
                if (time() - $completed_at < 300) {
                    return $job_data;
                }
            }
        }

        return null;
    }

    /**
     * Update bulk job data
     *
     * @param string $job_id Job ID
     * @param array  $update Data to update
     * @return bool
     */
    private function update_bulk_job(string $job_id, array $update): bool {
        $job_data = get_option('prorank_bulk_job_' . $job_id, []);
        if (empty($job_data)) {
            return false;
        }

        $job_data = array_merge($job_data, $update);
        $job_data['updated_at'] = time();

        // Update in database
        update_option('prorank_bulk_job_' . $job_id, $job_data, false);

        // Sync to transient for backward compatibility
        $this->sync_job_to_transient($job_data);

        return true;
    }

    /**
     * Public wrapper to update bulk job data and sync progress.
     *
     * @param string $job_id Job ID.
     * @param array  $update Data to update.
     * @return void
     */
    public function record_bulk_job_update(string $job_id, array $update): void {
        if ($job_id === '') {
            return;
        }

        $this->update_bulk_job($job_id, $update);
    }

    /**
     * Sync job data to transient for backward compatibility with UI
     *
     * @param array $job_data Job data
     * @return void
     */
    private function sync_job_to_transient(array $job_data): void {
        $progress = [
            'current' => $job_data['processed'] ?? 0,
            'total' => $job_data['total'] ?? 0,
            'current_image' => $job_data['current_image'] ?? '',
            'bytes_saved' => $job_data['bytes_saved'] ?? 0,
            'completed' => ($job_data['status'] ?? '') === 'completed',
            'log' => $job_data['log'] ?? [],
            'percent' => ($job_data['total'] ?? 0) > 0
                ? round((($job_data['processed'] ?? 0) / $job_data['total']) * 100, 2)
                : 0,
            'updated_at' => $job_data['updated_at'] ?? time(),
            'active_index' => ($job_data['processed'] ?? 0) + 1,
            'summary' => [
                'processed' => $job_data['processed'] ?? 0,
                'saved_bytes' => $job_data['bytes_saved'] ?? 0,
                'failed' => $job_data['failed'] ?? 0,
                'skipped' => $job_data['skipped'] ?? 0,
            ],
            'job_id' => $job_data['job_id'] ?? '',
            'status' => $job_data['status'] ?? 'idle',
            'error' => $job_data['error'] ?? null,
        ];

        set_transient('prorank_image_optimization_progress', $progress, 2 * HOUR_IN_SECONDS);

        $is_running = in_array($job_data['status'] ?? '', ['running', 'processing'], true);
        if ($is_running) {
            set_transient('prorank_image_conversion_running', true, 2 * HOUR_IN_SECONDS);
        } else {
            delete_transient('prorank_image_conversion_running');
        }
    }

    /**
     * Trigger Action Scheduler to run immediately
     *
     * @return void
     */
    private function trigger_action_scheduler(): void {
        // Spawn cron
        if (function_exists('spawn_cron')) {
            spawn_cron();
        }

        // Try to run Action Scheduler directly
        if (class_exists('\ActionScheduler_QueueRunner')) {
            try {
                \ActionScheduler_QueueRunner::instance()->run();
            } catch (\Exception $e) {
                // Ignore - cron will handle it
            }
        }
    }

    /**
     * Force local-only optimization for bulk runs and store previous method.
     *
     * @return void
     */
    private function force_local_bulk_optimization_settings(): void {
        $settings = get_option('prorank_image_optimization_settings', []);
        $current_method = isset($settings['optimization_method']) ? (string) $settings['optimization_method'] : 'local';
        $previous_method = get_option('prorank_bulk_prev_optimization_method', '');

        if ($current_method !== 'local' && $previous_method === '') {
            update_option('prorank_bulk_prev_optimization_method', $current_method, false);
        }

        $settings['optimization_method'] = 'local';
        update_option('prorank_image_optimization_settings', $settings, false);
    }

    /**
     * Restore the optimization method after bulk completes/stops.
     *
     * @return void
     */
    private function restore_bulk_optimization_settings(): void {
        $previous_method = get_option('prorank_bulk_prev_optimization_method', '');
        if ($previous_method === '') {
            return;
        }

        $settings = get_option('prorank_image_optimization_settings', []);
        $current_method = isset($settings['optimization_method']) ? (string) $settings['optimization_method'] : 'local';

        if ($current_method === 'local') {
            $settings['optimization_method'] = $previous_method;
            update_option('prorank_image_optimization_settings', $settings, false);
        }

        delete_option('prorank_bulk_prev_optimization_method');
    }
    
    /**
     * Get conversion progress - IMPROVED VERSION
     *
     * Reads from persistent database storage and falls back to transients
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_progress(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Get stats for overall progress
        $stats = $module->get_stats();

        // First try to get active job from database (persistent)
        $active_job = $this->get_active_bulk_job();

        // Fall back to transient for backward compatibility
        $progress_data = get_transient('prorank_image_optimization_progress');
        $running = get_transient('prorank_image_conversion_running');

        // Determine status from job or transient
        $status = 'idle';
        if ($active_job) {
            $status = $active_job['status'] ?? 'idle';
            // Check for stale job (no update in 10 minutes)
            $last_update = $active_job['updated_at'] ?? 0;
            if ($status === 'running' && time() - $last_update > 600) {
                $status = 'stale';
            }
        } elseif ($running) {
            $status = 'running';
        } elseif ($progress_data && isset($progress_data['completed']) && $progress_data['completed']) {
            $status = 'completed';
        } elseif ($progress_data && isset($progress_data['error']) && $progress_data['error']) {
            $status = 'error';
        }

        $active_job_id = get_option('prorank_bulk_job_active', '');
        $job_is_active = !empty($active_job_id)
            && $active_job
            && ($active_job['job_id'] ?? '') === $active_job_id;

        if (!$job_is_active && !$running && in_array($status, ['completed', 'stopped', 'failed', 'error', 'stale', 'idle'], true)) {
            $this->restore_bulk_optimization_settings();
        }

        // Build progress response - prefer job data over transient
        $current = $active_job['processed'] ?? $progress_data['current'] ?? 0;
        $total = $active_job['total'] ?? $progress_data['total'] ?? $stats['total'] ?? 0;
        $bytes_saved = $active_job['bytes_saved'] ?? $progress_data['bytes_saved'] ?? $stats['bytes_saved'] ?? 0;

        $percent = $total > 0 ? round(($current / $total) * 100, 2) : 0;

        // Calculate estimated time remaining
        $eta = null;
        if ($active_job && $status === 'running') {
            $elapsed = time() - ($active_job['started_at'] ?? time());
            $processed = $active_job['processed'] ?? 0;
            if ($processed > 0 && $elapsed > 0) {
                $avg_time_per_image = $elapsed / $processed;
                $remaining_images = $total - $processed;
                $eta = (int) ($avg_time_per_image * $remaining_images);
            }
        }

        $progress = [
            'current' => $current,
            'total' => $total,
            'current_image' => $active_job['current_image'] ?? $progress_data['current_image'] ?? '',
            'bytes_saved' => $bytes_saved,
            'status' => $status,
            'log' => $active_job['log'] ?? (isset($progress_data['log']) && is_array($progress_data['log'])
                ? array_values($progress_data['log'])
                : []),
            'percent' => $percent,
            'active_index' => $current + 1,
            'summary' => [
                'processed' => $current,
                'saved_bytes' => $bytes_saved,
                'failed' => $active_job['failed'] ?? $progress_data['summary']['failed'] ?? 0,
                'skipped' => $active_job['skipped'] ?? $progress_data['summary']['skipped'] ?? 0,
            ],
            'job_id' => $active_job['job_id'] ?? $progress_data['job_id'] ?? null,
            'eta_seconds' => $eta,
            'started_at' => $active_job['started_at'] ?? null,
        ];

        if ($status === 'error') {
            $progress['error_message'] = $active_job['error'] ?? $progress_data['error_message'] ?? __('An error occurred', 'prorank-seo');
        }

        // Include failed images list if any
        if (!empty($active_job['failed_images'])) {
            $progress['failed_images'] = array_slice($active_job['failed_images'], -20); // Last 20 failures
        }

        return rest_ensure_response([
            'success' => true,
            'progress' => $progress,
            // Keep legacy format for backwards compatibility
            'running' => in_array($status, ['running', 'processing'], true),
            'optimized' => $stats['optimized'] ?? 0,
            'total' => $stats['total'] ?? 0,
            'remaining' => ($stats['total'] ?? 0) - ($stats['optimized'] ?? 0),
        ]);
    }

    /**
     * Stop the current optimization process - IMPROVED VERSION
     *
     * Stops the active job and cancels all pending scheduled actions
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function stop_optimization(WP_REST_Request $request) {
        // Get active job
        $active_job = $this->get_active_bulk_job();

        if ($active_job) {
            $job_id = $active_job['job_id'];

            // Update job status to stopped
            $this->update_bulk_job($job_id, [
                'status' => 'stopped',
                'stopped_at' => time(),
            ]);

            // Cancel all pending actions for this job using Action Scheduler
            if (function_exists('as_unschedule_all_actions')) {
                // Cancel individual image optimization actions
                as_unschedule_all_actions('prorank_optimize_image', null, 'prorank-image-optimization');
                // Cancel continuation actions
                as_unschedule_all_actions('prorank_bulk_optimization_continue', null, 'prorank-image-optimization');
            }

            // Clear active job marker
            delete_option('prorank_bulk_job_active');
        }

        // Clear transients for backward compatibility
        delete_transient('prorank_image_conversion_running');

        $progress_data = get_transient('prorank_image_optimization_progress');
        if ($progress_data) {
            $progress_data['status'] = 'stopped';
            $progress_data['completed'] = false;
            $progress_data['stopped'] = true;
            set_transient('prorank_image_optimization_progress', $progress_data, HOUR_IN_SECONDS);
        }

        // Clear any WP-Cron scheduled tasks
        $timestamp = wp_next_scheduled('prorank_image_optimization_batch');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'prorank_image_optimization_batch');
        }
        $timestamp = wp_next_scheduled('prorank_bulk_optimize_images');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'prorank_bulk_optimize_images');
        }

        $this->restore_bulk_optimization_settings();

        return rest_ensure_response([
            'success' => true,
            'message' => __('Optimization stopped successfully. Pending images have been cancelled.', 'prorank-seo'),
            'stopped_job' => $active_job['job_id'] ?? null,
            'images_processed' => $active_job['processed'] ?? 0,
        ]);
    }

    /**
     * Restore all backups
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function restore_backups(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        if (!method_exists($module, 'restore_all_backups')) {
            return new WP_Error(
                'not_supported',
                __('Backup restoration is not available.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        $result = $module->restore_all_backups();

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response([
            'success' => true,
            'restored' => $result['restored'] ?? 0,
            'failed' => $result['failed'] ?? 0,
            'message' => sprintf(
                /* translators: %d: number of images restored */
                __('Restored %d images from backups', 'prorank-seo'),
                $result['restored'] ?? 0
            ),
        ]);
    }

    /**
     * Delete all backup files
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_backups(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        if (!$module->is_enabled()) {
            return new WP_Error(
                'module_disabled',
                __('Image optimization module is disabled.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $success = $module->delete_all_backups();
        
        if (!$success) {
            return new WP_Error(
                'delete_failed',
                __('Failed to delete backup files.', 'prorank-seo'),
                ['status' => 500]
            );
        }
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('All backup files have been deleted.', 'prorank-seo'),
        ]);
    }
    
    /**
     * Get registered image sizes
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_sizes(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            // Return default sizes even if module not found
            return rest_ensure_response([
                'success' => true,
                'sizes' => [
                    ['name' => 'thumbnail', 'label' => 'Thumbnail', 'width' => 150, 'height' => 150],
                    ['name' => 'medium', 'label' => 'Medium', 'width' => 300, 'height' => 300],
                    ['name' => 'medium_large', 'label' => 'Medium Large', 'width' => 768, 'height' => 0],
                    ['name' => 'large', 'label' => 'Large', 'width' => 1024, 'height' => 1024],
                ],
            ]);
        }
        
        $sizes = $module->get_image_sizes();
        
        return rest_ensure_response([
            'success' => true,
            'sizes' => $sizes,
        ]);
    }
    
    /**
     * Check server support for image optimization
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function check_support(WP_REST_Request $request) {
        $support = [
            'gd' => false,
            'imagick' => false,
            'webp' => false,
            'avif' => false,
            'details' => [],
        ];
        
        try {
            // Check GD
            if (function_exists('gd_info')) {
                $gd_info = gd_info();
                $support['gd'] = true;
                $support['details']['gd_version'] = $gd_info['GD Version'] ?? 'Unknown';
                
                // Check WebP support in GD
                if (function_exists('imagewebp')) {
                    $support['webp'] = true;
                }
                
                // Check AVIF support in GD (PHP 8.1+)
                if (function_exists('imageavif')) {
                    $support['avif'] = true;
                }
            }
            
            // Check Imagick
            if (extension_loaded('imagick') && class_exists('Imagick')) {
                try {
                    $imagick = new \Imagick();
                    $support['imagick'] = true;
                    $support['details']['imagick_version'] = \Imagick::getVersion()['versionString'] ?? 'Unknown';
                    
                    // Check WebP support in Imagick
                    $formats = $imagick->queryFormats('WEBP');
                    if (!empty($formats)) {
                        $support['webp'] = true;
                    }
                    
                    // Check AVIF support in Imagick
                    $formats = $imagick->queryFormats('AVIF');
                    if (!empty($formats)) {
                        $support['avif'] = true;
                    }
                } catch (\Exception $e) {
                    $support['details']['imagick_error'] = esc_html($e->getMessage());
                }
            }
            
            // Add server info
            $server_software = \prorank_get_server_var( 'SERVER_SOFTWARE' );
            $support['server'] = [
                'software' => $server_software !== '' ? $server_software : 'Unknown',
                'php_version' => PHP_VERSION,
                'memory_limit' => ini_get('memory_limit'),
                'max_execution_time' => ini_get('max_execution_time'),
                'upload_max_filesize' => ini_get('upload_max_filesize'),
            ];
            
            // Check if .htaccess is supported
            $support['htaccess'] = false;
            if ($server_software !== '' && stripos($server_software, 'apache') !== false) {
                $support['htaccess'] = true;
                if (function_exists('apache_get_modules')) {
                    $modules = apache_get_modules();
                    $support['htaccess'] = in_array('mod_rewrite', $modules, true);
                }
            }

            // Add JXL support detection (not native in PHP yet)
            $support['jxl'] = false;

            // Add Jpegli support detection
            $support['jpegli'] = false;

            // Read cloud_opt_in from image optimization settings
            $image_settings = get_option('prorank_image_optimization_settings', []);
            $cloud_opt_in = !empty($image_settings['cloud_opt_in']);

            // Get license key - check multiple possible sources
            $license_key = get_option('prorank_seo_license_key', '');
            if (empty($license_key)) {
                $license_key = get_option('prorank_license_key', '');
            }

            // Also try via LicenseManager if still empty
            if (empty($license_key) && class_exists('\\ProRank\\SEO\\Plugin')) {
                try {
                    $plugin = \ProRank\SEO\Plugin::get_instance();
                    if ($plugin && method_exists($plugin, 'license')) {
                        $license = $plugin->license();
                        if ($license && method_exists($license, 'get_license_key')) {
                            $license_key = (string) $license->get_license_key();
                        }
                    }
                } catch (\Throwable $e) {
                    $support['details']['license_error'] = esc_html($e->getMessage());
                }
            }

            $cloud_allowed = $cloud_opt_in && !empty($license_key);

            if ($cloud_allowed) {
                // Check ProRank Cloud optimizer status
                $health_endpoint = defined('PRORANK_IMAGE_OPTIMIZER_HEALTH')
                    ? PRORANK_IMAGE_OPTIMIZER_HEALTH
                    : 'http://88.198.201.243:3002/health';

                $cloud_response = wp_remote_get($health_endpoint, ['timeout' => 5]);

                if (!is_wp_error($cloud_response) && wp_remote_retrieve_response_code($cloud_response) === 200) {
                    $cloud_data = json_decode(wp_remote_retrieve_body($cloud_response), true);
                    $support['cloud'] = [
                        'available' => true,
                        'opt_in_required' => true,
                        'license_required' => true,
                        'formats' => ['webp', 'avif', 'jxl'],
                        'version' => $cloud_data['version'] ?? null,
                        'capabilities' => [
                            'webp' => true,  // Cloud always supports WebP
                            'avif' => true,  // Cloud always supports AVIF
                            'jxl' => $cloud_data['capabilities']['jxl'] ?? true,
                            'jpegli' => $cloud_data['capabilities']['jpegli'] ?? true,
                            'png_pipeline' => $cloud_data['capabilities']['png_pipeline'] ?? false,
                        ],
                    ];

                    if (isset($cloud_data['capabilities'])) {
                        $support['jxl'] = $cloud_data['capabilities']['jxl'] ?? false;
                        $support['jpegli'] = $cloud_data['capabilities']['jpegli'] ?? false;
                        $support['png_pipeline'] = $cloud_data['capabilities']['png_pipeline'] ?? false;
                    }
                } else {
                    $support['cloud'] = [
                        'available' => false,
                        'opt_in_required' => true,
                        'license_required' => true,
                        'formats' => [],
                        'capabilities' => [
                            'webp' => false,
                            'avif' => false,
                            'jxl' => false,
                            'jpegli' => false,
                            'png_pipeline' => false,
                        ],
                    ];
                }
            } else {
                $support['cloud'] = [
                    'available' => false,
                    'opt_in_required' => true,
                    'license_required' => true,
                    'formats' => [],
                    'capabilities' => [
                        'webp' => false,
                        'avif' => false,
                        'jxl' => false,
                        'jpegli' => false,
                        'png_pipeline' => false,
                    ],
                ];
            }

            return rest_ensure_response([
                'success' => true,
                'support' => $support,
            ]);
        } catch (\Throwable $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank SEO] Image support check failed: ' . $e->getMessage());
            }
            return rest_ensure_response([
                'success' => false,
                'support' => $support,
                'message' => __('Unable to detect server image capabilities.', 'prorank-seo'),
            ]);
        }
    }
    
    /**
     * Get quota status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_quota($request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        $quota_status = $module->get_quota_status();

        // Add formatted numbers for display
        if (isset($quota_status['quota'])) {
            $quota_status['quota_formatted'] = [
                'yearly' => isset($quota_status['quota']['yearly'])
                    ? $this->format_number($module, $quota_status['quota']['yearly'])
                    : $this->format_number($module, 0),
            ];

            if (isset($quota_status['quota']['monthly'])) {
                $quota_status['quota_formatted']['monthly'] = $this->format_number($module, $quota_status['quota']['monthly']);
            }
        }

        if (isset($quota_status['used'])) {
            $quota_status['used_formatted'] = $this->format_number($module, $quota_status['used']);
        }

        if (isset($quota_status['remaining'])) {
            $quota_status['remaining_formatted'] = $this->format_number($module, $quota_status['remaining']);
        }

        return rest_ensure_response($quota_status);
    }
    
    /**
     * Check API quota (force refresh from API)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function check_quota($request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        // Try to get fresh data from API
        $api_data = $module->check_api_quota();

        if ($api_data === false) {
            // Fall back to cached data
            $api_data = $module->get_quota_status();
        }

        return rest_ensure_response($api_data);
    }

    /**
     * Optimize a single image
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function optimize_single_image(WP_REST_Request $request) {
        $attachment_id = (int) $request->get_param('id');

        if (!wp_attachment_is_image($attachment_id)) {
            return new WP_Error(
                'not_an_image',
                __('The specified attachment is not an image.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!$metadata) {
            return new WP_Error(
                'no_metadata',
                __('Unable to get attachment metadata.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        $result = $module->generate_optimized_images($metadata, $attachment_id);

        if (is_wp_error($result)) {
            return $result;
        }

        // Get updated stats
        $stats = $this->get_attachment_optimization_stats($attachment_id);

        return rest_ensure_response([
            'success' => true,
            'message' => __('Image optimized successfully.', 'prorank-seo'),
            'stats'   => $stats,
            'saved_percent' => $stats['percent_saved'] ?? 0,
            'saved_size'    => $stats['bytes_saved_formatted'] ?? '0 KB',
            'webp_generated' => $stats['webp_generated'] ?? false,
            'avif_generated' => $stats['avif_generated'] ?? false,
        ]);
    }

    /**
     * Optimize image by attachment ID (POST body parameter)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function optimize_image_by_id(WP_REST_Request $request) {
        $attachment_id = (int) $request->get_param('attachment_id');

        if (!$attachment_id) {
            return new WP_Error(
                'missing_id',
                __('Attachment ID is required.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        // Reuse the optimize_single_image logic
        $request->set_param('id', $attachment_id);
        return $this->optimize_single_image($request);
    }

    /**
     * Get single image optimization stats
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_single_image_stats(WP_REST_Request $request) {
        $attachment_id = (int) $request->get_param('id');

        if (!wp_attachment_is_image($attachment_id)) {
            return new WP_Error(
                'not_an_image',
                __('The specified attachment is not an image.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        $stats = $this->get_attachment_optimization_stats($attachment_id);

        return rest_ensure_response([
            'success'   => true,
            'optimized' => $stats['optimized'],
            'stats'     => $stats,
        ]);
    }

    /**
     * Restore single image to original
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function restore_single_image(WP_REST_Request $request) {
        $attachment_id = (int) $request->get_param('id');

        if (!wp_attachment_is_image($attachment_id)) {
            return new WP_Error(
                'not_an_image',
                __('The specified attachment is not an image.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Image optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        $result = $module->restore_original($attachment_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response([
            'success' => true,
            'message' => __('Original image restored successfully.', 'prorank-seo'),
        ]);
    }

    /**
     * Get list of unoptimized images
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_unoptimized_images(WP_REST_Request $request) {
        $args = [
            'post_type'      => 'attachment',
            'post_mime_type' => 'image',
            'post_status'    => 'inherit',
            'posts_per_page' => 100,
            'fields'         => 'ids',
            'meta_query'     => [
                [
                    'key'     => '_prorank_optimized',
                    'compare' => 'NOT EXISTS',
                ],
            ],
        ];

        $attachment_ids = get_posts($args);

        return rest_ensure_response([
            'success'        => true,
            'attachment_ids' => $attachment_ids,
            'count'          => count($attachment_ids),
        ]);
    }

    /**
     * Get optimization stats for a single attachment
     *
     * @param int $attachment_id Attachment ID
     * @return array Stats array
     */
    private function get_attachment_optimization_stats(int $attachment_id): array {
        $optimized = get_post_meta($attachment_id, '_prorank_optimized', true);
        $original_size = (int) get_post_meta($attachment_id, '_prorank_original_size', true);
        $bytes_saved = (int) get_post_meta($attachment_id, '_prorank_bytes_saved', true);
        $webp_generated = get_post_meta($attachment_id, '_prorank_webp_generated', true);
        $avif_generated = get_post_meta($attachment_id, '_prorank_avif_generated', true);
        $backup_path = get_post_meta($attachment_id, '_prorank_backup_path', true);

        $file_path = get_attached_file($attachment_id);
        $current_size = file_exists($file_path) ? filesize($file_path) : 0;

        $percent = ($original_size > 0) ? round(($bytes_saved / $original_size) * 100, 1) : 0;

        return [
            'optimized'              => $optimized === '1',
            'original_size'          => $original_size,
            'current_size'           => $current_size,
            'bytes_saved'            => $bytes_saved,
            'percent_saved'          => $percent,
            'saved_percent'          => $percent, // alias for compatibility
            'webp_generated'         => (bool) $webp_generated,
            'avif_generated'         => (bool) $avif_generated,
            'has_backup'             => $backup_path && file_exists($backup_path),
            'original_size_formatted' => size_format($original_size),
            'current_size_formatted' => size_format($current_size),
            'bytes_saved_formatted'  => size_format($bytes_saved),
        ];
    }

    /**
     * Format numbers using the active Image Optimization module if available.
     *
     * @param object|null $module
     * @param mixed       $number
     * @return string
     */
    private function format_number($module, $number): string {
        if (is_object($module)) {
            $class = get_class($module);
            if (is_callable([$class, 'format_number'])) {
                return $class::format_number($number);
            }
        }

        if (is_numeric($number)) {
            return function_exists('number_format_i18n')
                ? number_format_i18n((float) $number)
                : (string) $number;
        }

        return (string) $number;
    }

}
