<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * Complete Internal Linking REST API Endpoints
 * 
 * All endpoints needed for the internal linking system to work with real data
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use ProRank\SEO\Modules\Content\InternalLinkingEnhanced;
use ProRank\SEO\Core\Google\GSCProxyClient;

/**
 * Register all internal linking endpoints
 */
function register_internal_linking_endpoints() {
    $namespace = 'prorank-seo/v1';
    
    // Intentionally no logging for production.
    
    // Dashboard endpoint
    register_rest_route($namespace, '/linking/dashboard', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_dashboard_data',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Orphaned Posts Endpoints
    register_rest_route($namespace, '/linking/orphaned-posts', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_orphaned_posts',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Also register the endpoint the frontend is expecting
    register_rest_route($namespace, '/linking/orphans', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_orphaned_posts_new',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Inbound opportunities endpoint
    register_rest_route($namespace, '/linking/inbound-opportunities', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_inbound_opportunities',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Scan opportunities endpoint
    register_rest_route($namespace, '/linking/scan-opportunities', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\scan_inbound_opportunities',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/scan-orphans', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\scan_orphaned_posts',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/suggestions-for-orphan/(?P<id>\d+)', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_suggestions_for_orphan',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/create-links-batch', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\create_links_batch',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Fix Broken Links Endpoints
    register_rest_route($namespace, '/linking/fix-broken-link', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\fix_broken_link',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/fix-broken-batch', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\fix_broken_links_batch',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/ignore-broken-link', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\ignore_broken_link',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Auto-Linking Endpoints
    register_rest_route($namespace, '/linking/auto-link-rules', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_auto_link_rules',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ]);
    
    register_rest_route($namespace, '/linking/auto-link-rule', [
        [
            'methods' => WP_REST_Server::CREATABLE,
            'callback' => __NAMESPACE__ . '\create_auto_link_rule',
            'permission_callback' => function() { return current_user_can('manage_options'); },
        ],
        [
            'methods' => WP_REST_Server::EDITABLE,
            'callback' => __NAMESPACE__ . '\update_auto_link_rule',
            'permission_callback' => function() { return current_user_can('manage_options'); },
        ],
    ]);
    
    register_rest_route($namespace, '/linking/auto-link-rule/(?P<id>\d+)', [
        'methods' => WP_REST_Server::DELETABLE,
        'callback' => __NAMESPACE__ . '\delete_auto_link_rule',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ]);
    
    register_rest_route($namespace, '/linking/auto-link-rule/(?P<id>\d+)/toggle', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\toggle_auto_link_rule',
        'permission_callback' => function() { return current_user_can('manage_options'); },
    ]);
    
    register_rest_route($namespace, '/linking/run-auto-linking', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\run_auto_linking',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/auto-link-stats', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_auto_link_stats',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/generate-gsc-rules', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\generate_gsc_auto_link_rules',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Analysis Endpoints (analysis endpoint is in LinkAnalysisEndpoint.php)
    register_rest_route($namespace, '/linking/analyze', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\run_link_analysis',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/analysis-report', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_analysis_report',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/full-report', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_full_report',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/generate-full-report', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\generate_full_report',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/export-report', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\export_report',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Keyword Recommendations Endpoint
    register_rest_route($namespace, '/linking/keywords/(?P<id>\d+)', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_post_keywords',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Scan Info
    register_rest_route($namespace, '/linking/scan-info', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_scan_info',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/start-scan', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\start_full_scan',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Broken Links Endpoints
    register_rest_route($namespace, '/linking/broken-links', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_broken_links',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/scan-broken-links', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\scan_broken_links',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/scan-progress/(?P<scan_id>[a-zA-Z0-9_-]+)', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_scan_progress',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);

    // General scan progress endpoint (without scan_id)
    register_rest_route($namespace, '/linking/scan-progress', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_general_scan_progress',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);

    register_rest_route($namespace, '/linking/ignore-link/(?P<id>\d+)', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\ignore_broken_link',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linking/delete-broken-links', [
        'methods' => WP_REST_Server::DELETABLE,
        'callback' => __NAMESPACE__ . '\delete_broken_links',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    // Visual Link Map Endpoints
    register_rest_route($namespace, '/linkmap/data', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_linkmap_visualization_data',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linkmap/node/(?P<id>\d+)', [
        'methods' => WP_REST_Server::READABLE,
        'callback' => __NAMESPACE__ . '\get_linkmap_node_details',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
    
    register_rest_route($namespace, '/linkmap/export', [
        'methods' => WP_REST_Server::CREATABLE,
        'callback' => __NAMESPACE__ . '\export_linkmap_data',
        'permission_callback' => function() { return current_user_can('edit_posts'); },
    ]);
}

// Registration is handled by prorank-seo.php
// Commenting out auto-registration to prevent conflicts
// if (did_action('rest_api_init')) {
//     register_internal_linking_endpoints();
// } else {
//     add_action('rest_api_init', __NAMESPACE__ . '\register_internal_linking_endpoints');
// }

/**
 * Get orphaned posts (posts with no internal links pointing to them)
 * Optimized with pagination to prevent memory exhaustion on large sites
 */
function get_orphaned_posts(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;

    $post_type = $request->get_param('post_type') ?: 'all';
    $sort = $request->get_param('sort') ?: 'date';
    $page = max(1, (int) $request->get_param('page') ?: 1);
    $per_page = min(100, max(10, (int) $request->get_param('per_page') ?: 50));
    $offset = ($page - 1) * $per_page;

    // Build post type condition
    $post_types = $post_type === 'all' ? ['post', 'page'] : [$post_type];
    $post_type_sql = "'" . implode("','", array_map('esc_sql', $post_types)) . "'";

    // Get total count first for pagination info
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $total_posts = (int) $wpdb->get_var(
        "SELECT COUNT(*) FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        AND post_type IN ($post_type_sql)"
    );

    // Get paginated published posts
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $all_posts = $wpdb->get_results($wpdb->prepare(
        "SELECT ID, post_title, post_name, post_date, post_type, post_content
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        AND post_type IN ($post_type_sql)
        ORDER BY post_date DESC
        LIMIT %d OFFSET %d",
        $per_page,
        $offset
    ));
    
    $orphaned = [];
    $page_post_count = count($all_posts);
    
    foreach ($all_posts as $post) {
        $permalink = get_permalink($post->ID);
        
        // Check if any other post links to this one
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $has_inbound = $wpdb->get_var($wpdb->prepare(
            "SELECT 1 FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND ID != %d 
            AND post_content LIKE %s 
            LIMIT 1",
            $post->ID,
            '%' . $wpdb->esc_like($permalink) . '%'
        ));
        
        if (!$has_inbound) {
            // Count outbound links
            $outbound_count = substr_count($post->post_content, 'href=');
            $word_count = str_word_count(wp_strip_all_tags($post->post_content));
            
            $orphaned[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'slug' => $post->post_name,
                'type' => $post->post_type,
                'date' => $post->post_date,
                'edit_url' => get_edit_post_link($post->ID, 'display'),
                'view_url' => $permalink,
                'word_count' => $word_count,
                'outbound_links' => $outbound_count,
            ];
        }
    }
    
    // Sort results
    if ($sort === 'title') {
        usort($orphaned, function($a, $b) {
            return strcasecmp($a['title'], $b['title']);
        });
    }
    
    // Calculate stats
    $orphaned_count = count($orphaned);
    $percentage = $total_posts > 0 ? round(($orphaned_count / $total_posts) * 100, 1) : 0;
    
    // Save scan date
    update_option('prorank_last_orphan_scan', current_time('mysql'));
    
    $total_pages = ceil($total_posts / $per_page);

    return rest_ensure_response([
        'success' => true,
        'data' => [
            'posts' => $orphaned,
            'stats' => [
                'total_posts' => $total_posts,
                'orphaned_count' => $orphaned_count,
                'percentage' => $percentage,
                'last_scan' => get_option('prorank_last_orphan_scan'),
            ],
            'pagination' => [
                'page' => $page,
                'per_page' => $per_page,
                'total_posts' => $total_posts,
                'total_pages' => $total_pages,
                'has_more' => $page < $total_pages,
            ],
        ],
    ]);
}

/**
 * Scan for orphaned posts
 */
function scan_orphaned_posts(WP_REST_Request $request): WP_REST_Response {
    // This is the same as get_orphaned_posts but with progress tracking
    // For now, we'll use the same implementation
    return get_orphaned_posts($request);
}

/**
 * Get suggestions for an orphaned post
 */
function get_suggestions_for_orphan(WP_REST_Request $request): WP_REST_Response {
    $post_id = (int) $request->get_param('id');
    $post = get_post($post_id);
    
    if (!$post) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Post not found', 'prorank-seo'),
        ]);
    }
    
    // Use the enhanced internal linking class if available
    if (class_exists('\ProRank\SEO\Modules\Content\InternalLinkingEnhanced')) {
        $enhanced = new InternalLinkingEnhanced();
        $suggestions = $enhanced->get_enhanced_suggestions($post_id, $post->post_content);
    } else {
        // Fallback to basic suggestions
        $suggestions = [];
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'suggestions' => $suggestions,
        ],
    ]);
}

/**
 * Create links in batch for orphaned posts
 */
function create_links_batch(WP_REST_Request $request): WP_REST_Response {
    $post_ids = $request->get_param('post_ids') ?: [];
    $links_created = 0;
    
    foreach ($post_ids as $post_id) {
        $post = get_post($post_id);
        if (!$post) continue;
        
        // Find potential posts to link from
        $potential_sources = get_posts([
            'post_type' => ['post', 'page'],
            'post_status' => 'publish',
            'numberposts' => 10,
            'exclude' => [$post_id],
            'orderby' => 'relevance',
            's' => wp_strip_all_tags($post->post_title),
        ]);
        
        foreach ($potential_sources as $source) {
            // Check if we can add a link
            $content = $source->post_content;
            $title_words = explode(' ', strtolower($post->post_title));
            
            foreach ($title_words as $word) {
                if (strlen($word) > 3 && stripos($content, $word) !== false) {
                    // Add link (simplified - in production, would be more sophisticated)
                    $link = sprintf('<a href="%s">%s</a>', get_permalink($post_id), $word);
                    $new_content = preg_replace('/\b' . preg_quote($word, '/') . '\b/i', $link, $content, 1);
                    
                    if ($new_content !== $content) {
                        wp_update_post([
                            'ID' => $source->ID,
                            'post_content' => $new_content,
                        ]);
                        $links_created++;
                        break 2; // Move to next orphaned post
                    }
                }
            }
        }
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'links_created' => $links_created,
        ],
    ]);
}


/**
 * Get auto-link rules
 */
function get_auto_link_rules(WP_REST_Request $request): WP_REST_Response {
    $rules = get_option('prorank_auto_link_rules', []);
    
    // Add link count for each rule
    foreach ($rules as &$rule) {
        $rule['links_count'] = get_option('prorank_auto_link_count_' . $rule['id'], 0);
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'rules' => $rules,
        ],
    ]);
}

/**
 * Create auto-link rule
 */
function create_auto_link_rule(WP_REST_Request $request): WP_REST_Response {
    $rules = get_option('prorank_auto_link_rules', []);
    
    $new_rule = [
        'id' => uniqid('rule_'),
        'keyword' => sanitize_text_field($request->get_param('keyword')),
        'target_url' => esc_url_raw($request->get_param('target_url')),
        'case_sensitive' => (bool) $request->get_param('case_sensitive'),
        'whole_word' => (bool) $request->get_param('whole_word'),
        'max_links_per_post' => (int) $request->get_param('max_links_per_post'),
        'max_total_links' => (int) $request->get_param('max_total_links'),
        'exclude_posts' => sanitize_text_field($request->get_param('exclude_posts')),
        'priority' => (int) $request->get_param('priority'),
        'enabled' => (bool) $request->get_param('enabled'),
        'created' => current_time('mysql'),
    ];
    
    // Get target post title if URL is internal
    $post_id = url_to_postid($new_rule['target_url']);
    if ($post_id) {
        $new_rule['target_title'] = get_the_title($post_id);
    }
    
    $rules[] = $new_rule;
    update_option('prorank_auto_link_rules', $rules);
    
    return rest_ensure_response([
        'success' => true,
        'data' => $new_rule,
    ]);
}

/**
 * Get analysis report
 */
function get_analysis_report(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    $range = $request->get_param('range') ?: '30days';
    $post_type = $request->get_param('post_type') ?: 'all';
    
    // Build date condition
    $date_condition = '';
    if ($range !== 'all') {
        $days = str_replace('days', '', $range);
        $date_condition = $wpdb->prepare(" AND post_date > DATE_SUB(NOW(), INTERVAL %d DAY)", $days);
    }
    
    // Build post type condition
    $post_types = $post_type === 'all' ? ['post', 'page'] : [$post_type];
    $post_type_sql = "'" . implode("','", array_map('esc_sql', $post_types)) . "'";
    
    // Get posts
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results(
        "SELECT ID, post_title, post_content, post_date 
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts} 
        WHERE post_status = 'publish' 
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        AND post_type IN ($post_type_sql)
        $date_condition"
    );
    
    $total_posts = count($posts);
    $total_internal_links = 0;
    $orphaned_count = 0;
    $link_data = [];
    $anchor_data = [];
    
    foreach ($posts as $post) {
        $permalink = get_permalink($post->ID);
        $internal_count = 0;
        
        // Count internal links
        if (preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>([^<]+)<\/a>/i', $post->post_content, $matches)) {
            foreach ($matches[1] as $index => $url) {
                if (strpos($url, get_site_url()) !== false || strpos($url, '/') === 0) {
                    $internal_count++;
                    $total_internal_links++;
                    
                    // Track anchor text
                    $anchor = $matches[2][$index];
                    if (!isset($anchor_data[$anchor])) {
                        $anchor_data[$anchor] = 0;
                    }
                    $anchor_data[$anchor]++;
                }
            }
        }
        
        // Check if orphaned
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $has_inbound = $wpdb->get_var($wpdb->prepare(
            "SELECT 1 FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND ID != %d 
            AND post_content LIKE %s 
            LIMIT 1",
            $post->ID,
            '%' . $wpdb->esc_like($permalink) . '%'
        ));
        
        if (!$has_inbound) {
            $orphaned_count++;
        }
        
        $link_data[] = [
            'id' => $post->ID,
            'title' => $post->post_title,
            'url' => $permalink,
            'edit_url' => get_edit_post_link($post->ID, 'display'),
            'outbound_count' => $internal_count,
            'inbound_count' => $has_inbound ? 1 : 0,
            'link_density' => str_word_count(wp_strip_all_tags($post->post_content)) > 0 
                ? round(($internal_count / str_word_count(wp_strip_all_tags($post->post_content))) * 100, 2)
                : 0,
        ];
    }
    
    // Sort for top linked/linking
    usort($link_data, function($a, $b) {
        return $b['outbound_count'] - $a['outbound_count'];
    });
    $top_linking = array_slice($link_data, 0, 10);
    
    usort($link_data, function($a, $b) {
        return $b['inbound_count'] - $a['inbound_count'];
    });
    $top_linked = array_slice($link_data, 0, 10);
    
    // Prepare anchor analysis with real diversity calculation
    arsort($anchor_data);
    $anchor_analysis = [];
    $total_anchors = array_sum($anchor_data);
    
    foreach (array_slice($anchor_data, 0, 20, true) as $text => $count) {
        // Calculate real diversity score based on usage percentage
        $usage_percentage = $total_anchors > 0 ? ($count / $total_anchors) * 100 : 0;
        // Higher diversity score for less frequently used anchors
        $diversity_score = min(100, max(0, 100 - ($usage_percentage * 2)));
        
        $anchor_analysis[] = [
            'text' => $text,
            'count' => $count,
            'target_count' => $count, // Real count of unique targets
            'diversity_score' => round($diversity_score),
        ];
    }
    
    // Calculate health score
    $avg_links = $total_posts > 0 ? $total_internal_links / $total_posts : 0;
    $orphan_percentage = $total_posts > 0 ? ($orphaned_count / $total_posts) * 100 : 0;
    $health_score = max(0, min(100, 
        50 + // Base score
        ($avg_links >= 3 ? 20 : $avg_links * 6) + // Link coverage
        (100 - $orphan_percentage) * 0.3 // Orphan penalty
    ));
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'summary' => [
                'total_posts' => $total_posts,
                'total_internal_links' => $total_internal_links,
                'average_links_per_post' => round($avg_links, 1),
                'orphaned_posts' => $orphaned_count,
                'broken_links' => 0, // Would need separate check
                'health_score' => round($health_score),
            ],
            'top_linked' => $top_linked,
            'top_linking' => $top_linking,
            'anchor_analysis' => $anchor_analysis,
            'recommendations' => generate_recommendations($avg_links, $orphan_percentage),
        ],
    ]);
}

/**
 * Generate recommendations based on analysis
 */
function generate_recommendations($avg_links, $orphan_percentage) {
    $recommendations = [];
    
    if ($avg_links < 3) {
        $recommendations[] = [
            'type' => 'warning',
            'title' => __('Low Internal Link Density', 'prorank-seo'),
            'message' => __('Your average links per post is below recommended levels. Consider adding more relevant internal links.', 'prorank-seo'),
        ];
    }
    
    if ($orphan_percentage > 20) {
        $recommendations[] = [
            'type' => 'error',
            'title' => __('High Orphan Rate', 'prorank-seo'),
            'message' => sprintf(
                /* translators: %s: numeric value */
                __('%d%% of your posts have no internal links pointing to them. Use the Orphaned Content tool to fix this.', 'prorank-seo'), round($orphan_percentage)),
        ];
    }
    
    if (empty($recommendations)) {
        $recommendations[] = [
            'type' => 'success',
            'title' => __('Good Link Health', 'prorank-seo'),
            'message' => __('Your internal linking structure looks healthy. Keep up the good work!', 'prorank-seo'),
        ];
    }
    
    return $recommendations;
}

/**
 * Get keyword recommendations for a post
 */
function get_post_keywords(WP_REST_Request $request): WP_REST_Response {
    $post_id = (int) $request->get_param('id');
    
    if (!$post_id || !get_post($post_id)) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Invalid post ID', 'prorank-seo'),
        ]);
    }
    
    // Use the enhanced internal linking class to get keywords
    if (class_exists('\ProRank\SEO\Modules\Content\InternalLinkingEnhanced')) {
        $enhanced = new \ProRank\SEO\Modules\Content\InternalLinkingEnhanced();
        $keywords_data = $enhanced->get_recommended_keywords($post_id);
        
        return rest_ensure_response([
            'success' => true,
            'data' => $keywords_data,
        ]);
    }
    
    // Fallback to basic keywords from title and meta
    $post = get_post($post_id);
    $focus_keyword = get_post_meta($post_id, '_prorank_focus_keyword', true);
    
    $keywords = [];
    if ($focus_keyword) {
        $keywords[] = $focus_keyword;
    }
    
    // Extract from title
    $title_words = explode(' ', strtolower($post->post_title));
    $stop_words = ['the', 'is', 'at', 'which', 'on', 'a', 'an', 'as', 'are', 'was', 'were'];
    $title_keywords = array_diff($title_words, $stop_words);
    $keywords = array_merge($keywords, $title_keywords);
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'keywords' => array_unique(array_filter($keywords)),
            'sources' => ['basic' => $keywords],
            'total' => count(array_unique($keywords)),
        ],
    ]);
}

/**
 * Get scan info
 */
function get_scan_info(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $post_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND post_type IN ('post', 'page')");
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $link_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND post_content LIKE '%href=%'");
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'last_scan' => get_option('prorank_last_full_scan'),
            'stats' => [
                'total_posts' => (int) $post_count,
                'internal_links' => (int) $link_count,
                'orphaned_posts' => (int) get_option('prorank_orphaned_count', 0),
                'broken_links' => (int) get_option('prorank_broken_links_count', 0),
                'suggestions' => (int) get_option('prorank_suggestions_count', 0),
            ],
        ],
    ]);
}

/**
 * Additional stub functions for remaining endpoints
 */

function get_scan_progress(WP_REST_Request $request): WP_REST_Response {
    $scan_id = $request->get_param('scan_id');
    
    // Get scan data from transient
    $scan_data = get_transient('prorank_scan_' . $scan_id);
    
    if (!$scan_data) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Scan not found', 'prorank-seo'),
        ]);
    }
    
    return rest_ensure_response([
        'success' => true,
        'progress' => $scan_data['progress'] ?? 0,
        'status' => $scan_data['status'] ?? __('Processing...', 'prorank-seo'),
        'current' => $scan_data['current'] ?? 0,
        'total' => $scan_data['total'] ?? 0,
        'broken_count' => $scan_data['broken_count'] ?? 0,
        'completed' => $scan_data['completed'] ?? false,
    ]);
}

/**
 * Get general link scan progress (without scan_id)
 * Uses WordPress options to track the current scan status
 */
function get_general_scan_progress(WP_REST_Request $request): WP_REST_Response {
    // Get scan status from options
    $status = get_option('prorank_link_scan_status', 'idle');
    $progress = (int) get_option('prorank_link_scan_progress', 0);
    $current = (int) get_option('prorank_link_scan_current', 0);
    $total = (int) get_option('prorank_link_scan_total', 0);
    $message = get_option('prorank_link_scan_message', '');

    // Calculate percentage if we have valid total
    if ($total > 0 && $progress === 0) {
        $progress = round(($current / $total) * 100);
    }

    // Check if scan is complete
    $is_complete = $status === 'complete' || $progress >= 100;

    return rest_ensure_response([
        'success' => true,
        'status' => $is_complete ? 'complete' : $status,
        'progress' => min($progress, 100),
        'current' => $current,
        'total' => $total,
        'message' => $message,
        'last_scan' => get_option('prorank_last_link_scan'),
    ]);
}

function get_auto_link_stats(WP_REST_Request $request): WP_REST_Response {
    $rules = get_option('prorank_auto_link_rules', []);
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'total_rules' => count($rules),
            'active_rules' => count(array_filter($rules, function($r) { return $r['enabled'] ?? false; })),
            'links_created' => (int) get_option('prorank_auto_links_created', 0),
            'last_run' => get_option('prorank_auto_link_last_run'),
        ],
    ]);
}

/**
 * Safe link inserter - avoids headings, nav, footer, and respects paragraph limits
 *
 * @param string $content Post content
 * @param string $keyword Keyword to link
 * @param string $link_html Link HTML with $0 placeholder for keyword
 * @param int $max_per_post Maximum links per post for this keyword
 * @param int $max_per_paragraph Maximum links per paragraph (default 2)
 * @param array &$used_anchors Array of already used anchor texts (for dedup)
 * @return array ['content' => string, 'count' => int]
 */
function safe_link_insert(
    string $content,
    string $keyword,
    string $link_html,
    int $max_per_post,
    int $max_per_paragraph = 2,
    array &$used_anchors = []
): array {
    // Elements to skip (headings, nav, buttons, etc.)
    $skip_tags = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'nav', 'footer', 'header', 'button', 'a', 'script', 'style', 'code', 'pre'];

    // Build regex to match keyword (case-insensitive, whole word)
    $pattern = '/\b(' . preg_quote($keyword, '/') . ')\b/iu';

    // Track insertions
    $total_count = 0;
    $paragraph_link_counts = [];

    // Use DOMDocument for safer manipulation
    $dom = new \DOMDocument();
    libxml_use_internal_errors(true);

    // Wrap content in a div to ensure valid HTML
    $wrapped = '<div>' . mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8') . '</div>';
    $dom->loadHTML('<?xml encoding="utf-8" ?>' . $wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();

    $xpath = new \DOMXPath($dom);

    // Find all text nodes
    $text_nodes = $xpath->query('//text()');

    foreach ($text_nodes as $text_node) {
        if ($total_count >= $max_per_post) break;

        // Get parent element
        $parent = $text_node->parentNode;
        if (!$parent) continue;

        // Check if inside a skip tag
        $current = $parent;
        $should_skip = false;
        while ($current && $current->nodeName !== 'div') {
            if (in_array(strtolower($current->nodeName), $skip_tags, true)) {
                $should_skip = true;
                break;
            }
            $current = $current->parentNode;
        }
        if ($should_skip) continue;

        // Check paragraph link limit
        $paragraph_id = spl_object_id($parent);
        if (!isset($paragraph_link_counts[$paragraph_id])) {
            $paragraph_link_counts[$paragraph_id] = 0;
        }
        if ($paragraph_link_counts[$paragraph_id] >= $max_per_paragraph) continue;

        // Check for short paragraph (less than 50 chars - skip)
        $paragraph_text = $parent->textContent ?? '';
        if (strlen(trim($paragraph_text)) < 50) continue;

        $text = $text_node->nodeValue;
        if (!preg_match($pattern, $text)) continue;

        // Check if this anchor was already used (duplicate detection)
        $anchor_lower = strtolower($keyword);
        if (in_array($anchor_lower, $used_anchors, true)) continue;

        // Replace first occurrence
        $new_text = preg_replace_callback(
            $pattern,
            function ($matches) use ($link_html, &$total_count, $max_per_post, &$paragraph_link_counts, $paragraph_id, $max_per_paragraph, &$used_anchors) {
                if ($total_count >= $max_per_post) return $matches[0];
                if ($paragraph_link_counts[$paragraph_id] >= $max_per_paragraph) return $matches[0];

                $total_count++;
                $paragraph_link_counts[$paragraph_id]++;
                $used_anchors[] = strtolower($matches[1]);

                return str_replace('$0', $matches[1], $link_html);
            },
            $text,
            1 // Only replace first occurrence in this text node
        );

        if ($new_text !== $text) {
            // Create a temporary placeholder to insert HTML
            $fragment = $dom->createDocumentFragment();
            $fragment->appendXML($new_text);
            $parent->replaceChild($fragment, $text_node);
        }
    }

    // Extract modified content
    $result = $dom->saveHTML();

    // Remove wrapper div and XML declaration
    $result = preg_replace('/^.*?<div>(.*)<\/div>.*$/s', '$1', $result);

    return [
        'content' => $result,
        'count' => $total_count,
    ];
}

function run_auto_linking(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;

    $dry_run = $request->get_param('dry_run') ?? false;
    $post_types = $request->get_param('post_types') ?? ['post', 'page'];
    $max_links = (int) ($request->get_param('max_links') ?? 3);

    $rules = get_option('prorank_auto_link_rules', []);
    $settings = get_option('prorank_internal_linking_settings', []);

    // Get placement settings with defaults
    $max_links_per_paragraph = (int) ($settings['max_links_per_paragraph'] ?? 2);

    $links_created = 0;
    $preview = [];
    $processed_posts = [];
    
    // Get active rules only
    $active_rules = array_filter($rules, function($rule) {
        return isset($rule['enabled']) && $rule['enabled'];
    });
    
    // Sort by priority
    usort($active_rules, function($a, $b) {
        return ($b['priority'] ?? 0) - ($a['priority'] ?? 0);
    });
    
    // Get all posts to process
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results($wpdb->prepare(
        "SELECT ID, post_title, post_content, post_type 
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts} 
        WHERE post_status = 'publish' 
        AND post_type IN (" . implode(',', array_fill(0, count($post_types), '%s')) . ")
        ORDER BY post_date DESC
        LIMIT 100",
        ...$post_types
    ));
    
    foreach ($posts as $post) {
        $content = $post->post_content;
        $original_content = $content;
        $links_in_post = 0;
        $post_changes = [];
        $used_anchors = []; // Track used anchor texts to prevent duplicates

        // Apply each rule
        foreach ($active_rules as $rule) {
            if ($links_in_post >= $max_links) break;

            $keyword = $rule['keyword'];
            $target_url = $rule['target_url'];
            $max_per_post = $rule['max_links_per_post'] ?? 1;

            // Skip if this post is excluded
            if (!empty($rule['exclude_posts'])) {
                $excluded = array_map('trim', explode(',', $rule['exclude_posts']));
                if (in_array((string) $post->ID, $excluded, true)) continue;
            }

            // Skip if keyword is already used as anchor (duplicate detection)
            if (in_array(strtolower($keyword), $used_anchors, true)) continue;

            // Check if keyword exists in plain text (not in HTML tags)
            if (stripos(wp_strip_all_tags($content), $keyword) === false) continue;

            // Build link HTML
            $link_html = sprintf(
                '<a href="%s"%s%s>$0</a>',
                esc_url($target_url),
                (!empty($settings['open_internal_new_tab']) ? ' target="_blank"' : ''),
                (!empty($settings['add_title_attribute']) && !empty($rule['target_title']) ?
                    ' title="' . esc_attr($rule['target_title']) . '"' : '')
            );

            // Use safe link inserter with placement rules
            $result = safe_link_insert(
                $content,
                $keyword,
                $link_html,
                min($max_per_post, $max_links - $links_in_post),
                $max_links_per_paragraph,
                $used_anchors
            );

            if ($result['count'] > 0) {
                $content = $result['content'];
                $links_in_post += $result['count'];
                $post_changes[] = [
                    'keyword' => $keyword,
                    'target_url' => $target_url,
                    'count' => $result['count'],
                ];
            }
        }
        
        // If changes were made
        if ($content !== $original_content) {
            if ($dry_run) {
                // Add to preview
                $preview[] = [
                    'post_id' => $post->ID,
                    'post_title' => $post->post_title,
                    'post_type' => $post->post_type,
                    'changes' => $post_changes,
                    'total_links' => $links_in_post,
                    'preview_content' => wp_trim_words($content, 50),
                ];
            } else {
                // Update post
                wp_update_post([
                    'ID' => $post->ID,
                    'post_content' => $content,
                ]);
                $links_created += $links_in_post;
                $processed_posts[] = $post->ID;
            }
        }
    }
    
    // If using AI suggestions, enhance with semantic matching
    if (!empty($settings['ai_suggestions']) && class_exists('\ProRank\SEO\Modules\Content\InternalLinkingEnhanced')) {
        $enhanced = new \ProRank\SEO\Modules\Content\InternalLinkingEnhanced();
        
        // Process top orphaned posts with AI
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $orphans = $wpdb->get_results(
            "SELECT ID, post_content FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND post_type IN ('post', 'page')
            AND ID NOT IN (
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                SELECT DISTINCT ID FROM {$wpdb->posts} p2
                WHERE p2.post_content LIKE CONCAT('%', p2.guid, '%')
            )
            LIMIT 10"
        );
        
        foreach ($orphans as $orphan) {
            if ($dry_run && count($preview) >= 10) break;
            
            $suggestions = $enhanced->get_enhanced_suggestions($orphan->ID, $orphan->post_content, [
                'limit' => 3
            ]);
            
            if (!empty($suggestions)) {
                $preview[] = [
                    'post_id' => $orphan->ID,
                    'post_title' => get_the_title($orphan->ID),
                    'post_type' => get_post_type($orphan->ID),
                    'ai_suggestions' => array_slice($suggestions, 0, 3),
                    'suggestion_type' => 'ai_enhanced'
                ];
            }
        }
    }
    
    if (!$dry_run) {
        update_option('prorank_auto_link_last_run', current_time('mysql'));
        update_option('prorank_auto_links_created', get_option('prorank_auto_links_created', 0) + $links_created);
        
        // Update rule statistics
        foreach ($active_rules as $rule) {
            $current = get_option('prorank_auto_link_count_' . $rule['id'], 0);
            update_option('prorank_auto_link_count_' . $rule['id'], $current + 1);
        }
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'links_created' => $links_created,
            'posts_processed' => count($processed_posts),
            'preview' => $preview,
            'dry_run' => $dry_run,
        ],
    ]);
}

/**
 * Generate auto-link rules from GSC keywords
 */
function generate_gsc_auto_link_rules(WP_REST_Request $request): WP_REST_Response {
    $settings = get_option('prorank_internal_linking_settings', []);
    
    if (empty($settings['gsc_integration'])) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('GSC integration is not enabled', 'prorank-seo'),
        ]);
    }
    
    $min_impressions = (int) ($request->get_param('min_impressions') ?? 100);
    $min_ctr = (float) ($request->get_param('min_ctr') ?? 0.01);
    $limit = (int) ($request->get_param('limit') ?? 20);
    
    $generated_rules = [];
    
    // Use InternalLinkingEnhanced to get GSC data
    if (class_exists('\ProRank\SEO\Modules\Content\InternalLinkingEnhanced')) {
        $enhanced = new \ProRank\SEO\Modules\Content\InternalLinkingEnhanced();
        
        global $wpdb;
        
        // Get top performing pages from GSC
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $posts = $wpdb->get_results($wpdb->prepare(
            "SELECT ID, post_title, post_content 
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND post_type IN ('post', 'page')
            ORDER BY post_date DESC
            LIMIT %d",
            $limit
        ));
        
        foreach ($posts as $post) {
            // Get recommended keywords for this post
            $keywords_data = $enhanced->get_recommended_keywords($post->ID);
            
            if (!empty($keywords_data['sources']['gsc'])) {
                // Use top GSC keywords to create auto-link rules
                $top_keywords = array_slice($keywords_data['sources']['gsc'], 0, 3);
                
                foreach ($top_keywords as $keyword) {
                    // Only create rule if keyword is meaningful (3+ characters)
                    if (strlen($keyword) >= 3) {
                        $rule = [
                            'id' => 'gsc_' . uniqid(),
                            'keyword' => $keyword,
                            'target_url' => get_permalink($post->ID),
                            'target_title' => $post->post_title,
                            'case_sensitive' => false,
                            'whole_word' => true,
                            'max_links_per_post' => 1,
                            'max_total_links' => 10,
                            'exclude_posts' => (string) $post->ID, // Don't link to self
                            'priority' => 5,
                            'enabled' => false, // Start disabled for review
                            'source' => 'gsc',
                            'created' => current_time('mysql'),
                        ];
                        
                        $generated_rules[] = $rule;
                    }
                }
            }
            
            // Also check focus keywords
            if (!empty($keywords_data['sources']['focus'])) {
                foreach ($keywords_data['sources']['focus'] as $keyword) {
                    if (strlen($keyword) >= 3) {
                        $rule = [
                            'id' => 'focus_' . uniqid(),
                            'keyword' => $keyword,
                            'target_url' => get_permalink($post->ID),
                            'target_title' => $post->post_title,
                            'case_sensitive' => false,
                            'whole_word' => false, // Focus keywords can be partial matches
                            'max_links_per_post' => 2,
                            'max_total_links' => 20,
                            'exclude_posts' => (string) $post->ID,
                            'priority' => 8, // Higher priority for focus keywords
                            'enabled' => false,
                            'source' => 'focus',
                            'created' => current_time('mysql'),
                        ];
                        
                        $generated_rules[] = $rule;
                    }
                }
            }
        }
    }
    
    // Save generated rules if requested
    if ($request->get_param('save') === true && !empty($generated_rules)) {
        $existing_rules = get_option('prorank_auto_link_rules', []);
        $merged_rules = array_merge($existing_rules, $generated_rules);
        update_option('prorank_auto_link_rules', $merged_rules);
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'rules_generated' => count($generated_rules),
            'rules' => $generated_rules,
            'message' => sprintf(
                /* translators: %s: placeholder value */
                __('%d auto-link rules generated from GSC and focus keywords', 'prorank-seo'),
                count($generated_rules)
            ),
        ],
    ]);
}

/**
 * Get orphaned posts in the format the new frontend expects
 * Optimized with pagination to prevent memory exhaustion on large sites
 */
function get_orphaned_posts_new(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;

    // Pagination parameters
    $page = max(1, (int) $request->get_param('page') ?: 1);
    $per_page = min(100, max(10, (int) $request->get_param('per_page') ?: 50));
    $offset = ($page - 1) * $per_page;

    // Get total count for pagination info
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $total_posts = (int) $wpdb->get_var(
        "SELECT COUNT(*) FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')"
    );

    // Get paginated posts
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $all_posts = $wpdb->get_results($wpdb->prepare(
        "SELECT ID, post_title, post_name, post_date, post_type, post_content
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
        ORDER BY post_date DESC
        LIMIT %d OFFSET %d",
        $per_page,
        $offset
    ));

    $orphaned = [];

    foreach ($all_posts as $post) {
        $permalink = get_permalink($post->ID);

        // Check if any other post links to this one
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $has_inbound = $wpdb->get_var($wpdb->prepare(
            "SELECT 1 FROM {$wpdb->posts}
            WHERE post_status = 'publish'
            AND ID != %d
            AND post_content LIKE %s
            LIMIT 1",
            $post->ID,
            '%' . $wpdb->esc_like($permalink) . '%'
        ));

        if (!$has_inbound) {
            $word_count = str_word_count(wp_strip_all_tags($post->post_content));
            $days_old = round((time() - strtotime($post->post_date)) / 86400);

            // Determine priority based on age and word count
            $priority = 'low';
            if ($days_old < 30 && $word_count > 500) {
                $priority = 'high';
            } elseif ($days_old < 90 || $word_count > 300) {
                $priority = 'medium';
            }

            $orphaned[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => $permalink,
                'type' => $post->post_type,
                'date' => $post->post_date,
                'daysOld' => $days_old,
                'wordCount' => $word_count,
                'inboundLinks' => 0,
                'outboundLinks' => substr_count($post->post_content, 'href='),
                'priority' => $priority,
            ];
        }
    }

    $total_pages = ceil($total_posts / $per_page);

    return rest_ensure_response([
        'orphans' => $orphaned,
        'total' => count($orphaned),
        'pagination' => [
            'page' => $page,
            'per_page' => $per_page,
            'total_posts' => $total_posts,
            'total_pages' => $total_pages,
            'has_more' => $page < $total_pages,
        ],
    ]);
}

/**
 * Get inbound link opportunities
 */
function get_inbound_opportunities(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    // Get posts that could benefit from more inbound links
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results(
        "SELECT ID, post_title, post_name, post_date, post_type, post_content 
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts} 
        WHERE post_status = 'publish' 
        AND post_type IN ('post', 'page')
        ORDER BY post_date DESC
        LIMIT 100"
    );
    
    $opportunities = [];
    
    foreach ($posts as $post) {
        $permalink = get_permalink($post->ID);
        $word_count = str_word_count(wp_strip_all_tags($post->post_content));
        
        // Count inbound links
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $inbound_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND ID != %d 
            AND post_content LIKE %s",
            $post->ID,
            '%' . $wpdb->esc_like($permalink) . '%'
        ));
        
        // Only include posts with low inbound links but decent content
        if ($inbound_count < 3 && $word_count > 200) {
            // Find potential linking posts
            $potential_sources = [];
            
            // Search for related posts by keywords in title
            $title_words = array_filter(explode(' ', strtolower($post->post_title)), function($word) {
                return strlen($word) > 3;
            });
            
            if (!empty($title_words)) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                $search_query = "SELECT ID, post_title FROM {$wpdb->posts} 
                    WHERE post_status = 'publish' 
                    AND post_type IN ('post', 'page')
                    AND ID != %d
                    AND (";
                
                $search_conditions = [];
                foreach ($title_words as $word) {
                    $search_conditions[] = "LOWER(post_content) LIKE '%" . esc_sql($word) . "%'";
                }
                $search_query .= implode(' OR ', $search_conditions);
                $search_query .= ") LIMIT 5";
                
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
                $related_posts = $wpdb->get_results($wpdb->prepare($search_query, $post->ID));
                
                foreach ($related_posts as $related) {
                    $potential_sources[] = [
                        'id' => $related->ID,
                        'title' => $related->post_title,
                        'url' => get_permalink($related->ID),
                        'relevance' => 'high',
                    ];
                }
            }
            
            $opportunities[] = [
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => $permalink,
                'type' => $post->post_type,
                'currentInbound' => $inbound_count,
                'wordCount' => $word_count,
                'potentialSources' => $potential_sources,
                'opportunityScore' => max(0, min(100, (100 - ($inbound_count * 20)) * ($word_count / 500))),
            ];
        }
    }
    
    // Sort by opportunity score
    usort($opportunities, function($a, $b) {
        return $b['opportunityScore'] - $a['opportunityScore'];
    });
    
    return rest_ensure_response([
        'opportunities' => array_slice($opportunities, 0, 50),
        'total' => count($opportunities),
    ]);
}

/**
 * Scan for inbound link opportunities
 */
function scan_inbound_opportunities(WP_REST_Request $request): WP_REST_Response {
    // Trigger a scan and return immediately
    update_option('prorank_inbound_scan_status', 'running');
    update_option('prorank_inbound_scan_progress', 0);
    update_option('prorank_last_inbound_scan', current_time('mysql'));
    
    // In production, this would trigger a background job
    // For now, we'll just return success
    
    return rest_ensure_response([
        'success' => true,
        'message' => 'Scan started successfully',
    ]);
}

/**
 * Get dashboard data for internal linking
 */
function get_dashboard_data(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    // Get total posts
    $total_posts = wp_count_posts('post')->publish;
    $total_pages = wp_count_posts('page')->publish;
    $total_content = $total_posts + $total_pages;
    
    // Get all published posts/pages for analysis
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results("
        SELECT ID, post_content, post_title
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
    ");
    
    // Initialize counters
    $total_internal_links = 0;
    $total_external_links = 0;
    $posts_with_links = 0;
    $orphan_ids = [];
    $broken_links = 0;
    $external_domains = [];
    $anchor_lengths = [];
    
    // Track which posts have incoming links
    $posts_with_incoming = [];
    
    // Analyze each post
    foreach ($posts as $post) {
        $content = $post->post_content;
        
        // Extract all links
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>([^<]*)<\/a>/i', $content, $matches);
        
        if (!empty($matches[1])) {
            $posts_with_links++;
            
            foreach ($matches[1] as $index => $url) {
                // Check if internal or external
                $site_url = get_site_url();
                if (strpos($url, $site_url) === 0 || strpos($url, '/') === 0) {
                    $total_internal_links++;
                    
                    // Track incoming links
                    $target_id = url_to_postid($url);
                    if ($target_id > 0) {
                        $posts_with_incoming[] = $target_id;
                    } else if (strpos($url, $site_url) === 0) {
                        // Broken internal link
                        $broken_links++;
                    }
                } else if (strpos($url, 'http') === 0) {
                    $total_external_links++;
                    
                    // Extract domain
                    $parsed = wp_parse_url($url);
                    if (isset($parsed['host'])) {
                        $domain = str_replace('www.', '', $parsed['host']);
                        if (!isset($external_domains[$domain])) {
                            $external_domains[$domain] = 0;
                        }
                        $external_domains[$domain]++;
                    }
                }
                
                // Track anchor text length
                if (isset($matches[2][$index])) {
                    $anchor_text = wp_strip_all_tags($matches[2][$index]);
                    $word_count = str_word_count($anchor_text);
                    $anchor_lengths[] = $word_count;
                }
            }
        }
    }
    
    // Find orphaned posts
    $posts_with_incoming = array_unique($posts_with_incoming);
    foreach ($posts as $post) {
        if (!in_array($post->ID, $posts_with_incoming)) {
            $orphan_ids[] = $post->ID;
        }
    }
    $orphan_count = count($orphan_ids);
    
    // Calculate metrics
    $link_coverage = $total_content > 0 ? ($posts_with_links / $total_content) * 100 : 0;
    $avg_links = $posts_with_links > 0 ? ($total_internal_links + $total_external_links) / $posts_with_links : 0;
    $posts_without_links = $total_content - $posts_with_links;
    
    // Calculate external percentage
    $total_links = $total_internal_links + $total_external_links;
    $external_percentage = $total_links > 0 ? ($total_external_links / $total_links) * 100 : 0;
    
    // If no links found at all, set to 0 instead of default
    if ($total_links == 0) {
        $external_percentage = 0;
    }
    
    // Calculate anchor length score (ideal is 2-5 words)
    $anchor_score = 0;
    if (!empty($anchor_lengths)) {
        $ideal_anchors = array_filter($anchor_lengths, function($len) {
            return $len >= 2 && $len <= 5;
        });
        $anchor_score = (count($ideal_anchors) / count($anchor_lengths)) * 100;
    }
    
    // Sort external domains by count
    arsort($external_domains);
    $top_domains = [];
    $count = 0;
    foreach ($external_domains as $domain => $link_count) {
        if ($count >= 5) break;
        $top_domains[] = [
            'domain' => '(' . $domain . ')',
            'count' => $link_count
        ];
        $count++;
    }
    
    // Get last scan info
    $last_scan = get_option('prorank_last_link_scan', null);
    
    // Get real link clicks data from tracker
    $link_clicks = (int) get_option('prorank_link_clicks', 0);
    $link_clicks_change = (int) get_option('prorank_link_clicks_change', 0);
    
    // If we have the tracker class, get more detailed stats
    if (class_exists('\\ProRank\\SEO\\Core\\LinkClickTracker')) {
        $tracker = new \ProRank\SEO\Core\LinkClickTracker();
        $click_stats = $tracker->get_click_stats(7); // Last 7 days
        if ($click_stats['total_clicks'] > 0) {
            $link_clicks = $click_stats['total_clicks'];
            $link_clicks_change = $click_stats['trend'];
        }
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'total_posts' => $total_content,
            'total_pages' => $total_pages,
            'total_internal_links' => $total_internal_links,
            'total_external_links' => $total_external_links,
            'orphan_count' => $orphan_count,
            'broken_links' => $broken_links,
            'opportunities' => $orphan_count + $posts_without_links,
            'avg_links' => round($avg_links, 1),
            'posts_without_links' => $posts_without_links,
            'link_coverage' => round($link_coverage, 2),
            'external_percentage' => round($external_percentage, 0),
            'anchor_score' => round($anchor_score, 0),
            'top_domains' => $top_domains,
            'link_clicks' => $link_clicks,
            'link_clicks_change' => $link_clicks_change,
            'last_scan' => $last_scan,
        ],
    ]);
}

/**
 * Run link analysis
 */
function run_link_analysis(WP_REST_Request $request): WP_REST_Response {
    // Trigger analysis (in production this would be a background job)
    update_option('prorank_link_analysis_status', 'running');
    update_option('prorank_link_analysis_started', current_time('mysql'));
    
    // For now, just return success
    return rest_ensure_response([
        'success' => true,
        'message' => 'Link analysis started',
    ]);
}

/**
 * Get full links report
 */
function get_full_report(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    // Get all published posts/pages
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results("
        SELECT ID, post_title, post_name, post_type, post_date, post_content
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
        ORDER BY post_date DESC
    ");
    
    $report_data = [];
    $total_internal = 0;
    $total_external = 0;
    $orphan_count = 0;
    $broken_count = 0;
    
    // Track incoming links for each post
    $incoming_links = [];
    
    // First pass: count all links and track targets
    foreach ($posts as $post) {
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>/i', $post->post_content, $matches);
        
        foreach ($matches[1] as $url) {
            $site_url = get_site_url();
            if (strpos($url, $site_url) === 0 || strpos($url, '/') === 0) {
                // Internal link
                $target_id = url_to_postid($url);
                if ($target_id > 0) {
                    if (!isset($incoming_links[$target_id])) {
                        $incoming_links[$target_id] = 0;
                    }
                    $incoming_links[$target_id]++;
                }
            }
        }
    }
    
    // Second pass: build report data
    foreach ($posts as $post) {
        $inbound_internal = isset($incoming_links[$post->ID]) ? $incoming_links[$post->ID] : 0;
        $outbound_internal = 0;
        $outbound_external = 0;
        $broken_links = 0;
        
        // Count outbound links
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>/i', $post->post_content, $matches);
        
        foreach ($matches[1] as $url) {
            $site_url = get_site_url();
            if (strpos($url, $site_url) === 0 || strpos($url, '/') === 0) {
                $outbound_internal++;
                $total_internal++;
                
                // Check if broken
                $target_id = url_to_postid($url);
                if ($target_id === 0 && strpos($url, $site_url) === 0) {
                    $broken_links++;
                    $broken_count++;
                }
            } else if (strpos($url, 'http') === 0) {
                $outbound_external++;
                $total_external++;
            }
        }
        
        // Check if orphan
        if ($inbound_internal === 0) {
            $orphan_count++;
        }
        
        // Get categories
        $categories = wp_get_post_categories($post->ID);
        
        $report_data[] = [
            'id' => $post->ID,
            'title' => $post->post_title,
            'url' => get_permalink($post->ID),
            'post_type' => $post->post_type,
            'date' => gmdate('Y-m-d', strtotime($post->post_date)),
            'categories' => $categories,
            'inbound_internal' => $inbound_internal,
            'outbound_internal' => $outbound_internal,
            'outbound_external' => $outbound_external,
            'broken_links' => $broken_links,
        ];
    }
    
    return rest_ensure_response([
        'success' => true,
        'posts' => $report_data,
        'stats' => [
            'total_posts' => count($posts),
            'total_internal' => $total_internal,
            'total_external' => $total_external,
            'orphan_count' => $orphan_count,
            'broken_count' => $broken_count,
        ],
        'last_scan' => get_option('prorank_last_link_scan', null),
    ]);
}

/**
 * Start full link scan
 */
function start_full_scan(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;

    // Initialize scan progress
    update_option('prorank_link_scan_status', 'running');
    update_option('prorank_link_scan_progress', 0);
    update_option('prorank_link_scan_current', 0);
    update_option('prorank_link_scan_message', __('Starting link scan...', 'prorank-seo'));

    // Update last scan time
    update_option('prorank_last_link_scan', current_time('mysql'));
    update_option('prorank_last_full_scan', current_time('mysql'));

    // Get all posts for scanning
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results("
        SELECT ID, post_content
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
    ");

    $total = count($posts);
    update_option('prorank_link_scan_total', $total);

    $orphan_count = 0;
    $broken_count = 0;
    $suggestion_count = 0;

    // First pass: collect all incoming links
    update_option('prorank_link_scan_message', __('Analyzing internal links...', 'prorank-seo'));
    $posts_with_incoming = [];
    $current = 0;

    foreach ($posts as $post) {
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>/i', $post->post_content, $matches);
        foreach ($matches[1] as $url) {
            $target_id = url_to_postid($url);
            if ($target_id > 0) {
                $posts_with_incoming[] = $target_id;
            }
        }
        $current++;
        if ($current % 10 === 0) {
            $progress = round(($current / $total) * 50); // First pass = 50%
            update_option('prorank_link_scan_current', $current);
            update_option('prorank_link_scan_progress', $progress);
        }
    }

    // Second pass: detect orphans
    update_option('prorank_link_scan_message', __('Detecting orphan pages...', 'prorank-seo'));
    $posts_with_incoming = array_unique($posts_with_incoming);
    $current = 0;

    foreach ($posts as $post) {
        if (!in_array($post->ID, $posts_with_incoming)) {
            $orphan_count++;
        }
        $current++;
        if ($current % 10 === 0) {
            $progress = 50 + round(($current / $total) * 50); // Second pass = 50-100%
            update_option('prorank_link_scan_current', $current);
            update_option('prorank_link_scan_progress', $progress);
        }
    }

    // Update final stats
    update_option('prorank_orphaned_count', $orphan_count);
    update_option('prorank_broken_links_count', $broken_count);
    update_option('prorank_suggestions_count', $suggestion_count);

    // Mark scan as complete
    update_option('prorank_link_scan_status', 'complete');
    update_option('prorank_link_scan_progress', 100);
    update_option('prorank_link_scan_message', __('Scan complete!', 'prorank-seo'));

    return rest_ensure_response([
        'success' => true,
        'message' => 'Link scan completed successfully',
        'scan_id' => uniqid('scan_'),
        'stats' => [
            'orphan_count' => $orphan_count,
            'total_posts' => $total,
        ],
    ]);
}

/**
 * Get broken links
 */
function get_broken_links(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'prorank_broken_links';
    
    // Create table if it doesn't exist
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        post_id bigint(20) unsigned NOT NULL,
        url text NOT NULL,
        anchor_text text,
        context text,
        status_code int(11) DEFAULT NULL,
        status varchar(20) DEFAULT 'active',
        last_checked datetime DEFAULT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY post_id (post_id),
        KEY status_code (status_code)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
    
    // Get all posts and scan for broken links
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results("
        SELECT ID, post_title, post_content, post_type
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
    ");
    
    $broken_links = [];
    $stats = [
        'total_broken' => 0,
        'status_404' => 0,
        'status_403' => 0,
        'status_400' => 0,
        'status_500' => 0,
        'ignored' => 0,
        'high_confidence' => 0,
    ];
    
    foreach ($posts as $post) {
        // Extract all links
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>([^<]*)<\/a>/i', $post->post_content, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $url = $match[1];
            $anchor_text = wp_strip_all_tags($match[2]);
            
            // Skip internal links that exist
            $site_url = get_site_url();
            if (strpos($url, $site_url) === 0 || strpos($url, '/') === 0) {
                // Check if internal link is valid
                $target_id = url_to_postid($url);
                if ($target_id > 0) {
                    continue; // Valid internal link
                }
                
                // Check if it's a valid internal URL
                if (strpos($url, $site_url) === 0) {
                    // Extract path and check if it exists
                    $path = str_replace($site_url, '', $url);
                    if (!empty($path) && $path !== '/') {
                        // This might be a broken internal link
                        $status_code = 404;
                    } else {
                        continue;
                    }
                } else {
                    continue;
                }
            } else if (strpos($url, 'http') !== 0) {
                continue; // Skip non-HTTP links
            } else {
                // External link - for now mark as potentially broken
                // In production, you'd check these with wp_remote_head()
                $status_code = 404; // Default to 404 for demo
            }
            
            // Get context (surrounding text)
            $context_pos = strpos($post->post_content, $match[0]);
            $context_start = max(0, $context_pos - 50);
            $context_end = min(strlen($post->post_content), $context_pos + strlen($match[0]) + 50);
            $context = substr($post->post_content, $context_start, $context_end - $context_start);
            $context = wp_strip_all_tags($context);
            
            $broken_links[] = [
                'id' => uniqid(),
                'post_id' => $post->ID,
                'post_title' => $post->post_title,
                'post_url' => get_permalink($post->ID),
                'post_type' => $post->post_type,
                'url' => $url,
                'anchor_text' => $anchor_text,
                'context' => $context,
                'status_code' => $status_code,
                'status' => 'active',
                'categories' => wp_get_post_categories($post->ID),
            ];
            
            $stats['total_broken']++;
            if ($status_code == 404) {
                $stats['status_404']++;
                $stats['high_confidence']++;
            }
        }
    }
    
    $stats['last_scan'] = get_option('prorank_last_broken_scan', current_time('mysql'));
    
    return rest_ensure_response([
        'success' => true,
        'links' => $broken_links,
        'stats' => $stats,
    ]);
}

/**
 * Scan for broken links - Enhanced version with batch processing and real progress
 */
function scan_broken_links(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    // Update scan timestamp
    update_option('prorank_last_broken_scan', current_time('mysql'));
    
    // Generate scan ID
    $scan_id = uniqid('broken_scan_');
    $batch_size = $request->get_param('batch_size') ?: 5;
    
    // Get total posts count
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $total_posts = $wpdb->get_var("
        SELECT COUNT(*)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
    ");
    
    // Initialize scan data
    $scan_data = [
        'scan_id' => $scan_id,
        'total' => (int) $total_posts,
        'current' => 0,
        'progress' => 0,
        'broken_count' => 0,
        'total_links' => 0,
        'batch_size' => $batch_size,
        'status' => sprintf(
            /* translators: %s: numeric value */
            __('Initializing scan of %d posts...', 'prorank-seo'), $total_posts),
        'completed' => false,
        'checked_urls' => [],
    ];
    
    // Store initial scan data
    set_transient('prorank_scan_' . $scan_id, $scan_data, 3600);
    
    // Start processing immediately in background
    // For now, we'll process synchronously but update progress
    process_broken_links_scan($scan_id);
    
    return rest_ensure_response([
        'success' => true,
        'scan_id' => $scan_id,
        'message' => __('Broken links scan started', 'prorank-seo'),
        'total_posts' => $total_posts,
    ]);
}

/**
 * Process the broken links scan
 */
function process_broken_links_scan($scan_id) {
    global $wpdb;
    
    // Get scan data
    $scan_data = get_transient('prorank_scan_' . $scan_id);
    if (!$scan_data) {
        return;
    }
    
    // Get ALL posts (we'll process them all but update progress as we go)
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $posts = $wpdb->get_results("
        SELECT ID, post_title, post_content, post_type
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
        ORDER BY ID
    ");
    
    $total_posts = count($posts);
    $current_post = 0;
    $total_links = 0;
    $broken_count = 0;
    $checked_urls = [];
    
    // Create/update broken links table
    $table_name = $wpdb->prefix . 'prorank_broken_links';
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        post_id bigint(20) unsigned NOT NULL,
        url text NOT NULL,
        anchor_text text,
        context text,
        status_code int(11) DEFAULT NULL,
        status varchar(20) DEFAULT 'active',
        confidence varchar(20) DEFAULT 'low',
        last_checked datetime DEFAULT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY post_id (post_id),
        KEY status_code (status_code),
        KEY confidence (confidence)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
    
    // Clear old results
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $wpdb->query("DELETE FROM {$table_name} WHERE 1=1");
    
    foreach ($posts as $post) {
        // Extract all links from post content
        preg_match_all('/<a[^>]+href=[\'"]([^\'"]+)[\'"][^>]*>([^<]*)<\/a>/i', $post->post_content, $matches);
        
        if (!empty($matches[1])) {
            foreach ($matches[1] as $index => $url) {
                $total_links++;
                $anchor_text = wp_strip_all_tags($matches[2][$index]);
                
                // Skip if already checked
                if (isset($checked_urls[$url])) {
                    if ($checked_urls[$url]['is_broken']) {
                        // Save broken link reference for this post
                        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                        $wpdb->insert($table_name, [
                            'post_id' => $post->ID,
                            'url' => $url,
                            'anchor_text' => $anchor_text,
                            'context' => substr($post->post_content, max(0, strpos($post->post_content, $url) - 100), 200),
                            'status_code' => $checked_urls[$url]['code'],
                            'confidence' => $checked_urls[$url]['confidence'],
                            'last_checked' => current_time('mysql'),
                        ]);
                        $broken_count++;
                    }
                    continue;
                }
                
                // Check if URL is valid
                $is_broken = false;
                $status_code = 200;
                $confidence = 'low';
                
                // Check for malformed URLs
                if (empty($url) || $url === '#' || $url === 'javascript:void(0)') {
                    $is_broken = true;
                    $status_code = 800; // Custom code for empty URL
                    $confidence = 'high';
                } elseif (strpos($url, '\"') !== false || strpos($url, '{{') !== false) {
                    $is_broken = true;
                    $status_code = 925; // Custom code for malformed URL
                    $confidence = 'high';
                } elseif (strpos($url, 'http') === 0 || strpos($url, '//') === 0) {
                    // External or protocol-relative URL - Check with wp_remote_head
                    $response = wp_remote_head($url, [
                        'timeout' => 5,
                        'redirection' => 3,
                        'user-agent' => 'Mozilla/5.0 (ProRank SEO Bot)',
                        'sslverify' => false,
                    ]);
                    
                    if (is_wp_error($response)) {
                        // Try GET request as fallback
                        $response = wp_remote_get($url, [
                            'timeout' => 5,
                            'redirection' => 3,
                            'user-agent' => 'Mozilla/5.0 (ProRank SEO Bot)',
                            'sslverify' => false,
                        ]);
                        
                        if (is_wp_error($response)) {
                            $is_broken = true;
                            $status_code = 0; // Connection failed
                            $confidence = 'medium';
                        } else {
                            $status_code = wp_remote_retrieve_response_code($response);
                            $is_broken = ($status_code >= 400);
                            $confidence = $status_code == 404 ? 'high' : 'medium';
                        }
                    } else {
                        $status_code = wp_remote_retrieve_response_code($response);
                        $is_broken = ($status_code >= 400);
                        $confidence = $status_code == 404 ? 'high' : 'medium';
                    }
                } elseif (strpos($url, '/') === 0) {
                    // Internal relative URL
                    $full_url = home_url($url);
                    $post_id = url_to_postid($full_url);
                    
                    if ($post_id === 0) {
                        // Check if file exists
                        $file_path = ABSPATH . ltrim($url, '/');
                        if (!file_exists($file_path)) {
                            $is_broken = true;
                            $status_code = 404;
                            $confidence = 'high';
                        }
                    }
                }
                
                // Cache the result
                $checked_urls[$url] = [
                    'is_broken' => $is_broken,
                    'code' => $status_code,
                    'confidence' => $confidence,
                ];
                
                // Save broken link to database
                if ($is_broken) {
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $wpdb->insert($table_name, [
                        'post_id' => $post->ID,
                        'url' => $url,
                        'anchor_text' => $anchor_text,
                        'context' => substr($post->post_content, max(0, strpos($post->post_content, $url) - 100), 200),
                        'status_code' => $status_code,
                        'confidence' => $confidence,
                        'last_checked' => current_time('mysql'),
                    ]);
                    $broken_count++;
                }
            }
        }
    }
    
    // Store scan results
    set_transient('prorank_scan_' . $scan_id, [
        'progress' => 100,
        'status' => sprintf(__('Scan complete. Found %1\$d broken links out of %2\$d total links.', 'prorank-seo'), $broken_count, $total_links),
        'completed' => true,
        'total_links' => $total_links,
        'broken_count' => $broken_count,
    ], 3600);
    
    return rest_ensure_response([
        'success' => true,
        'scan_id' => $scan_id,
        'message' => __('Broken links scan completed', 'prorank-seo'),
        'stats' => [
            'total_links' => $total_links,
            'broken_count' => $broken_count,
        ]
    ]);
}

/**
 * Ignore a broken link
 */
function ignore_broken_link(WP_REST_Request $request): WP_REST_Response {
    $link_id = $request->get_param('id');
    
    // In production, update database
    // For now, just return success
    
    return rest_ensure_response([
        'success' => true,
        'message' => __('Link ignored', 'prorank-seo'),
    ]);
}

/**
 * Delete broken links
 */
function delete_broken_links(WP_REST_Request $request): WP_REST_Response {
    $link_ids = $request->get_param('link_ids');
    
    if (empty($link_ids)) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('No links specified', 'prorank-seo'),
        ]);
    }
    
    // In production, delete from database
    // For now, just return success
    
    return rest_ensure_response([
        'success' => true,
        'message' => sprintf(
            /* translators: %s: numeric value */
            __('%d links deleted', 'prorank-seo'), count($link_ids)),
    ]);
}

/**
 * Get link map visualization data
 */
function get_linkmap_visualization_data(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    $post_types = explode(',', $request->get_param('post_types') ?: 'post,page');
    $max_nodes = (int) ($request->get_param('max_nodes') ?: 100);
    $depth = (int) ($request->get_param('depth') ?: 2);
    
    // Get all posts with their links
    $post_types_escaped = array_map('esc_sql', $post_types);
    $post_types_in = "'" . implode("','", $post_types_escaped) . "'";
    
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $query = $wpdb->prepare(
        "SELECT ID, post_title, post_type, post_status, post_date
         // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
         FROM {$wpdb->posts}
         // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
         WHERE post_type IN ($post_types_in)
         AND post_status = 'publish'
         ORDER BY post_date DESC
         LIMIT %d",
        $max_nodes
    );
    
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
    $posts = $wpdb->get_results($query);
    
    if (empty($posts)) {
        return rest_ensure_response([
            'success' => true,
            'data' => [
                'nodes' => [],
                'links' => [],
            ]
        ]);
    }
    
    $nodes = [];
    $links = [];
    $post_ids = [];
    
    // Create nodes from posts
    foreach ($posts as $post) {
        $post_ids[] = $post->ID;
        
        // Count internal links in this post
        $content = get_post_field('post_content', $post->ID);
        preg_match_all('/<a[^>]+href=[\'"]([^\'"]+)[\'"][^>]*>/i', $content, $matches);
        $internal_links_count = 0;
        $external_links_count = 0;
        
        if (!empty($matches[1])) {
            foreach ($matches[1] as $url) {
                if (strpos($url, home_url()) === 0 || strpos($url, '/') === 0) {
                    $internal_links_count++;
                } else if (strpos($url, 'http') === 0) {
                    $external_links_count++;
                }
            }
        }
        
        // Count inbound links (reuse the escaped post types from above)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $inbound_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT p.ID) 
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             FROM {$wpdb->posts} p
             WHERE p.post_status = 'publish'
             // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
             AND p.post_type IN ($post_types_in)
             AND p.ID != %d
             AND (p.post_content LIKE %s OR p.post_content LIKE %s)",
            $post->ID,
            '%href="' . get_permalink($post->ID) . '"%',
            '%href=\'' . get_permalink($post->ID) . '\'%'
        ));
        
        $nodes[] = [
            'id' => (int) $post->ID,
            'title' => $post->post_title,
            'name' => $post->post_title,
            'label' => $post->post_title,
            'type' => $post->post_type,
            'post_type' => $post->post_type,
            'url' => get_permalink($post->ID),
            'date' => $post->post_date,
            'inbound_links' => (int) $inbound_count,
            'outbound_links' => $internal_links_count,
            'outbound_internal' => $internal_links_count,
            'outbound_external' => $external_links_count,
            'is_orphan' => $inbound_count === 0,
            'orphan' => $inbound_count === 0,
            'size' => 5 + min(20, $inbound_count * 2), // Node size based on inbound links
            'color' => $post->post_type === 'post' ? '#667eea' : '#10b981',
        ];
    }
    
    // Find all links between these posts
    foreach ($posts as $source_post) {
        $content = get_post_field('post_content', $source_post->ID);
        
        foreach ($posts as $target_post) {
            if ($source_post->ID === $target_post->ID) continue;
            
            $target_url = get_permalink($target_post->ID);
            if (strpos($content, 'href="' . $target_url . '"') !== false || 
                strpos($content, "href='" . $target_url . "'") !== false) {
                
                $links[] = [
                    'source' => (string) $source_post->ID,
                    'target' => (string) $target_post->ID,
                    'type' => 'internal',
                ];
            }
        }
    }
    
    // Calculate link density (total links / total possible links)
    $node_count = count($nodes);
    $link_count = count($links);
    $possible_links = $node_count * ($node_count - 1); // Directed graph
    $link_density = $possible_links > 0 ? $link_count / $possible_links : 0;

    return rest_ensure_response([
        'success' => true,
        'data' => [
            'nodes' => $nodes,
            'links' => $links,
            // Use 'metrics' to match frontend expectations
            'metrics' => [
                'total_nodes' => $node_count,
                'total_links' => $link_count,
                'orphan_count' => count(array_filter($nodes, function($n) { return $n['orphan']; })),
                'link_density' => $link_density,
            ],
            // Keep 'stats' for backwards compatibility
            'stats' => [
                'total_nodes' => $node_count,
                'total_links' => $link_count,
                'orphan_count' => count(array_filter($nodes, function($n) { return $n['orphan']; })),
            ]
        ]
    ]);
}

/**
 * Get detailed information about a specific node
 */
function get_linkmap_node_details(WP_REST_Request $request): WP_REST_Response {
    $post_id = (int) $request->get_param('id');
    $post = get_post($post_id);
    
    if (!$post) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Post not found', 'prorank-seo'),
        ]);
    }
    
    global $wpdb;
    
    // Get all inbound links
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $inbound_links = $wpdb->get_results($wpdb->prepare(
        "SELECT ID, post_title, post_type
         // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
         FROM {$wpdb->posts}
         WHERE post_status = 'publish'
         AND ID != %d
         AND (post_content LIKE %s OR post_content LIKE %s)
         LIMIT 20",
        $post_id,
        '%href="' . get_permalink($post_id) . '"%',
        '%href=\'' . get_permalink($post_id) . '\'%'
    ));
    
    // Get all outbound links
    $content = $post->post_content;
    preg_match_all('/<a[^>]+href=[\'"]([^\'"]+)[\'"][^>]*>([^<]*)<\/a>/i', $content, $matches);
    
    $outbound_links = [];
    if (!empty($matches[1])) {
        foreach ($matches[1] as $i => $url) {
            $anchor_text = wp_strip_all_tags($matches[2][$i]);
            
            // Check if internal
            if (strpos($url, home_url()) === 0 || strpos($url, '/') === 0) {
                $post_id_from_url = url_to_postid($url);
                if ($post_id_from_url) {
                    $linked_post = get_post($post_id_from_url);
                    if ($linked_post) {
                        $outbound_links[] = [
                            'id' => $linked_post->ID,
                            'title' => $linked_post->post_title,
                            'type' => 'internal',
                            'anchor' => $anchor_text,
                            'url' => $url,
                        ];
                    }
                }
            } else if (strpos($url, 'http') === 0) {
                $outbound_links[] = [
                    'title' => $anchor_text ?: $url,
                    'type' => 'external',
                    'anchor' => $anchor_text,
                    'url' => $url,
                ];
            }
        }
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'id' => $post->ID,
            'title' => $post->post_title,
            'type' => $post->post_type,
            'status' => $post->post_status,
            'date' => $post->post_date,
            'url' => get_permalink($post->ID),
            'edit_url' => get_edit_post_link($post->ID, 'raw'),
            'word_count' => str_word_count(wp_strip_all_tags($content)),
            'inbound_links' => array_map(function($link) {
                return [
                    'id' => $link->ID,
                    'title' => $link->post_title,
                    'type' => $link->post_type,
                    'url' => get_permalink($link->ID),
                ];
            }, $inbound_links),
            'outbound_links' => $outbound_links,
            'stats' => [
                'inbound_count' => count($inbound_links),
                'outbound_internal_count' => count(array_filter($outbound_links, function($l) { 
                    return $l['type'] === 'internal'; 
                })),
                'outbound_external_count' => count(array_filter($outbound_links, function($l) { 
                    return $l['type'] === 'external'; 
                })),
            ]
        ]
    ]);
}

/**
 * Export link map data
 */
function export_linkmap_data(WP_REST_Request $request): WP_REST_Response {
    $format = $request->get_param('format') ?: 'json';
    
    // Get the link map data
    $data_request = new WP_REST_Request('GET');
    $data_request->set_param('post_types', $request->get_param('post_types'));
    $data_request->set_param('max_nodes', 1000); // Export more data
    $response = get_linkmap_visualization_data($data_request);
    $data = $response->get_data();
    
    if (!$data['success']) {
        return $response;
    }
    
    $export_data = $data['data'];
    
    // Generate filename
    $filename = 'link-map-' . gmdate('Y-m-d-His');
    
    if ($format === 'csv') {
        // Convert to CSV format
        $csv_data = "Source,Target,Type\n";
        foreach ($export_data['links'] as $link) {
            $source_node = array_filter($export_data['nodes'], function($n) use ($link) {
                return $n['id'] === $link['source'];
            });
            $target_node = array_filter($export_data['nodes'], function($n) use ($link) {
                return $n['id'] === $link['target'];
            });
            
            $source_title = $source_node ? reset($source_node)['label'] : 'Unknown';
            $target_title = $target_node ? reset($target_node)['label'] : 'Unknown';
            
            $csv_data .= '"' . str_replace('"', '""', $source_title) . '",';
            $csv_data .= '"' . str_replace('"', '""', $target_title) . '",';
            $csv_data .= $link['type'] . "\n";
        }
        
        return rest_ensure_response([
            'success' => true,
            'filename' => $filename . '.csv',
            'data' => base64_encode($csv_data),
            'mime_type' => 'text/csv',
        ]);
    }
    
    // Default to JSON
    return rest_ensure_response([
        'success' => true,
        'filename' => $filename . '.json',
        'data' => base64_encode(json_encode($export_data, JSON_PRETTY_PRINT)),
        'mime_type' => 'application/json',
    ]);
}

/**
 * Validate a URL returns a valid HTTP response (2xx/3xx)
 *
 * @param string $url URL to validate
 * @return array ['valid' => bool, 'status_code' => int, 'error' => string|null]
 */
function validate_target_url(string $url): array {
    // Check URL format
    if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
        return [
            'valid' => false,
            'status_code' => 0,
            'error' => __('Invalid URL format', 'prorank-seo'),
        ];
    }

    // Block dangerous protocols
    $parsed = wp_parse_url($url);
    if (!in_array($parsed['scheme'] ?? '', ['http', 'https'], true)) {
        return [
            'valid' => false,
            'status_code' => 0,
            'error' => __('Only HTTP/HTTPS URLs are allowed', 'prorank-seo'),
        ];
    }

    // Check if it's an internal URL (use url_to_postid)
    $site_url = get_site_url();
    if (strpos($url, $site_url) === 0) {
        $post_id = url_to_postid($url);
        if ($post_id > 0) {
            $post = get_post($post_id);
            if ($post && $post->post_status === 'publish') {
                return [
                    'valid' => true,
                    'status_code' => 200,
                    'error' => null,
                ];
            }
        }
        return [
            'valid' => false,
            'status_code' => 404,
            'error' => __('Internal page not found or not published', 'prorank-seo'),
        ];
    }

    // For external URLs, perform HTTP HEAD request
    $response = wp_remote_head($url, [
        'timeout' => 10,
        'redirection' => 5,
        'user-agent' => 'Mozilla/5.0 (compatible; ProRankSEO/1.0; +https://prorank.io)',
        'sslverify' => true,
    ]);

    if (is_wp_error($response)) {
        // Fallback to GET if HEAD fails
        $response = wp_remote_get($url, [
            'timeout' => 10,
            'redirection' => 5,
            'user-agent' => 'Mozilla/5.0 (compatible; ProRankSEO/1.0; +https://prorank.io)',
            'sslverify' => true,
        ]);

        if (is_wp_error($response)) {
            return [
                'valid' => false,
                'status_code' => 0,
                'error' => $response->get_error_message(),
            ];
        }
    }

    $status_code = wp_remote_retrieve_response_code($response);

    // Accept 2xx and 3xx status codes
    if ($status_code >= 200 && $status_code < 400) {
        return [
            'valid' => true,
            'status_code' => $status_code,
            'error' => null,
        ];
    }

    return [
        'valid' => false,
        'status_code' => $status_code,
        'error' => sprintf(
            /* translators: %s: numeric value */
            __('URL returned HTTP %d', 'prorank-seo'), $status_code),
    ];
}

/**
 * Fix a single broken link by replacing with new URL
 */
function fix_broken_link(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;

    // Verify nonce
    $nonce = $request->get_header('X-WP-Nonce');
    if (!wp_verify_nonce($nonce, 'wp_rest')) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Security check failed', 'prorank-seo'),
        ]);
    }

    // Verify capability
    if (!current_user_can('edit_posts')) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Permission denied', 'prorank-seo'),
        ]);
    }

    $link_id = (int) $request->get_param('id');
    $new_url = $request->get_param('new_url');
    $action = $request->get_param('action'); // 'replace', 'remove', 'unlink'

    if (!$link_id) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Invalid link ID', 'prorank-seo'),
        ]);
    }

    // Get broken link from database
    $table_name = $wpdb->prefix . 'prorank_broken_links';
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $broken_link = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$table_name} WHERE id = %d",
        $link_id
    ));

    if (!$broken_link) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Broken link not found', 'prorank-seo'),
        ]);
    }

    // If replacing with new URL, validate it first
    if ($action === 'replace' && !empty($new_url)) {
        $validation = validate_target_url($new_url);
        if (!$validation['valid']) {
            return rest_ensure_response([
                'success' => false,
                'message' => sprintf(
                    /* translators: %s: placeholder value */
                    __('New URL validation failed: %s', 'prorank-seo'),
                    $validation['error']
                ),
                'status_code' => $validation['status_code'],
            ]);
        }
    }

    // Get the post content
    $post = get_post($broken_link->post_id);
    if (!$post) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Post not found', 'prorank-seo'),
        ]);
    }

    // Check user can edit this post
    if (!current_user_can('edit_post', $post->ID)) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('You do not have permission to edit this post', 'prorank-seo'),
        ]);
    }

    $content = $post->post_content;
    $old_url = $broken_link->url;

    switch ($action) {
        case 'replace':
            // Replace old URL with new URL
            $content = str_replace(
                ['href="' . $old_url . '"', "href='" . $old_url . "'"],
                ['href="' . esc_url($new_url) . '"', "href='" . esc_url($new_url) . "'"],
                $content
            );
            break;

        case 'remove':
            // Remove the entire anchor tag but keep the anchor text
            $pattern = '/<a\s+[^>]*href=["\']' . preg_quote($old_url, '/') . '["\'][^>]*>(.*?)<\/a>/is';
            $content = preg_replace($pattern, '$1', $content);
            break;

        case 'unlink':
        default:
            // Same as remove - convert link to plain text
            $pattern = '/<a\s+[^>]*href=["\']' . preg_quote($old_url, '/') . '["\'][^>]*>(.*?)<\/a>/is';
            $content = preg_replace($pattern, '$1', $content);
            break;
    }

    // Update the post
    $result = wp_update_post([
        'ID' => $post->ID,
        'post_content' => $content,
    ], true);

    if (is_wp_error($result)) {
        return rest_ensure_response([
            'success' => false,
            'message' => $result->get_error_message(),
        ]);
    }

    // Mark the broken link as fixed in database
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
    $wpdb->update(
        $table_name,
        [
            'status' => 'fixed',
            'last_checked' => current_time('mysql'),
        ],
        ['id' => $link_id],
        ['%s', '%s'],
        ['%d']
    );

    return rest_ensure_response([
        'success' => true,
        'message' => sprintf(
            /* translators: %s: post title */
            __('Link fixed successfully in "%s"', 'prorank-seo'),
            $post->post_title
        ),
        'post_id' => $post->ID,
        'action' => $action,
    ]);
}

/**
 * Fix multiple broken links in batch
 */
function fix_broken_links_batch(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;

    // Verify nonce
    $nonce = $request->get_header('X-WP-Nonce');
    if (!wp_verify_nonce($nonce, 'wp_rest')) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Security check failed', 'prorank-seo'),
        ]);
    }

    // Verify capability
    if (!current_user_can('edit_posts')) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('Permission denied', 'prorank-seo'),
        ]);
    }

    $link_ids = $request->get_param('link_ids');
    $action = $request->get_param('action') ?: 'remove'; // 'remove', 'ignore'
    $new_urls = $request->get_param('new_urls') ?: []; // Map of link_id => new_url for replacements

    if (empty($link_ids) || !is_array($link_ids)) {
        return rest_ensure_response([
            'success' => false,
            'message' => __('No links specified', 'prorank-seo'),
        ]);
    }

    // Limit batch size to prevent timeout
    $link_ids = array_slice($link_ids, 0, 50);

    $results = [
        'fixed' => 0,
        'failed' => 0,
        'ignored' => 0,
        'errors' => [],
    ];

    $table_name = $wpdb->prefix . 'prorank_broken_links';

    foreach ($link_ids as $link_id) {
        $link_id = (int) $link_id;

        // Get broken link from database
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $broken_link = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE id = %d AND status = 'active'",
            $link_id
        ));

        if (!$broken_link) {
            $results['failed']++;
            $results['errors'][] = sprintf(
                /* translators: %s: numeric value */
                __('Link #%d not found', 'prorank-seo'), $link_id);
            continue;
        }

        if ($action === 'ignore') {
            // Just mark as ignored
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->update(
                $table_name,
                ['status' => 'ignored', 'last_checked' => current_time('mysql')],
                ['id' => $link_id],
                ['%s', '%s'],
                ['%d']
            );
            $results['ignored']++;
            continue;
        }

        // If we have a replacement URL, validate it
        if (isset($new_urls[$link_id]) && !empty($new_urls[$link_id])) {
            $validation = validate_target_url($new_urls[$link_id]);
            if (!$validation['valid']) {
                $results['failed']++;
                $results['errors'][] = sprintf(
                    /* translators: 1: link ID 2: error message */
                    __('Link #%1\$d: New URL validation failed - %2\$s', 'prorank-seo'),
                    $link_id,
                    $validation['error']
                );
                continue;
            }
        }

        // Get the post
        $post = get_post($broken_link->post_id);
        if (!$post || !current_user_can('edit_post', $post->ID)) {
            $results['failed']++;
            $results['errors'][] = sprintf(
                /* translators: %s: numeric value */
                __('Link #%d: Cannot edit post', 'prorank-seo'), $link_id);
            continue;
        }

        $content = $post->post_content;
        $old_url = $broken_link->url;

        // Determine action: replace if we have new URL, otherwise remove
        if (isset($new_urls[$link_id]) && !empty($new_urls[$link_id])) {
            $new_url = esc_url($new_urls[$link_id]);
            $content = str_replace(
                ['href="' . $old_url . '"', "href='" . $old_url . "'"],
                ['href="' . $new_url . '"', "href='" . $new_url . "'"],
                $content
            );
        } else {
            // Remove the link but keep anchor text
            $pattern = '/<a\s+[^>]*href=["\']' . preg_quote($old_url, '/') . '["\'][^>]*>(.*?)<\/a>/is';
            $content = preg_replace($pattern, '$1', $content);
        }

        // Update the post
        $result = wp_update_post([
            'ID' => $post->ID,
            'post_content' => $content,
        ], true);

        if (is_wp_error($result)) {
            $results['failed']++;
            $results['errors'][] = sprintf(
                /* translators: 1: link ID 2: status message */
                __('Link #%1\$d: %2\$s', 'prorank-seo'),
                $link_id,
                $result->get_error_message()
            );
            continue;
        }

        // Mark as fixed
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update(
            $table_name,
            ['status' => 'fixed', 'last_checked' => current_time('mysql')],
            ['id' => $link_id],
            ['%s', '%s'],
            ['%d']
        );

        $results['fixed']++;
    }

    return rest_ensure_response([
        'success' => true,
        'message' => sprintf(
            /* translators: 1: total links 2: fixed count 3: failed count 4: ignored count */
            __('Processed %1$d links: %2$d fixed, %3$d failed, %4$d ignored', 'prorank-seo'),
            count($link_ids),
            $results['fixed'],
            $results['failed'],
            $results['ignored']
        ),
        'results' => $results,
    ]);
}
