<?php
/**
 * Internal Linking REST API Controller (Free)
 *
 * Handles basic internal linking suggestions.
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\Content\InternalLinking;
use WP_REST_Request;

/**
 * LinkingController class
 */
class LinkingController extends BaseController {
    /**
     * Constructor
     */
    public function __construct() {
        $this->rest_base = 'linking';
    }

    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/basic-suggestions/(?P<post_id>\d+)',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_basic_suggestions'],
                    'permission_callback' => [$this, 'check_permission_edit_posts_basic'],
                    'args'                => [
                        'post_id' => [
                            'description' => __('Post ID to get suggestions for', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                            'minimum'     => 1,
                        ],
                    ],
                ],
            ]
        );
    }

    /**
     * Get basic internal linking suggestions
     *
     * @param WP_REST_Request $request Request object.
     * @return \WP_REST_Response|\WP_Error
     */
    public function get_basic_suggestions(WP_REST_Request $request) {
        return $this->try_callback(function() use ($request) {
            $post_id = (int) $request->get_param('post_id');

            $post = get_post($post_id);
            if (!$post) {
                return $this->get_error_response(
                    __('Post not found.', 'prorank-seo'),
                    404,
                    'post_not_found'
                );
            }

            $supported_post_types = apply_filters(
                'prorank_seo_internal_linking_post_types',
                ['post', 'page']
            );

            if (!is_array($supported_post_types)) {
                $supported_post_types = array_filter((array) $supported_post_types);
            }

            if (empty($supported_post_types)) {
                $supported_post_types = ['post', 'page'];
            }

            if (!in_array($post->post_type, $supported_post_types, true)) {
                return $this->get_error_response(
                    __('Post type not supported for internal linking.', 'prorank-seo'),
                    400,
                    'unsupported_post_type'
                );
            }

            if (!class_exists(InternalLinking::class)) {
                return $this->get_success_response([
                    'suggestions' => [],
                    'message' => __('Internal Linking engine is not available.', 'prorank-seo'),
                ]);
            }

            $post_content = $post->post_content;
            $post_title = $post->post_title;

            if (empty($post_content)) {
                return $this->get_success_response([
                    'suggestions' => [],
                    'message' => __('No content to analyze for suggestions.', 'prorank-seo'),
                ]);
            }

            $suggestions = InternalLinking::get_basic_suggestions(
                $post_id,
                $post_content,
                $post_title
            );

            $response_data = [
                'suggestions' => $suggestions,
                'stats' => [
                    'total_suggestions' => count($suggestions),
                    'link_density' => InternalLinking::get_link_density($post_content),
                ],
            ];

            if (empty($suggestions)) {
                $response_data['message'] = __(
                    'No internal linking suggestions found. Try adding more content or ensure you have other published posts.',
                    'prorank-seo'
                );
            }

            return $this->get_success_response($response_data);
        });
    }

    /**
     * Check permission for endpoints used by the editor sidebar.
     *
     * @return bool
     */
    public function check_permission_edit_posts_basic(): bool {
        $method = strtoupper( \prorank_get_server_var( 'REQUEST_METHOD' ) );
        if ( 'OPTIONS' === $method ) {
            return true;
        }

        return current_user_can('edit_posts');
    }
}
