<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * URL Cleanup REST API Controller
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WP_Error;

/**
 * URL Cleanup Controller class
 */
class UrlCleanupController extends WP_REST_Controller {
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->namespace = 'prorank-seo/v1';
        $this->rest_base = 'settings/url-cleanup';
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Get/Update URL cleanup settings
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => $this->get_settings_args(),
                ],
            ]
        );
        
        // Test URL cleanup
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/test',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'test_cleanup'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'url' => [
                            'required' => true,
                            'type' => 'string',
                            'format' => 'uri',
                        ],
                    ],
                ],
            ]
        );
        
        // Get 404 log
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/404-log',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_404_log'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::DELETABLE,
                    'callback'            => [$this, 'clear_404_log'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Bulk URL cleanup
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/bulk-cleanup',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'bulk_cleanup'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Detect redirect chains (Pro feature)
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/detect-chains',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'detect_redirect_chains'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Export 404 log as CSV
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/404-log/export',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'export_404_log'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
    }
    
    /**
     * Check permissions
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error
     */
    public function check_permission(WP_REST_Request $request) {
        if (!current_user_can('manage_options')) {
            return new WP_Error(
                'prorank_seo_rest_forbidden',
                __('You do not have permission to access this resource.', 'prorank-seo'),
                ['status' => 403]
            );
        }
        
        return true;
    }
    
    /**
     * Get URL cleanup settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_settings(WP_REST_Request $request) {
        try {
            $defaults = $this->get_default_settings();
            $settings = [];
            
            // Get individual settings
            foreach ($defaults as $key => $default) {
                $settings[$key] = get_option('prorank_seo_url_cleanup_' . $key, $default);
            }
            
            return rest_ensure_response([
                'success' => true,
                'settings' => $settings,
            ]);
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }
    
    /**
     * Update URL cleanup settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_settings(WP_REST_Request $request) {
        try {
            $settings = $request->get_params();
            
            // Validate settings
            $validated_settings = $this->validate_settings($settings);
            
            // Save individual settings for the module
            foreach ($validated_settings as $key => $value) {
                update_option('prorank_seo_url_cleanup_' . $key, $value);
            }
            
            // Flush rewrite rules if permalink structure changes
            if (
                isset($settings['strip_category_base']) || 
                isset($settings['strip_tag_base']) ||
                isset($settings['force_trailing_slash']) || 
                isset($settings['remove_trailing_slash'])
            ) {
                flush_rewrite_rules();
            }
            
            return rest_ensure_response([
                'success' => true,
                'message' => __('URL cleanup settings updated successfully.', 'prorank-seo'),
                'settings' => $validated_settings,
            ]);
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }
    
    /**
     * Test URL cleanup on a given URL
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function test_cleanup(WP_REST_Request $request) {
        try {
            $test_url = $request->get_param('url');
            
            // Get settings
            $defaults = $this->get_default_settings();
            $settings = [];
            foreach ($defaults as $key => $default) {
                $settings[$key] = get_option('prorank_seo_url_cleanup_' . $key, $default);
            }
            
            $original_url = $test_url;
            $cleaned_url = $test_url;
            $rules_applied = [];
            
            // Parse URL
            $parsed = wp_parse_url($test_url);
            if (!$parsed || !isset($parsed['host'])) {
                return new WP_Error(
                    'prorank_seo_invalid_url',
                    __('Invalid URL provided.', 'prorank-seo'),
                    ['status' => 400]
                );
            }
            
            // Remove UTM parameters
            if ($settings['remove_utm_parameters']) {
                if (isset($parsed['query'])) {
                    $query_params = [];
                    wp_parse_str($parsed['query'], $query_params);
                    
                    $utm_params = ['utm_source', 'utm_medium', 'utm_campaign', 'utm_term', 'utm_content'];
                    $original_count = count($query_params);
                    
                    foreach ($utm_params as $param) {
                        unset($query_params[$param]);
                    }
                    
                    if (count($query_params) < $original_count) {
                        $rules_applied[] = __('Removed UTM parameters', 'prorank-seo');
                        $parsed['query'] = http_build_query($query_params);
                    }
                }
            }
            
            // Apply trailing slash rules
            if (isset($parsed['path']) && $parsed['path'] !== '/') {
                $has_trailing_slash = substr($parsed['path'], -1) === '/';
                
                if ($settings['force_trailing_slash'] && !$has_trailing_slash) {
                    $parsed['path'] .= '/';
                    $rules_applied[] = __('Added trailing slash', 'prorank-seo');
                } elseif ($settings['remove_trailing_slash'] && $has_trailing_slash) {
                    $parsed['path'] = rtrim($parsed['path'], '/');
                    $rules_applied[] = __('Removed trailing slash', 'prorank-seo');
                }
            }
            
            // Convert to lowercase
            if ($settings['convert_to_lowercase']) {
                if (isset($parsed['path'])) {
                    $lowercase_path = strtolower($parsed['path']);
                    if ($lowercase_path !== $parsed['path']) {
                        $parsed['path'] = $lowercase_path;
                        $rules_applied[] = __('Converted URL to lowercase', 'prorank-seo');
                    }
                }
            }
            
            // Rebuild URL
            $cleaned_url = $this->build_url($parsed);
            
            return rest_ensure_response([
                'success' => true,
                'original_url' => $original_url,
                'cleaned_url' => $cleaned_url,
                'changes' => $rules_applied,  // Frontend expects 'changes' property
                'redirect_needed' => $original_url !== $cleaned_url,
            ]);
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }
    
    /**
     * Get 404 error log
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_404_log(WP_REST_Request $request) {
        try {
            $log = get_option('prorank_seo_404_log', []);

            // Transform log entries with consistent naming and add IDs
            $log = array_map(function($item) {
                return [
                    'id' => md5($item['url']),
                    'url' => $item['url'],
                    'hits' => $item['count'] ?? $item['hits'] ?? 1,
                    'referrer' => $item['referrer'] ?? '',
                    'last_hit' => $item['last_hit'] ?? $item['timestamp'] ?? '',
                    'user_agent' => $item['user_agent'] ?? '',
                ];
            }, $log);

            // Sort by hits (most frequent first)
            usort($log, function($a, $b) {
                return $b['hits'] - $a['hits'];
            });

            // Get total before pagination
            $total = count($log);

            // Apply pagination if params provided
            $page = (int) ($request->get_param('page') ?? 1);
            $per_page = (int) ($request->get_param('per_page') ?? 10);

            // Validate pagination params
            $page = max(1, $page);
            $per_page = max(1, min(100, $per_page)); // Cap at 100 per page

            $offset = ($page - 1) * $per_page;
            $paginated_log = array_slice($log, $offset, $per_page);

            return rest_ensure_response([
                'success' => true,
                'log' => $paginated_log,
                'total' => $total,
                'page' => $page,
                'per_page' => $per_page,
                'total_pages' => (int) ceil($total / $per_page),
            ]);
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }
    
    /**
     * Clear 404 error log (all or individual entry)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function clear_404_log(WP_REST_Request $request) {
        try {
            $log = get_option('prorank_seo_404_log', []);

            // Check for clear_all parameter (clear entire log)
            if ($request->get_param('clear_all')) {
                delete_option('prorank_seo_404_log');
                return rest_ensure_response([
                    'success' => true,
                    'message' => __('All 404 logs cleared successfully.', 'prorank-seo'),
                ]);
            }

            // Check for individual entry deletion by ID (md5 hash of URL)
            $id = $request->get_param('id');
            if ($id) {
                $original_count = count($log);
                $log = array_filter($log, function($item) use ($id) {
                    return md5($item['url']) !== $id;
                });

                if (count($log) < $original_count) {
                    update_option('prorank_seo_404_log', array_values($log));
                    return rest_ensure_response([
                        'success' => true,
                        'message' => __('404 entry deleted successfully.', 'prorank-seo'),
                    ]);
                }

                return new WP_Error(
                    'prorank_seo_entry_not_found',
                    __('404 entry not found.', 'prorank-seo'),
                    ['status' => 404]
                );
            }

            // No parameters provided - return error
            return new WP_Error(
                'prorank_seo_invalid_params',
                __('Missing id or clear_all parameter.', 'prorank-seo'),
                ['status' => 400]
            );
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }
    
    /**
     * Bulk cleanup URLs
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function bulk_cleanup(WP_REST_Request $request) {
        try {
            global $wpdb;
            
            $settings = get_option('prorank_seo_url_cleanup_settings', $this->get_default_settings());
            $processed = 0;
            $updated = 0;
            
            // Get all posts and pages
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $posts = $wpdb->get_results(
                "SELECT ID, post_name, post_type FROM {$wpdb->posts} 
                 WHERE post_status = 'publish' 
                 AND post_type IN ('post', 'page')"
            );
            
            foreach ($posts as $post) {
                $processed++;
                $new_slug = $post->post_name;
                $changed = false;
                
                // Remove stop words if enabled
                if (!empty($settings['remove_stop_words'])) {
                    $stop_words = [
                        'a', 'an', 'and', 'are', 'as', 'at', 'be', 'by', 'for', 'from',
                        'has', 'he', 'in', 'is', 'it', 'its', 'of', 'on', 'that', 'the',
                        'to', 'was', 'will', 'with', 'this', 'but', 'they', 'have'
                    ];
                    
                    $words = explode('-', $new_slug);
                    $filtered = array_filter($words, function($word) use ($stop_words) {
                        return !in_array(strtolower($word), $stop_words);
                    });
                    
                    $filtered_slug = implode('-', $filtered);
                    if ($filtered_slug !== $new_slug && !empty($filtered_slug)) {
                        $new_slug = $filtered_slug;
                        $changed = true;
                    }
                }
                
                // Convert to lowercase if enabled
                if (!empty($settings['convert_to_lowercase'])) {
                    $lower_slug = strtolower($new_slug);
                    if ($lower_slug !== $new_slug) {
                        $new_slug = $lower_slug;
                        $changed = true;
                    }
                }
                
                // Update if changed
                if ($changed) {
                    // Make sure slug is unique
                    $new_slug = wp_unique_post_slug($new_slug, $post->ID, 'publish', $post->post_type, 0);
                    
                    $wpdb->update(
                        $wpdb->posts,
                        ['post_name' => $new_slug],
                        ['ID' => $post->ID]
                    );
                    
                    $updated++;
                }
            }
            
            // Flush rewrite rules
            if ($updated > 0) {
                flush_rewrite_rules();
            }
            
            return rest_ensure_response([
                'success' => true,
                'message' => sprintf(
                    /* translators: 1: processed count 2: updated count */
                    __('Bulk cleanup completed. Processed %1$d URLs, updated %2$d.', 'prorank-seo'),
                    $processed,
                    $updated
                ),
                'processed' => $processed,
                'updated' => $updated,
            ]);
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }

    /**
     * Detect redirect chains (Pro feature)
     *
     * Scans all redirects and identifies chains that could be consolidated
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function detect_redirect_chains(WP_REST_Request $request) {
        try {
            // Get all redirects from the Redirect Manager
            $redirects = get_option('prorank_seo_redirects', []);

            if (empty($redirects)) {
                return rest_ensure_response([
                    'success' => true,
                    'chains' => [],
                    'message' => __('No redirects found to analyze.', 'prorank-seo'),
                ]);
            }

            // Build a lookup map for quick source-to-destination resolution
            $redirect_map = [];
            foreach ($redirects as $redirect) {
                $source = rtrim($redirect['source'] ?? '', '/');
                $destination = $redirect['destination'] ?? '';
                if ($source && $destination) {
                    $redirect_map[$source] = $destination;
                }
            }

            $chains = [];
            $visited = [];

            foreach ($redirects as $redirect) {
                $source = rtrim($redirect['source'] ?? '', '/');

                // Skip if already processed
                if (isset($visited[$source])) {
                    continue;
                }

                // Trace the chain
                $chain = [$source];
                $current = $source;
                $max_hops = 10; // Prevent infinite loops
                $hop_count = 0;

                while (isset($redirect_map[$current]) && $hop_count < $max_hops) {
                    $next = $redirect_map[$current];
                    $next_normalized = rtrim(wp_parse_url($next, PHP_URL_PATH) ?? $next, '/');

                    // Check for circular redirect
                    if (in_array($next_normalized, $chain)) {
                        $chain[] = $next_normalized . ' (circular)';
                        break;
                    }

                    $chain[] = $next;
                    $visited[$current] = true;
                    $current = $next_normalized;
                    $hop_count++;
                }

                // Only report chains with more than 1 hop (more than 2 URLs in chain)
                if (count($chain) > 2) {
                    $chains[] = [
                        'id' => md5(implode('|', $chain)),
                        'chain' => $chain,
                        'hops' => count($chain) - 1,
                        'source' => $chain[0],
                        'final_destination' => end($chain),
                        'suggestion' => sprintf(
                            /* translators: 1: source URL 2: destination URL */
                            __('Consolidate: %1$s → %2$s', 'prorank-seo'),
                            $chain[0],
                            end($chain)
                        ),
                    ];
                }
            }

            // Sort by number of hops (longest chains first)
            usort($chains, function($a, $b) {
                return $b['hops'] - $a['hops'];
            });

            return rest_ensure_response([
                'success' => true,
                'chains' => $chains,
                'total' => count($chains),
                'message' => count($chains) > 0
                    ? sprintf(
                        /* translators: %s: numeric value */
                        __('Found %d redirect chain(s) that can be optimized.', 'prorank-seo'), count($chains))
                    : __('No redirect chains detected. Your redirects are optimized!', 'prorank-seo'),
            ]);
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }

    /**
     * Export 404 log as CSV
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function export_404_log(WP_REST_Request $request) {
        try {
            $log = get_option('prorank_seo_404_log', []);

            if (empty($log)) {
                return new WP_Error(
                    'prorank_seo_no_data',
                    __('No 404 logs to export.', 'prorank-seo'),
                    ['status' => 404]
                );
            }

            // Transform and prepare CSV data
            $csv_data = [];
            $csv_data[] = ['URL', 'Hits', 'Referrer', 'Last Hit', 'User Agent'];

            foreach ($log as $item) {
                $csv_data[] = [
                    $item['url'] ?? '',
                    $item['count'] ?? $item['hits'] ?? 1,
                    $item['referrer'] ?? '',
                    $item['last_hit'] ?? $item['timestamp'] ?? '',
                    $item['user_agent'] ?? '',
                ];
            }

            // Convert to CSV string
            $csv_string = '';
            foreach ($csv_data as $row) {
                $csv_string .= implode(',', array_map(function($cell) {
                    // Escape quotes and wrap in quotes if contains comma or quote
                    $cell = str_replace('"', '""', $cell);
                    if (strpos($cell, ',') !== false || strpos($cell, '"') !== false || strpos($cell, "\n") !== false) {
                        $cell = '"' . $cell . '"';
                    }
                    return $cell;
                }, $row)) . "\n";
            }

            return rest_ensure_response([
                'success' => true,
                'csv' => $csv_string,
                'filename' => 'prorank-404-log-' . gmdate('Y-m-d') . '.csv',
                'total_entries' => count($log),
            ]);
        } catch (\Exception $e) {
            return new WP_Error(
                'prorank_seo_rest_error',
                esc_html($e->getMessage()),
                ['status' => 500]
            );
        }
    }

    /**
     * Get default settings
     *
     * @return array
     */
    private function get_default_settings(): array {
        return [
            'strip_category_base' => false,
            'strip_tag_base' => false,
            'remove_replytocom' => false,
            'remove_utm_parameters' => false,
            'utm_to_fragment' => false,
            'force_trailing_slash' => false,
            'remove_trailing_slash' => false,
            'convert_to_lowercase' => false,
            'remove_stop_words' => false,
            'monitor_404' => false,
            'attachment_redirect' => 'off',
            'author_archive_handling' => 'default',
            'date_archive_handling' => 'default',
            'remove_query_strings' => false,
            'remove_feed_links' => false,
            'remove_shortlink' => false,
            'remove_rest_api_links' => false,
            'clean_permalinks' => false,
            'param_whitelist' => 'gclid,fbclid,msclkid,_ga',
        ];
    }
    
    /**
     * Validate settings
     *
     * @param array $settings Settings to validate
     * @return array Validated settings
     */
    private function validate_settings(array $settings): array {
        $defaults = $this->get_default_settings();
        $validated = [];

        // Validate boolean settings
        foreach ($defaults as $key => $default_value) {
            if (is_bool($default_value)) {
                $validated[$key] = isset($settings[$key]) ? (bool) $settings[$key] : $default_value;
            }
        }

        // Validate attachment_redirect
        $valid_attachment_options = ['off', 'parent', 'file'];
        $validated['attachment_redirect'] = isset($settings['attachment_redirect']) && in_array($settings['attachment_redirect'], $valid_attachment_options)
            ? $settings['attachment_redirect']
            : 'off';

        // Validate archive handling settings
        $valid_archive_options = ['default', 'redirect_home', 'disable_404', 'noindex'];

        $validated['author_archive_handling'] = isset($settings['author_archive_handling']) && in_array($settings['author_archive_handling'], $valid_archive_options)
            ? $settings['author_archive_handling']
            : 'default';

        $validated['date_archive_handling'] = isset($settings['date_archive_handling']) && in_array($settings['date_archive_handling'], $valid_archive_options)
            ? $settings['date_archive_handling']
            : 'default';

        // Ensure trailing slash settings are mutually exclusive
        if ($validated['force_trailing_slash'] && $validated['remove_trailing_slash']) {
            $validated['remove_trailing_slash'] = false;
        }

        // Validate param_whitelist (comma-separated string)
        $validated['param_whitelist'] = isset($settings['param_whitelist'])
            ? sanitize_text_field($settings['param_whitelist'])
            : 'gclid,fbclid,msclkid,_ga';

        return $validated;
    }
    
    /**
     * Build URL from parsed components
     *
     * @param array $parsed Parsed URL components
     * @return string
     */
    private function build_url(array $parsed): string {
        $url = '';
        
        if (isset($parsed['scheme'])) {
            $url .= $parsed['scheme'] . '://';
        }
        
        if (isset($parsed['host'])) {
            $url .= $parsed['host'];
        }
        
        if (isset($parsed['port'])) {
            $url .= ':' . $parsed['port'];
        }
        
        if (isset($parsed['path'])) {
            $url .= $parsed['path'];
        }
        
        if (isset($parsed['query']) && !empty($parsed['query'])) {
            $url .= '?' . $parsed['query'];
        }
        
        if (isset($parsed['fragment'])) {
            $url .= '#' . $parsed['fragment'];
        }
        
        return $url;
    }
    
    /**
     * Get settings args for REST API
     *
     * @return array
     */
    private function get_settings_args(): array {
        return [
            'strip_category_base' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'strip_tag_base' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'remove_replytocom' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'remove_utm_parameters' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'force_trailing_slash' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'remove_trailing_slash' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'convert_to_lowercase' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'remove_stop_words' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'monitor_404' => [
                'type' => 'boolean',
                'default' => false,
            ],
            'attachment_redirect' => [
                'type' => 'string',
                'default' => 'off',
                'enum' => ['off', 'parent', 'file'],
            ],
            'author_archive_handling' => [
                'type' => 'string',
                'default' => 'default',
                'enum' => ['default', 'redirect_home', 'disable_404', 'noindex'],
            ],
            'date_archive_handling' => [
                'type' => 'string',
                'default' => 'default',
                'enum' => ['default', 'redirect_home', 'disable_404', 'noindex'],
            ],
        ];
    }
}
