<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * URL Preview REST Controller
 *
 * Provides URL preview and migration functionality for eCommerce SEO.
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   2.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;

/**
 * UrlPreviewController class
 */
class UrlPreviewController extends BaseController {
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        $this->namespace = 'prorank-seo/v1';
        $this->rest_base = 'url-preview';
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Preview URLs endpoint
        register_rest_route($this->namespace, '/' . $this->rest_base, [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_url_preview'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);
        
        // Analyze URLs endpoint
        register_rest_route($this->namespace, '/' . $this->rest_base . '/analyze', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'analyze_urls'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);
        
        // Migration status endpoint
        register_rest_route($this->namespace, '/' . $this->rest_base . '/migration-status', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_migration_status'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);
    }
    
    /**
     * Get URL preview
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_url_preview(WP_REST_Request $request): WP_REST_Response {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            return rest_ensure_response([
                'success' => false,
                'message' => __('WooCommerce is not active.', 'prorank-seo'),
            ]);
        }
        
        $settings = $request->get_params();
        
        $remove_product_base = !empty($settings['remove_product_base']) && $settings['remove_product_base'] !== '0';
        $remove_category_base = !empty($settings['remove_product_category_base']) && $settings['remove_product_category_base'] !== '0';
        
        $previews = [
            'products' => [],
            'categories' => [],
        ];
        
        // Get sample products
        if (function_exists('wc_get_products')) {
            $products = wc_get_products([
                'limit' => 3,
                'status' => 'publish',
            ]);
        } else {
            $products = [];
        }
        
        foreach ($products as $product) {
            $current_url = get_permalink($product->get_id());
            $new_url = $current_url;
            
            if ($remove_product_base) {
                $new_url = str_replace('/product/', '/', $new_url);
            }
            
            $previews['products'][] = [
                'title' => $product->get_name(),
                'current_url' => $current_url,
                'new_url' => $new_url,
                'will_redirect' => $current_url !== $new_url,
            ];
        }
        
        // Get sample categories
        $categories = get_terms([
            'taxonomy' => 'product_cat',
            'number' => 3,
            'hide_empty' => false,
        ]);
        
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $current_url = get_term_link($category);
                if (!is_wp_error($current_url)) {
                    $new_url = $current_url;
                    
                    if ($remove_category_base) {
                        $new_url = str_replace('/product-category/', '/', $new_url);
                    }
                    
                    $previews['categories'][] = [
                        'title' => $category->name,
                        'current_url' => $current_url,
                        'new_url' => $new_url,
                        'will_redirect' => $current_url !== $new_url,
                    ];
                }
            }
        }
        
        return rest_ensure_response([
            'success' => true,
            'data' => $previews,
        ]);
    }
    
    /**
     * Analyze current URL structure
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function analyze_urls(WP_REST_Request $request): WP_REST_Response {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            return rest_ensure_response([
                'success' => false,
                'message' => __('WooCommerce is not active.', 'prorank-seo'),
            ]);
        }
        
        global $wpdb;
        
        // Count products
        $product_count = wp_count_posts('product');
        $total_products = $product_count->publish ?? 0;
        
        // Count categories
        $category_count = wp_count_terms('product_cat');
        
        // Check current permalink structure
        $permalink_structure = get_option('permalink_structure');
        $wc_permalinks = get_option('woocommerce_permalinks', []);
        
        // Analyze URL lengths
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $long_urls = $wpdb->get_var("
            SELECT COUNT(*) 
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            FROM {$wpdb->posts} 
            WHERE post_type = 'product' 
            AND post_status = 'publish'
            AND LENGTH(post_name) > 60
        ");
        
        // Check for stop words in URLs
        $stop_words = ['the', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'from', 'as', 'is', 'was', 'are', 'been'];
        $stop_words_pattern = implode('|', array_map('preg_quote', $stop_words));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $urls_with_stop_words = $wpdb->get_var("
            SELECT COUNT(*) 
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            FROM {$wpdb->posts} 
            WHERE post_type = 'product' 
            AND post_status = 'publish'
            AND post_name REGEXP '-({$stop_words_pattern})-'
        ");
        
        // Check for special characters
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $urls_with_special_chars = $wpdb->get_var("
            SELECT COUNT(*) 
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            FROM {$wpdb->posts} 
            WHERE post_type = 'product' 
            AND post_status = 'publish'
            AND post_name REGEXP '[^a-z0-9-]'
        ");
        
        $analysis = [
            'total_products' => $total_products,
            'total_categories' => $category_count,
            'permalink_structure' => $permalink_structure,
            'product_base' => $wc_permalinks['product_base'] ?? 'product',
            'category_base' => $wc_permalinks['category_base'] ?? 'product-category',
            'issues' => [
                'long_urls' => $long_urls,
                'urls_with_stop_words' => $urls_with_stop_words,
                'urls_with_special_chars' => $urls_with_special_chars,
            ],
            'recommendations' => [],
        ];
        
        // Add recommendations
        if ($long_urls > 0) {
            $analysis['recommendations'][] = sprintf(
                /* translators: %s: placeholder value */
                __('%d products have URLs longer than 60 characters. Consider shortening them.', 'prorank-seo'),
                $long_urls
            );
        }
        
        if ($urls_with_stop_words > 0) {
            $analysis['recommendations'][] = sprintf(
                /* translators: %s: placeholder value */
                __('%d products have stop words in their URLs. Removing them can improve SEO.', 'prorank-seo'),
                $urls_with_stop_words
            );
        }
        
        if (!empty($wc_permalinks['product_base']) && $wc_permalinks['product_base'] !== '') {
            $analysis['recommendations'][] = __('Consider removing the product base from URLs for cleaner permalinks.', 'prorank-seo');
        }
        
        if (!empty($wc_permalinks['category_base']) && $wc_permalinks['category_base'] !== '') {
            $analysis['recommendations'][] = __('Consider removing the category base from URLs for better hierarchy.', 'prorank-seo');
        }
        
        return rest_ensure_response([
            'success' => true,
            'data' => $analysis,
        ]);
    }
    
    /**
     * Get migration status
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_migration_status(WP_REST_Request $request): WP_REST_Response {
        // Check if migration is needed
        $current_product_base = get_option('prorank_seo_remove_product_base', false);
        $current_category_base = get_option('prorank_seo_remove_product_category_base', false);
        
        // Count affected URLs
        $product_count = 0;
        $category_count = 0;
        
        if ($current_product_base) {
            $product_count = wp_count_posts('product')->publish ?? 0;
        }
        
        if ($current_category_base) {
            $category_count = wp_count_terms('product_cat');
        }
        
        // Check for existing redirects
        global $wpdb;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $redirect_count = $wpdb->get_var("
            SELECT COUNT(*) 
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_prorank_seo_redirect_from'
        ");
        
        return rest_ensure_response([
            'success' => true,
            'data' => [
                'products_affected' => $product_count,
                'categories_affected' => $category_count,
                'existing_redirects' => $redirect_count,
                'flush_needed' => get_option('prorank_seo_flush_rules', false),
                'last_migration' => get_option('prorank_seo_last_url_migration', null),
            ],
        ]);
    }
}