<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Visual Link Map Data Endpoint
 * 
 * Provides real link structure data for visualization
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;

// Register the endpoint
add_action('rest_api_init', function() {
    register_rest_route('prorank-seo/v1', '/linking/visual-map-data', [
        [
            'methods' => WP_REST_Server::READABLE,
            'callback' => __NAMESPACE__ . '\get_visual_map_data',
            'permission_callback' => function() {
                return current_user_can('edit_posts');
            },
            'args' => [
                'post_types' => [
                    'default' => ['post', 'page'],
                ],
                'max_nodes' => [
                    'default' => 100,
                    'sanitize_callback' => 'absint',
                ],
                'depth' => [
                    'default' => 2,
                    'sanitize_callback' => 'absint',
                ],
            ],
        ]
    ]);
});

/**
 * Get visual map data for link structure
 */
function get_visual_map_data(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    $post_types = $request->get_param('post_types');
    $max_nodes = min((int) $request->get_param('max_nodes'), 500);
    $depth = min((int) $request->get_param('depth'), 3);
    
    // Get all published posts/pages
    $posts = get_posts([
        'post_type' => $post_types,
        'post_status' => 'publish',
        'numberposts' => $max_nodes,
        'orderby' => 'menu_order date',
        'order' => 'DESC',
    ]);
    
    $nodes = [];
    $links = [];
    $processed_links = [];
    
    // Create nodes from posts
    foreach ($posts as $post) {
        // Count internal links in content
        $content = $post->post_content;
        $site_url = get_site_url();
        
        // Find all internal links in the content
        preg_match_all('/<a[^>]+href=["\']([^"\']+)["\'][^>]*>([^<]*)<\/a>/i', $content, $matches);
        
        $outbound_links = 0;
        $inbound_links = 0;
        
        // Process each link found
        if (!empty($matches[1])) {
            foreach ($matches[1] as $index => $url) {
                // Check if it's an internal link
                if (strpos($url, $site_url) !== false || strpos($url, '/') === 0) {
                    $outbound_links++;
                    
                    // Try to find the target post
                    $target_id = url_to_postid($url);
                    if ($target_id && $target_id !== $post->ID) {
                        // Create link relationship
                        $link_key = $post->ID . '-' . $target_id;
                        if (!isset($processed_links[$link_key])) {
                            $links[] = [
                                'source' => $post->ID,
                                'target' => $target_id,
                                'strength' => 1,
                                'anchor' => wp_strip_all_tags($matches[2][$index] ?? ''),
                            ];
                            $processed_links[$link_key] = true;
                        }
                    }
                }
            }
        }
        
        // Check if this post is orphaned (no incoming links)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $incoming_links_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND post_type IN ('post', 'page')
            AND ID != %d
            AND (post_content LIKE %s OR post_content LIKE %s)",
            $post->ID,
            '%' . get_permalink($post->ID) . '%',
            '%/?p=' . $post->ID . '%'
        )) ?: 0;
        
        // Calculate link equity (simplified scoring)
        $link_equity = min(100, ($outbound_links * 10) + ($incoming_links_count * 15) + 20);
        
        // Add node
        $nodes[] = [
            'id' => $post->ID,
            'title' => $post->post_title,
            'type' => $post->post_type,
            'url' => get_permalink($post->ID),
            'inboundLinks' => $incoming_links_count,
            'outboundLinks' => $outbound_links,
            'linkEquity' => $link_equity,
            'isOrphan' => $incoming_links_count === 0,
            'group' => $post->post_type === 'page' ? 1 : 2,
            'size' => 5 + sqrt($link_equity) * 2,
        ];
    }
    
    // Add some statistical data
    $total_posts = count($nodes);
    $orphan_count = count(array_filter($nodes, function($n) { return $n['isOrphan']; }));
    $total_links = count($links);
    
    $stats = [
        'total_nodes' => $total_posts,
        'total_links' => $total_links,
        'orphan_count' => $orphan_count,
        'orphan_percentage' => $total_posts > 0 ? round(($orphan_count / $total_posts) * 100, 1) : 0,
        'average_links_per_post' => $total_posts > 0 ? round($total_links / $total_posts, 1) : 0,
    ];
    
    return rest_ensure_response([
        'success' => true,
        'data' => [
            'nodes' => $nodes,
            'links' => $links,
            'stats' => $stats,
        ],
    ]);
}