<?php
/**
 * Schema Enhancement Default Settings
 *
 * Sets default values for AI enhancement settings on activation
 *
 * @package ProRank\SEO\Core\Schema
 * @since   2.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Schema;

defined( 'ABSPATH' ) || exit;

/**
 * SchemaEnhancementDefaults class
 */
class SchemaEnhancementDefaults {
    
    /**
     * Default settings for AI enhancements
     */
    private const DEFAULT_SETTINGS = [
        'prorank_seo_enable_ai_schema_enhancement' => true,
        'prorank_seo_ai_detect_brand' => true,
        'prorank_seo_ai_generate_descriptions' => true,
        'prorank_seo_ai_extract_features' => true,
        'prorank_seo_ai_generate_faq' => true,
        'prorank_seo_ai_detect_energy' => true,
        'prorank_seo_ai_detect_sustainability' => true,
        'prorank_seo_ai_detect_warranty' => true,
    ];
    
    /**
     * Initialize default settings
     *
     * @return void
     */
    public static function init(): void {
        // Set defaults if not already set
        foreach (self::DEFAULT_SETTINGS as $option_name => $default_value) {
            if (get_option($option_name, null) === null) {
                add_option($option_name, $default_value);
            }
        }
    }
    
    /**
     * Reset to defaults
     *
     * @return void
     */
    public static function reset(): void {
        foreach (self::DEFAULT_SETTINGS as $option_name => $default_value) {
            update_option($option_name, $default_value);
        }
    }
    
    /**
     * Get default value for a setting
     *
     * @param string $setting_key Setting key without prefix.
     * @return mixed Default value or null if not found.
     */
    public static function get_default(string $setting_key): mixed {
        $full_key = 'prorank_seo_' . $setting_key;
        return self::DEFAULT_SETTINGS[$full_key] ?? null;
    }
}