<?php
/**
 * Lightweight Speculation Rules output
 *
 * Outputs safe prefetch rules for faster client-side navigation.
 *
 * @package ProRank\SEO\Frontend
 */

declare(strict_types=1);

namespace ProRank\SEO\Frontend;

defined( 'ABSPATH' ) || exit;

/**
 * SpeculationRules
 */
class SpeculationRules {
    /**
     * Boot hooks
     */
    public function init(): void {
        add_action( 'wp_head', [ $this, 'output_rules' ], 1 );
    }

    /**
     * Output speculationrules script
     */
    public function output_rules(): void {
        if ( is_admin() || is_preview() || is_customize_preview() ) {
            return;
        }

        // Basic rules: prefetch document links without query strings; safe excludes
        $rules = [
            'prefetch' => [
                [
                    'source'    => 'document',
                    'where'     => [
                        'and' => array_values( array_filter( [
                            [ 'href_matches' => '/*' ],
                            [ 'not' => [ 'href_matches' => '/*\\?*' ] ],
                            [ 'not' => [ 'selector_matches' => '[data-no-prefetch], [rel~=nofollow]' ] ],
                            $this->woo_excludes(),
                        ] ) ),
                    ],
                    // moderate = hover with small delay; safest default
                    'eagerness' => 'moderate',
                ],
            ],
        ];

        $json = wp_json_encode( $rules, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
        if ( empty( $json ) ) {
            return;
        }

        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- JSON from wp_json_encode is safe
        echo '<script type="speculationrules">' . "\n" . $json . "\n" . '</script>' . "\n";
    }

    /**
     * Exclude WooCommerce carts/checkout from prefetch
     */
    private function woo_excludes() {
        if ( class_exists( 'WooCommerce' ) ) {
            return [
                'not' => [
                    'href_matches' => '/{cart,checkout,my-account}/*',
                ],
            ];
        }
        return null;
    }
}
