<?php
/**
 * Third-Party Script Defer/INP Guard
 *
 * Adds defer + data markers to heavy third-party scripts (analytics/ads/chat),
 * to reduce main-thread contention and improve INP.
 *
 * @package ProRank\SEO\Frontend
 */

declare(strict_types=1);

namespace ProRank\SEO\Frontend;

defined( 'ABSPATH' ) || exit;

class ThirdPartyDefer {
    /**
     * Boot hooks
     */
    public function init(): void {
        add_filter( 'script_loader_tag', [ $this, 'defer_third_party' ], 10, 3 );
    }

    /**
     * Add defer + marker to known third-party scripts
     */
    public function defer_third_party( string $tag, string $handle, string $src ): string {
        $domains = [
            'googletagmanager.com',
            'google-analytics.com',
            'gtag/js',
            'doubleclick.net',
            'facebook.net',
            'connect.facebook.net',
            'static.hotjar.com',
            'cdn.segment.com',
            'js.hs-analytics.net',
            'cdn.mouseflow.com',
            'cdn.fullstory.com',
            'snap.licdn.com',
            'js.adsrvr.org',
        ];

        $handles = [
            'gtag', 'google-analytics', 'ga-', 'gtm-', 'fbq', 'hotjar', 'segment',
            'clarity', 'hs-analytics', 'mouseflow', 'fullstory', 'linkedin_insight',
        ];

        foreach ( $domains as $domain ) {
            if ( strpos( $src, $domain ) !== false ) {
                // ensure defer attribute
                if ( strpos( $tag, 'defer' ) === false ) {
                    $tag = str_replace( '<script ', '<script defer ', $tag );
                }
                if ( strpos( $tag, 'data-prorank-3p' ) === false ) {
                    $tag = str_replace( '<script ', '<script data-prorank-3p="true" ', $tag );
                }
                break;
            }
        }

        foreach ( $handles as $h ) {
            if ( strpos( $handle, $h ) !== false ) {
                if ( strpos( $tag, 'defer' ) === false ) {
                    $tag = str_replace( '<script ', '<script defer ', $tag );
                }
                if ( strpos( $tag, 'data-prorank-3p' ) === false ) {
                    $tag = str_replace( '<script ', '<script data-prorank-3p="true" ', $tag );
                }
                break;
            }
        }

        return $tag;
    }
}
