<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * Author Sitemap Provider
 *
 * Generates sitemaps for author archives (Pro+ tier)
 *
 * @package ProRank\SEO\Modules\Indexing\Sitemaps
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Indexing\Sitemaps;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\SettingsManager;
use ProRank\SEO\Core\LicenseManager;

/**
 * Author sitemap provider class
 */
class AuthorSitemapProvider extends BaseSitemapProvider {
    
    /**
     * Provider slug
     *
     * @var string
     */
    protected string $slug = 'author';
    
    /**
     * Provider name
     *
     * @var string
     */
    protected string $name = 'Authors';
    
    /**
     * Settings manager instance
     *
     * @var SettingsManager|null
     */
    private ?SettingsManager $settings_manager = null;
    
    /**
     * License manager instance
     *
     * @var LicenseManager|null
     */
    private ?LicenseManager $license_manager = null;
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        // Settings manager and license manager will be injected if needed
    }
    
    /**
     * Check if provider is enabled
     *
     * @return bool
     */
    public function is_enabled(): bool {
        // Check Pro+ license
        if (!$this->license_manager || !$this->license_manager->is_tier_active('pro')) {
            return false;
        }
        
        if (!$this->settings_manager) {
            return false; // Default to disabled for author sitemaps
        }
        
        $settings = $this->settings_manager->get_settings('sitemaps');
        return $settings['include_author_archives'] ?? false;
    }
    
    /**
     * Get total number of pages
     *
     * @return int
     */
    public function get_total_pages(): int {
        $total_authors = $this->get_total_authors();
        return (int) ceil($total_authors / $this->max_entries);
    }
    
    /**
     * Get total number of authors
     *
     * @return int
     */
    private function get_total_authors(): int {
        $args = [
            'role__in' => $this->get_included_roles(),
            'has_published_posts' => true,
            'fields' => 'ID',
            'count_total' => true,
        ];
        
        $user_query = new \WP_User_Query($args);
        return $user_query->get_total();
    }
    
    /**
     * Get included user roles
     *
     * @return array
     */
    private function get_included_roles(): array {
        if (!$this->settings_manager) {
            return ['author', 'editor', 'administrator'];
        }
        
        $settings = $this->settings_manager->get_settings('sitemaps');
        return $settings['author_roles'] ?? ['author', 'editor', 'administrator'];
    }
    
    /**
     * Generate sitemap XML content
     *
     * @param int $page Page number
     * @return string XML content
     */
    public function generate_xml(int $page = 1): string {
        $urls = $this->get_author_urls($page);
        return $this->generate_urlset_xml($urls);
    }
    
    /**
     * Get author URLs for sitemap
     *
     * @param int $page Page number
     * @return array
     */
    private function get_author_urls(int $page): array {
        $offset = ($page - 1) * $this->max_entries;
        
        $args = [
            'role__in' => $this->get_included_roles(),
            'has_published_posts' => true,
            'orderby' => 'post_count',
            'order' => 'DESC',
            'number' => $this->max_entries,
            'offset' => $offset,
            'meta_query' => [
                'relation' => 'OR',
                [
                    'key' => '_prorank_seo_robots_noindex',
                    'value' => '1',
                    'compare' => '!='
                ],
                [
                    'key' => '_prorank_seo_robots_noindex',
                    'compare' => 'NOT EXISTS'
                ]
            ]
        ];
        
        $user_query = new \WP_User_Query($args);
        $authors = $user_query->get_results();
        $urls = [];
        
        foreach ($authors as $author) {
            $url_data = [
                'loc' => get_author_posts_url($author->ID),
                'lastmod' => $this->get_author_last_modified($author->ID),
                'changefreq' => 'weekly',
                'priority' => $this->get_author_priority($author->ID)
            ];
            
            $urls[] = $url_data;
        }
        
        return $urls;
    }
    
    /**
     * Get author priority based on post count
     *
     * @param int $author_id Author ID
     * @return float
     */
    private function get_author_priority(int $author_id): float {
        $post_count = count_user_posts($author_id, 'post', true);
        
        if ($post_count > 100) {
            return 0.7;
        } elseif ($post_count > 50) {
            return 0.6;
        } elseif ($post_count > 10) {
            return 0.5;
        } elseif ($post_count > 0) {
            return 0.4;
        }
        
        return 0.3;
    }
    
    /**
     * Get last modified date for an author
     *
     * @param int $author_id Author ID
     * @return string ISO 8601 date
     */
    private function get_author_last_modified(int $author_id): string {
        global $wpdb;
        
        // Get the most recent post by this author
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $last_post_date = $wpdb->get_var(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT MAX(post_modified_gmt) 
                FROM {$wpdb->posts} 
                WHERE post_author = %d
                AND post_type = 'post'
                AND post_status = 'publish'
                AND post_password = ''",
                $author_id
            )
        );
        
        return $last_post_date ? $this->format_date($last_post_date) : $this->format_date('now');
    }
    
    /**
     * Get last modified date for sitemap
     *
     * @param int $page Page number
     * @return string ISO 8601 date
     */
    protected function get_last_modified(int $page = 1): string {
        global $wpdb;
        
        // Get the most recent post modification across all included authors
        $roles = $this->get_included_roles();
        if (empty($roles)) {
            return $this->format_date('now');
        }
        
        // Build role placeholders
        $placeholders = array_fill(0, count($roles), '%s');
        $role_sql = implode(',', $placeholders);
        
        // Prepare query
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $query = $wpdb->prepare(
            "SELECT MAX(p.post_modified_gmt) 
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->users} u ON p.post_author = u.ID
            INNER JOIN {$wpdb->usermeta} um ON u.ID = um.user_id
            WHERE um.meta_key = '{$wpdb->prefix}capabilities'
            AND p.post_type = 'post'
            AND p.post_status = 'publish'
            AND p.post_password = ''",
            ...$roles
        );
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        $last_modified = $wpdb->get_var($query);
        
        return $last_modified ? $this->format_date($last_modified) : $this->format_date('now');
    }
}
