<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Category Sitemap Provider
 *
 * Generates sitemaps for categories
 *
 * @package ProRank\SEO\Modules\Indexing\Sitemaps
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Indexing\Sitemaps;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\SettingsManager;

/**
 * Category sitemap provider class
 */
class CategorySitemapProvider extends BaseSitemapProvider {
    
    /**
     * Settings manager instance
     *
     * @var SettingsManager|null
     */
    protected ?SettingsManager $settings_manager = null;
    
    /**
     * Provider slug
     *
     * @var string
     */
    protected string $slug = 'category';
    
    /**
     * Provider name
     *
     * @var string
     */
    protected string $name = 'Categories';
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        // Settings manager will be injected if needed
    }
    
    /**
     * Check if provider is enabled
     *
     * @return bool
     */
    public function is_enabled(): bool {
        if (!$this->settings_manager) {
            return true; // Default to enabled
        }
        
        $settings = $this->settings_manager->get_settings('sitemaps');
        $enabled_taxonomies = $settings['include_taxonomies'] ?? ['category', 'post_tag'];
        
        return in_array('category', $enabled_taxonomies, true);
    }
    
    /**
     * Get total number of pages
     *
     * @return int
     */
    public function get_total_pages(): int {
        $total_terms = $this->get_total_terms();
        return (int) ceil($total_terms / $this->max_entries);
    }
    
    /**
     * Get total number of terms
     *
     * @return int
     */
    private function get_total_terms(): int {
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $exclude_terms = $settings['exclude_terms'] ?? '';
        
        $args = [
            'taxonomy' => 'category',
            'hide_empty' => $this->should_exclude_empty_terms(),
            'fields' => 'count',
        ];
        
        // Add term exclusions
        if (!empty($exclude_terms)) {
            $excluded_ids = array_map('intval', array_filter(explode(',', $exclude_terms)));
            if (!empty($excluded_ids)) {
                $args['exclude'] = $excluded_ids;
            }
        }
        
        $count = wp_count_terms($args);
        
        return is_numeric($count) ? (int) $count : 0;
    }
    
    /**
     * Generate sitemap XML content
     *
     * @param int $page Page number
     * @return string XML content
     */
    public function generate_xml(int $page = 1): string {
        $urls = $this->get_term_urls($page);
        return $this->generate_urlset_xml($urls);
    }
    
    /**
     * Get term URLs for sitemap
     *
     * @param int $page Page number
     * @return array
     */
    private function get_term_urls(int $page): array {
        $offset = ($page - 1) * $this->max_entries;
        
        // Get settings
        $settings = $this->settings_manager ? $this->settings_manager->get_settings('sitemaps') : [];
        $exclude_terms = $settings['exclude_terms'] ?? '';
        
        $args = [
            'taxonomy' => 'category',
            'hide_empty' => $this->should_exclude_empty_terms(),
            'number' => $this->max_entries,
            'offset' => $offset,
            'orderby' => 'count',
            'order' => 'DESC',
            'meta_query' => [
                'relation' => 'OR',
                [
                    'key' => '_prorank_seo_robots_noindex',
                    'value' => '1',
                    'compare' => '!='
                ],
                [
                    'key' => '_prorank_seo_robots_noindex',
                    'compare' => 'NOT EXISTS'
                ]
            ]
        ];
        
        // Add term exclusions
        if (!empty($exclude_terms)) {
            $excluded_ids = array_map('intval', array_filter(explode(',', $exclude_terms)));
            if (!empty($excluded_ids)) {
                $args['exclude'] = $excluded_ids;
            }
        }
        
        $terms = get_terms($args);
        $urls = [];
        
        if (!is_wp_error($terms) && !empty($terms)) {
            foreach ($terms as $term) {
                $url_data = [
                    'loc' => get_term_link($term),
                    'lastmod' => $this->get_term_last_modified($term->term_id),
                    'changefreq' => 'weekly',
                    'priority' => $this->get_term_priority($term)
                ];
                
                // Allow other modules to modify URL data
                $url_data = apply_filters('prorank_seo_sitemap_url_data', $url_data, $term, 'term');
                
                $urls[] = $url_data;
            }
        }
        
        return $urls;
    }
    
    /**
     * Check if empty terms should be excluded
     *
     * @return bool
     */
    protected function should_exclude_empty_terms(): bool {
        if (!$this->settings_manager) {
            return true; // Default to exclude empty
        }
        
        $settings = $this->settings_manager->get_settings('sitemaps');
        return $settings['exclude_empty_terms'] ?? true;
    }
    
    /**
     * Get term priority based on post count
     *
     * @param \WP_Term $term Term object
     * @return float
     */
    private function get_term_priority(\WP_Term $term): float {
        // Categories with more posts are considered more important
        if ($term->count > 100) {
            return 0.8;
        } elseif ($term->count > 50) {
            return 0.7;
        } elseif ($term->count > 10) {
            return 0.6;
        } elseif ($term->count > 0) {
            return 0.5;
        }
        
        return 0.3;
    }
    
    /**
     * Get last modified date for a term
     *
     * @param int $term_id Term ID
     * @return string ISO 8601 date
     */
    protected function get_term_last_modified(int $term_id): string {
        global $wpdb;
        
        // Get the most recent post in this category
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $last_post_date = $wpdb->get_var(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT MAX(p.post_modified_gmt) 
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->term_relationships} tr ON p.ID = tr.object_id
                WHERE tr.term_taxonomy_id = %d
                AND p.post_status = 'publish'
                AND p.post_password = ''",
                $term_id
            )
        );
        
        return $last_post_date ? $this->format_date($last_post_date) : $this->format_date('now');
    }
    
    /**
     * Get last modified date for sitemap
     *
     * @param int $page Page number
     * @return string ISO 8601 date
     */
    protected function get_last_modified(int $page = 1): string {
        global $wpdb;
        
        // Get the most recent post modification across all categories
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $last_modified = $wpdb->get_var(
            "SELECT MAX(post_modified_gmt) 
            FROM {$wpdb->posts} 
            WHERE post_type = 'post'
            AND post_status = 'publish'
            AND post_password = ''"
        );
        
        return $last_modified ? $this->format_date($last_modified) : $this->format_date('now');
    }
}
