<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Video Sitemap Provider
 *
 * @package ProRank\SEO\Modules\Indexing\Sitemaps
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Indexing\Sitemaps;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Core\SettingsManager;

/**
 * Class VideoSitemapProvider
 * 
 * Generates XML sitemap for video content
 */
class VideoSitemapProvider extends BaseSitemapProvider {
    
    /**
     * Settings manager instance
     *
     * @var SettingsManager|null
     */
    private ?SettingsManager $settings_manager = null;
    /**
     * Get sitemap slug
     *
     * @return string
     */
    public function get_slug(): string {
        return 'video';
    }

    /**
     * Get sitemap name  
     *
     * @return string
     */
    public function get_name(): string {
        return __('Video Sitemap', 'prorank-seo');
    }

    /**
     * Build sitemap entries
     *
     * @return void
     */
    protected function build_entries(): void {
        // Get video sitemap settings
        $settings = $this->get_video_settings();
        
        // Get post types from settings
        $post_types = $settings['post_types'] ?? ['post', 'page'];

        $args = [
            'post_type'      => $post_types,
            'post_status'    => 'publish',
            'posts_per_page' => $this->get_posts_per_page(),
            'paged'          => $this->page,
        ];
        
        // Don't require video meta if we're auto-detecting or using custom fields
        if (!$settings['auto_detect'] && empty($settings['custom_fields'])) {
            $args['meta_query'] = [
                [
                    'key'     => '_prorank_seo_video_url',
                    'compare' => 'EXISTS',
                ],
            ];
        }

        // Exclude noindex posts
        if (true) { // Default to excluding noindex
            $args['meta_query'][] = [
                'relation' => 'OR',
                [
                    'key'     => '_prorank_seo_noindex',
                    'compare' => 'NOT EXISTS',
                ],
                [
                    'key'     => '_prorank_seo_noindex',
                    'value'   => 'on',
                    'compare' => '!=',
                ],
            ];
        }

        $query = new \WP_Query($args);

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $post_id = get_the_ID();

                // Get video data
                $video_data = $this->get_video_data($post_id);
                
                if (empty($video_data)) {
                    continue;
                }

                $entry = [
                    'loc'    => get_permalink($post_id),
                    'videos' => $video_data,
                ];

                $this->add_entry($entry);
            }
            wp_reset_postdata();
        }

        $this->total_pages = $query->max_num_pages;
    }

    /**
     * Get video sitemap settings
     *
     * @return array
     */
    private function get_video_settings(): array {
        $defaults = [
            'enabled' => false,
            'hide_sitemap' => false,
            'post_types' => ['post', 'page'],
            'youtube_api_key' => '',
            'custom_fields' => '',
            'auto_detect' => true,
            'supportedFormats' => ['youtube', 'vimeo', 'mp4'],
        ];
        
        $settings = get_option('prorank_seo_video_sitemap_settings', $defaults);
        return wp_parse_args($settings, $defaults);
    }
    
    /**
     * Get video data for a post
     *
     * @param int $post_id Post ID
     * @return array
     */
    private function get_video_data(int $post_id): array {
        $videos = [];
        $settings = $this->get_video_settings();
        
        // Get videos from meta
        $video_url = get_post_meta($post_id, '_prorank_seo_video_url', true);
        $video_title = get_post_meta($post_id, '_prorank_seo_video_title', true);
        $video_description = get_post_meta($post_id, '_prorank_seo_video_description', true);
        $video_thumbnail = get_post_meta($post_id, '_prorank_seo_video_thumbnail', true);
        $video_duration = get_post_meta($post_id, '_prorank_seo_video_duration', true);
        $video_upload_date = get_post_meta($post_id, '_prorank_seo_video_upload_date', true);
        
        // New 2025 fields
        $is_youtube_short = get_post_meta($post_id, '_prorank_seo_video_is_short', true);
        $contains_synthetic = get_post_meta($post_id, '_prorank_seo_video_synthetic_content', true);
        $view_count = get_post_meta($post_id, '_prorank_seo_video_view_count', true);

        if (!empty($video_url)) {
            $video = [
                'content_loc' => esc_url($video_url),
                'title'       => !empty($video_title) ? $video_title : get_the_title($post_id),
                'description' => !empty($video_description) ? $video_description : wp_trim_words(get_the_excerpt($post_id), 50),
            ];

            // Optional fields
            if (!empty($video_thumbnail)) {
                $video['thumbnail_loc'] = esc_url($video_thumbnail);
            } else {
                // Try to get featured image as thumbnail
                $thumbnail_id = get_post_thumbnail_id($post_id);
                if ($thumbnail_id) {
                    $thumbnail_url = wp_get_attachment_url($thumbnail_id);
                    if ($thumbnail_url) {
                        $video['thumbnail_loc'] = esc_url($thumbnail_url);
                    }
                }
            }

            if (!empty($video_duration) && is_numeric($video_duration)) {
                $video['duration'] = intval($video_duration);
            }

            if (!empty($video_upload_date)) {
                $video['publication_date'] = mysql2date('c', $video_upload_date, false);
            } else {
                $video['publication_date'] = get_post_time('c', true, $post_id);
            }

            // Additional metadata
            $video['family_friendly'] = get_post_meta($post_id, '_prorank_seo_video_family_friendly', true) !== 'no' ? 'yes' : 'no';
            
            $video_tags = get_post_meta($post_id, '_prorank_seo_video_tags', true);
            if (!empty($video_tags)) {
                $video['tag'] = array_map('trim', explode(',', $video_tags));
            }

            $video_category = get_post_meta($post_id, '_prorank_seo_video_category', true);
            if (!empty($video_category)) {
                $video['category'] = $video_category;
            }
            
            // Add 2025 YouTube Shorts support
            if ($is_youtube_short === 'yes' || $this->is_youtube_short($video_url)) {
                $video['is_short'] = 'yes';
                // Shorts don't require minimum watch time as of March 31, 2025
                $video['view_count_type'] = 'starts_and_replays';
            }
            
            // Add synthetic/AI content labeling (2025 requirement)
            if ($contains_synthetic === 'yes') {
                $video['contains_synthetic_media'] = 'yes';
            }
            
            // Add view count if available
            if (!empty($view_count) && is_numeric($view_count)) {
                $video['view_count'] = intval($view_count);
            }

            $videos[] = $video;
        }
        
        // Scan custom fields if configured
        if (!empty($settings['custom_fields'])) {
            $custom_field_videos = $this->scan_custom_fields($post_id, $settings['custom_fields']);
            if (!empty($custom_field_videos)) {
                $videos = array_merge($videos, $custom_field_videos);
            }
        }

        // Also parse content for embedded videos if auto-detect is enabled
        if ($settings['auto_detect']) {
            $content_videos = $this->parse_content_videos(get_post_field('post_content', $post_id));
            if (!empty($content_videos)) {
                $videos = array_merge($videos, $content_videos);
            }
        }

        return $videos;
    }

    /**
     * Scan custom fields for video URLs
     *
     * @param int $post_id Post ID
     * @param string $custom_fields Custom fields to scan (one per line)
     * @return array
     */
    private function scan_custom_fields(int $post_id, string $custom_fields): array {
        $videos = [];
        $fields = array_filter(array_map('trim', explode("\n", $custom_fields)));
        
        foreach ($fields as $field) {
            $value = get_post_meta($post_id, $field, true);
            if (!empty($value)) {
                // Check if it's a video URL or embed code
                $field_videos = $this->parse_content_videos($value);
                if (!empty($field_videos)) {
                    $videos = array_merge($videos, $field_videos);
                } elseif (filter_var($value, FILTER_VALIDATE_URL)) {
                    // It's a direct URL
                    $videos[] = [
                        'content_loc' => esc_url($value),
                        'title' => get_the_title($post_id),
                        'description' => wp_trim_words(get_the_excerpt($post_id), 50),
                    ];
                }
            }
        }
        
        return $videos;
    }
    
    /**
     * Check if a YouTube video is a Short
     *
     * @param string $video_url Video URL
     * @return bool
     */
    private function is_youtube_short(string $video_url): bool {
        // YouTube Shorts URLs contain /shorts/ in the path
        if (strpos($video_url, '/shorts/') !== false) {
            return true;
        }
        
        // Check if it's a regular YouTube video that might be a Short
        // Shorts are typically vertical videos under 60 seconds
        $video_id = $this->extract_youtube_id($video_url);
        if ($video_id) {
            $video_data = $this->fetch_youtube_video_data($video_id);
            if ($video_data && isset($video_data['duration'])) {
                // Shorts are 60 seconds or less
                return $video_data['duration'] <= 60;
            }
        }
        
        return false;
    }
    
    /**
     * Extract YouTube video ID from URL
     *
     * @param string $url YouTube URL
     * @return string|null
     */
    private function extract_youtube_id(string $url): ?string {
        preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/|youtube\.com\/shorts\/)([^"&?\/\s]{11})/i', $url, $matches);
        return isset($matches[1]) ? $matches[1] : null;
    }
    
    /**
     * Fetch YouTube video data with caching (2025 API quota optimization)
     *
     * @param string $video_id YouTube video ID
     * @return array|null
     */
    private function fetch_youtube_video_data(string $video_id): ?array {
        $settings = $this->get_video_settings();
        $api_key = $settings['youtube_api_key'] ?? '';
        
        if (empty($api_key)) {
            return null;
        }
        
        // Cache key for API response (15 minute cache to optimize quota)
        $cache_key = 'prorank_youtube_data_' . $video_id;
        $cached_data = get_transient($cache_key);
        
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        // Fetch from YouTube API
        $api_url = sprintf(
            'https://www.googleapis.com/youtube/v3/videos?id=%s&key=%s&part=snippet,contentDetails,statistics',
            $video_id,
            $api_key
        );
        
        $response = wp_remote_get($api_url);
        
        if (is_wp_error($response)) {
            return null;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!isset($data['items'][0])) {
            return null;
        }
        
        $item = $data['items'][0];
        $video_data = [
            'title' => $item['snippet']['title'] ?? '',
            'description' => $item['snippet']['description'] ?? '',
            'thumbnail' => $item['snippet']['thumbnails']['high']['url'] ?? '',
            'duration' => $this->parse_iso8601_duration($item['contentDetails']['duration'] ?? ''),
            'view_count' => intval($item['statistics']['viewCount'] ?? 0),
            'published_at' => $item['snippet']['publishedAt'] ?? '',
            'is_short' => $this->check_if_short_from_api($item),
        ];
        
        // Cache for 15 minutes to optimize API quota usage
        set_transient($cache_key, $video_data, 15 * MINUTE_IN_SECONDS);
        
        return $video_data;
    }
    
    /**
     * Check if video is a Short from API data
     *
     * @param array $item YouTube API item data
     * @return bool
     */
    private function check_if_short_from_api(array $item): bool {
        // Check duration (60 seconds or less)
        $duration = $this->parse_iso8601_duration($item['contentDetails']['duration'] ?? '');
        if ($duration <= 60) {
            // Check aspect ratio if available (Shorts are vertical)
            // This would require additional API calls, so we rely on duration for now
            return true;
        }
        return false;
    }
    
    /**
     * Parse ISO 8601 duration to seconds
     *
     * @param string $duration ISO 8601 duration
     * @return int Duration in seconds
     */
    private function parse_iso8601_duration(string $duration): int {
        if (empty($duration)) {
            return 0;
        }
        
        $interval = new \DateInterval($duration);
        return ($interval->h * 3600) + ($interval->i * 60) + $interval->s;
    }
    
    /**
     * Parse post content for embedded videos
     *
     * @param string $content Post content
     * @return array
     */
    private function parse_content_videos(string $content): array {
        $videos = [];
        $settings = $this->get_video_settings();
        
        // YouTube embeds (including Shorts)
        preg_match_all('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/|youtube\.com\/shorts\/)([^"&?\/\s]{11})/i', $content, $youtube_matches);
        
        foreach ($youtube_matches[1] as $youtube_id) {
            $video_url = 'https://www.youtube.com/watch?v=' . $youtube_id;
            $video_entry = [
                'content_loc' => $video_url,
                'player_loc'  => 'https://www.youtube.com/embed/' . $youtube_id,
                'allow_embed' => 'yes',
            ];
            
            // Check if it's a YouTube Short and add metadata
            if ($this->is_youtube_short($video_url)) {
                $video_entry['is_short'] = 'yes';
                $video_entry['view_count_type'] = 'starts_and_replays';
            }
            
            // Try to fetch additional data from YouTube API if available
            if (!empty($settings['youtube_api_key'])) {
                $api_data = $this->fetch_youtube_video_data($youtube_id);
                if ($api_data) {
                    $video_entry['title'] = $api_data['title'];
                    $video_entry['description'] = $api_data['description'];
                    $video_entry['thumbnail_loc'] = $api_data['thumbnail'];
                    $video_entry['duration'] = $api_data['duration'];
                    $video_entry['view_count'] = $api_data['view_count'];
                }
            }
            
            $videos[] = $video_entry;
        }

        // Vimeo embeds
        preg_match_all('/vimeo\.com\/(?:video\/)?(\d+)/i', $content, $vimeo_matches);
        
        foreach ($vimeo_matches[1] as $vimeo_id) {
            $videos[] = [
                'content_loc' => 'https://vimeo.com/' . $vimeo_id,
                'player_loc'  => 'https://player.vimeo.com/video/' . $vimeo_id,
                'allow_embed' => 'yes',
            ];
        }

        return $videos;
    }

    /**
     * Render sitemap
     *
     * @return void
     */
    public function render(): void {
        header('Content-Type: application/xml; charset=UTF-8');
        
        echo '<?xml version="1.0" encoding="UTF-8"?>';
        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" ';
        echo 'xmlns:video="http://www.google.com/schemas/sitemap-video/1.1">';
        echo "\n";

        foreach ($this->entries as $entry) {
            echo "\t<url>\n";
            echo "\t\t<loc>" . esc_url($entry['loc']) . "</loc>\n";
            
            foreach ($entry['videos'] as $video) {
                echo "\t\t<video:video>\n";
                
                if (isset($video['content_loc'])) {
                    echo "\t\t\t<video:content_loc>" . esc_url($video['content_loc']) . "</video:content_loc>\n";
                }
                
                if (isset($video['player_loc'])) {
                    echo "\t\t\t<video:player_loc>" . esc_url($video['player_loc']) . "</video:player_loc>\n";
                }
                
                echo "\t\t\t<video:title>" . esc_xml($video['title']) . "</video:title>\n";
                echo "\t\t\t<video:description>" . esc_xml($video['description']) . "</video:description>\n";
                
                if (isset($video['thumbnail_loc'])) {
                    echo "\t\t\t<video:thumbnail_loc>" . esc_url($video['thumbnail_loc']) . "</video:thumbnail_loc>\n";
                }
                
                if (isset($video['duration'])) {
                    echo "\t\t\t<video:duration>" . intval($video['duration']) . "</video:duration>\n";
                }
                
                if (isset($video['publication_date'])) {
                    echo "\t\t\t<video:publication_date>" . esc_xml($video['publication_date']) . "</video:publication_date>\n";
                }
                
                if (isset($video['family_friendly'])) {
                    echo "\t\t\t<video:family_friendly>" . esc_xml($video['family_friendly']) . "</video:family_friendly>\n";
                }
                
                if (isset($video['tag']) && is_array($video['tag'])) {
                    foreach ($video['tag'] as $tag) {
                        echo "\t\t\t<video:tag>" . esc_xml($tag) . "</video:tag>\n";
                    }
                }
                
                if (isset($video['category'])) {
                    echo "\t\t\t<video:category>" . esc_xml($video['category']) . "</video:category>\n";
                }
                
                if (isset($video['allow_embed'])) {
                    echo "\t\t\t<video:requires_subscription>no</video:requires_subscription>\n";
                }
                
                // 2025 additions for YouTube Shorts
                if (isset($video['is_short']) && $video['is_short'] === 'yes') {
                    echo "\t\t\t<!-- YouTube Short detected -->\n";
                    echo "\t\t\t<video:platform>YouTube Shorts</video:platform>\n";
                    if (isset($video['view_count_type'])) {
                        echo "\t\t\t<video:view_count_type>" . esc_xml($video['view_count_type']) . "</video:view_count_type>\n";
                    }
                }
                
                // 2025 synthetic/AI content labeling
                if (isset($video['contains_synthetic_media']) && $video['contains_synthetic_media'] === 'yes') {
                    echo "\t\t\t<video:contains_synthetic_media>yes</video:contains_synthetic_media>\n";
                }
                
                // View count if available
                if (isset($video['view_count'])) {
                    echo "\t\t\t<video:view_count>" . intval($video['view_count']) . "</video:view_count>\n";
                }
                
                echo "\t\t</video:video>\n";
            }
            
            echo "\t</url>\n";
        }

        echo '</urlset>';
    }

    /**
     * Get cache key
     *
     * @param int $page Page number
     * @return string
     */
    protected function get_cache_key(int $page = 1): string {
        return 'prorank_video_sitemap_' . $page;
    }

    /**
     * Check if provider should be registered
     *
     * @return bool
     */
    public function should_register(): bool {
        // Always register for now, let is_enabled() control visibility
        return true;
    }
    
    /**
     * Check if provider is enabled
     *
     * @return bool
     */
    public function is_enabled(): bool {
        $settings = $this->get_video_settings();
        return $settings['enabled'] ?? false;
    }
    
    /**
     * Get total number of pages
     *
     * @return int
     */
    public function get_total_pages(): int {
        if (isset($this->total_pages)) {
            return $this->total_pages;
        }
        
        // Run a count query
        $args = [
            'post_type'      => ['post', 'page'], // Default post types
            'post_status'    => 'publish',
            'posts_per_page' => 1,
            'meta_query'     => [
                [
                    'key'     => '_prorank_seo_video_url',
                    'compare' => 'EXISTS',
                ],
            ],
        ];
        
        $query = new \WP_Query($args);
        $total_posts = $query->found_posts;
        $posts_per_page = $this->get_posts_per_page();
        
        $this->total_pages = (int) ceil($total_posts / $posts_per_page);
        
        return $this->total_pages;
    }
    
    /**
     * Generate sitemap XML content
     *
     * @param int $page Page number
     * @return string XML content
     */
    public function generate_xml(int $page = 1): string {
        $this->page = $page;
        $this->entries = [];
        
        // Build entries
        $this->build_entries();
        
        // Generate XML
        ob_start();
        $this->render();
        return ob_get_clean();
    }
    
    /**
     * Get last modified date for sitemap
     *
     * @param int $page Page number
     * @return string ISO 8601 date
     */
    protected function get_last_modified(int $page = 1): string {
        $args = [
            'post_type'      => ['post', 'page'], // Default post types
            'post_status'    => 'publish',
            'posts_per_page' => 1,
            'paged'          => $page,
            'orderby'        => 'modified',
            'order'          => 'DESC',
            'meta_query'     => [
                [
                    'key'     => '_prorank_seo_video_url',
                    'compare' => 'EXISTS',
                ],
            ],
        ];
        
        $query = new \WP_Query($args);
        
        if ($query->have_posts()) {
            $query->the_post();
            $date = get_post_modified_time('c', true);
            wp_reset_postdata();
            return $date;
        }
        
        return gmdate('c');
    }
    
    /**
     * Get posts per page
     *
     * @return int
     */
    private function get_posts_per_page(): int {
        return min($this->max_entries, 1000);
    }
    
    /**
     * Add necessary properties
     */
    private $page = 1;
    private $entries = [];
    private $total_pages = 0;
    
    /**
     * Add entry to sitemap
     *
     * @param array $entry
     */
    private function add_entry(array $entry): void {
        $this->entries[] = $entry;
    }
    
    /**
     * Helper function to escape XML
     *
     * @param string $string
     * @return string
     */
    private function esc_xml(string $string): string {
        return htmlspecialchars($string, ENT_XML1 | ENT_QUOTES, 'UTF-8');
    }
}