<?php
/**
 * Lazy Loading Module
 *
 * Implements lazy loading for images, iframes, and videos
 *
 * @package ProRank\SEO\Modules\Performance
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\Performance;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;
use ProRank\SEO\Plugin;
use DOMDocument;
use DOMXPath;

/**
 * LazyLoadModule class
 */
class LazyLoadModule extends BaseModule {

    /**
     * Required tier for this module
     */
    protected string $feature_tier = 'free';

    /**
     * Excluded classes/patterns cache
     *
     * @var array
     */
    private array $exclusions = [];

    /**
     * JavaScript added flag
     *
     * @var bool
     */
    private bool $js_added = false;

    /**
     * Global image counter for tracking above-the-fold images
     *
     * @var int
     */
    private static int $image_count = 0;

    /**
     * Number of images to skip lazy loading (above the fold)
     * WP Rocket and Perfmatters typically skip 3-5 images
     *
     * @var int
     */
    private const SKIP_FIRST_IMAGES = 3;

    /**
     * Whether we've already added fetchpriority to an image
     *
     * @var bool
     */
    private static bool $fetchpriority_added = false;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->slug = 'lazyload';
        $this->name = 'Lazy Loading';
        $this->description = 'Lazy load images, iframes, and videos for better performance';
$this->parent_slug = 'performance';
    }

    /**
     * Get setting from unified image optimization settings
     * Overrides BaseModule to read from prorank_image_optimization_settings
     *
     * @param string $key Setting key
     * @param mixed $default Default value
     * @return mixed
     */
    protected function get_setting(string $key, $default = null) {
        // Try to get from central settings first (AI Doctor updates these)
        try {
            $plugin = Plugin::get_instance();
            if ($plugin->is_initialized()) {
                $settings_manager = $plugin->settings();
                
                // Map short keys to performance group keys
                $perf_key_map = [
                    'images' => 'lazy_load_images',
                    'iframes' => 'lazy_load_iframes',
                    'videos' => 'lazy_load_videos',
                ];
                
                if (isset($perf_key_map[$key])) {
                    $val = $settings_manager->get('performance', $perf_key_map[$key]);
                    if ($val !== null) {
                        return $val;
                    }
                }
            }
        } catch (\Exception $e) {
            // Fallback to legacy
        }

        static $settings = null;
        if ($settings === null) {
            $settings = get_option('prorank_image_optimization_settings', []);
        }
        // Map short keys to full keys
        $key_map = [
            'images' => 'lazyload_images',
            'iframes' => 'lazyload_iframes',
            'videos' => 'lazyload_videos',
            'threshold' => 'lazyload_threshold',
        ];
        $mapped_key = $key_map[$key] ?? $key;
        return $settings[$mapped_key] ?? $default;
    }
    
    /**
     * Initialize module hooks
     *
     * @return void
     */
    public function init_hooks(): void {
        // Check if module should run
        if (!$this->should_run()) {
            return;
        }
        
        // Hook into content filtering
        add_filter('the_content', [$this, 'process_content'], 999);
        add_filter('widget_text', [$this, 'process_content'], 999);
        add_filter('get_avatar', [$this, 'process_avatar'], 999);
        add_filter('post_thumbnail_html', [$this, 'process_content'], 999);
        
        // Hook into image attributes
        add_filter('wp_get_attachment_image_attributes', [$this, 'add_lazy_load_attributes'], 10, 3);
        
        // Add frontend JavaScript
        if ($this->get_setting('background_images', false)) {
            add_action('wp_footer', [$this, 'add_lazy_load_script'], 20);
        }
    }
    
    /**
     * Check if module should run
     *
     * @return bool
     */
    private function should_run(): bool {
        // Module is available for free tier
        if (!$this->is_enabled()) {
            return false;
        }
        
        // Don't run in admin, feeds, or for AMP
        if (is_admin() || is_feed() || $this->is_amp()) {
            return false;
        }
        
        // Check if any lazy loading is enabled
        return (bool) $this->get_setting('images', true) ||
               $this->get_setting('iframes', true) ||
               $this->get_setting('videos', false) ||
               $this->get_setting('background_images', false);
    }
    
    /**
     * Process content to add lazy loading
     *
     * @param string $content Content to process
     * @return string Modified content
     */
    public function process_content(string $content): string {
        if (empty($content)) {
            return $content;
        }
        
        // Use DOMDocument to parse HTML
        $dom = $this->get_dom_document($content);
        if (!$dom) {
            return $content;
        }
        
        $xpath = new DOMXPath($dom);
        $modified = false;
        
        // Process images
        if ($this->get_setting('images', true)) {
            $modified = $this->process_images($dom, $xpath) || $modified;
        }
        
        // Process iframes
        if ($this->get_setting('iframes', true)) {
            $modified = $this->process_iframes($dom, $xpath) || $modified;
        }
        
        // Process videos
        if ($this->get_setting('videos', false)) {
            $modified = $this->process_videos($dom, $xpath) || $modified;
        }
        
        // Process background images
        if ($this->get_setting('background_images', false)) {
            $modified = $this->process_background_images($dom, $xpath) || $modified;
        }
        
        // Return original content if nothing was modified
        if (!$modified) {
            return $content;
        }
        
        // Get the processed HTML
        $body = $dom->getElementsByTagName('body')->item(0);
        if (!$body) {
            return $content;
        }
        
        $processed = '';
        foreach ($body->childNodes as $child) {
            $processed .= $dom->saveHTML($child);
        }
        
        return $processed ?: $content;
    }
    
    /**
     * Process avatar images
     *
     * @param string $avatar Avatar HTML
     * @return string Modified avatar HTML
     */
    public function process_avatar(string $avatar): string {
        if (!$this->get_setting('images', true)) {
            return $avatar;
        }
        
        return $this->process_content($avatar);
    }
    
    /**
     * Add lazy load attributes to image
     *
     * @param array $attributes Image attributes
     * @param object $attachment Attachment object
     * @param string|array $size Image size
     * @return array Modified attributes
     */
    public function add_lazy_load_attributes(array $attributes, $attachment, $size): array {
        if (!$this->get_setting('images', true)) {
            return $attributes;
        }

        // Increment global image counter
        self::$image_count++;

        // Skip if excluded
        $class = $attributes['class'] ?? '';
        if ($this->is_excluded($class) || $this->is_above_fold_class($class)) {
            return $attributes;
        }

        // First image gets fetchpriority="high" and NO lazy loading (likely LCP)
        if (self::$image_count === 1 && !self::$fetchpriority_added) {
            $attributes['fetchpriority'] = 'high';
            // Remove lazy loading from LCP candidate
            unset($attributes['loading']);
            self::$fetchpriority_added = true;
            return $attributes;
        }

        // Skip lazy loading for first N images (above the fold)
        if (self::$image_count <= self::SKIP_FIRST_IMAGES) {
            // Don't add lazy loading to above-the-fold images
            unset($attributes['loading']);
            return $attributes;
        }

        // Skip if already has loading attribute
        if (isset($attributes['loading'])) {
            return $attributes;
        }

        // Add lazy loading for images below the fold
        $attributes['loading'] = 'lazy';

        return $attributes;
    }

    /**
     * Check if class indicates an above-the-fold element
     *
     * @param string $class Class string
     * @return bool
     */
    private function is_above_fold_class(string $class): bool {
        $above_fold_classes = [
            'custom-logo',
            'site-logo',
            'header-logo',
            'logo',
            'hero-image',
            'banner-image',
            'featured-image',
            'wp-post-image', // Featured images are often above fold
        ];

        foreach ($above_fold_classes as $atf_class) {
            if (strpos($class, $atf_class) !== false) {
                return true;
            }
        }

        return false;
    }
    
    /**
     * Process images in DOM
     *
     * @param DOMDocument $dom DOM document
     * @param DOMXPath $xpath XPath object
     * @return bool Whether any images were modified
     */
    private function process_images(DOMDocument $dom, DOMXPath $xpath): bool {
        $images = $xpath->query('//img');
        $modified = false;
        $local_count = 0;

        foreach ($images as $img) {
            $local_count++;

            // Check exclusions
            $class = $img->getAttribute('class');
            if ($this->is_excluded($class) || $this->is_above_fold_class($class)) {
                // Still add dimensions even for excluded images (CLS fix)
                if ($this->add_missing_dimensions($img)) {
                    $modified = true;
                }
                continue;
            }

            // Skip images without src
            if (!$img->hasAttribute('src') || empty($img->getAttribute('src'))) {
                continue;
            }

            // Add missing width/height attributes (CLS fix)
            if ($this->add_missing_dimensions($img)) {
                $modified = true;
            }

            // First image gets fetchpriority="high" and NO lazy loading
            if ($local_count === 1 && !$img->hasAttribute('fetchpriority')) {
                $img->setAttribute('fetchpriority', 'high');
                // Don't add lazy loading to LCP candidate
                $modified = true;
                continue;
            }

            // Skip lazy loading for first N images (above the fold)
            if ($local_count <= self::SKIP_FIRST_IMAGES) {
                continue;
            }

            // Add loading attribute for below-fold images (if not already set)
            if (!$img->hasAttribute('loading')) {
                $img->setAttribute('loading', 'lazy');
                $modified = true;
            }
        }

        return $modified;
    }

    /**
     * Add missing width/height attributes to image to prevent CLS
     *
     * @param \DOMElement $img Image element
     * @return bool Whether dimensions were added
     */
    private function add_missing_dimensions(\DOMElement $img): bool {
        // Skip if both dimensions already exist
        if ($img->hasAttribute('width') && $img->hasAttribute('height')) {
            return false;
        }

        $src = $img->getAttribute('src');
        if (empty($src)) {
            return false;
        }

        // Get dimensions from various sources
        $dimensions = $this->get_image_dimensions($src);
        if (!$dimensions) {
            return false;
        }

        $modified = false;

        // Add width if missing
        if (!$img->hasAttribute('width') && $dimensions['width'] > 0) {
            $img->setAttribute('width', (string) $dimensions['width']);
            $modified = true;
        }

        // Add height if missing
        if (!$img->hasAttribute('height') && $dimensions['height'] > 0) {
            $img->setAttribute('height', (string) $dimensions['height']);
            $modified = true;
        }

        return $modified;
    }

    /**
     * Get image dimensions from various sources
     *
     * @param string $src Image source URL
     * @return array|null ['width' => int, 'height' => int] or null
     */
    private function get_image_dimensions(string $src): ?array {
        // Cache dimensions to avoid repeated lookups
        static $cache = [];

        if (isset($cache[$src])) {
            return $cache[$src];
        }

        $dimensions = null;

        // Try to get attachment ID from URL
        $attachment_id = attachment_url_to_postid($src);
        if ($attachment_id) {
            $metadata = wp_get_attachment_metadata($attachment_id);
            if (!empty($metadata['width']) && !empty($metadata['height'])) {
                $dimensions = [
                    'width' => (int) $metadata['width'],
                    'height' => (int) $metadata['height'],
                ];
            }
        }

        // If not found, try to get from local file
        if (!$dimensions) {
            $local_path = $this->url_to_local_path($src);
            if ($local_path && file_exists($local_path)) {
                $size = @getimagesize($local_path);
                if ($size && $size[0] > 0 && $size[1] > 0) {
                    $dimensions = [
                        'width' => (int) $size[0],
                        'height' => (int) $size[1],
                    ];
                }
            }
        }

        $cache[$src] = $dimensions;
        return $dimensions;
    }

    /**
     * Convert URL to local file path
     *
     * @param string $url Image URL
     * @return string|null Local file path or null
     */
    private function url_to_local_path(string $url): ?string {
        // Get upload directory info
        $upload_dir = wp_upload_dir();

        // Check if URL is from uploads directory
        if (strpos($url, $upload_dir['baseurl']) === 0) {
            return str_replace($upload_dir['baseurl'], $upload_dir['basedir'], $url);
        }

        // Check if URL is from the site
        $site_url = site_url();
        $abspath = ABSPATH;
        if (strpos($url, $site_url) === 0) {
            return str_replace($site_url, rtrim($abspath, '/'), $url);
        }

        // Handle relative URLs
        if (strpos($url, '/') === 0 && strpos($url, '//') !== 0) {
            return rtrim($abspath, '/') . $url;
        }

        return null;
    }
    
    /**
     * Process iframes in DOM
     *
     * @param DOMDocument $dom DOM document
     * @param DOMXPath $xpath XPath object
     * @return bool Whether any iframes were modified
     */
    private function process_iframes(DOMDocument $dom, DOMXPath $xpath): bool {
        $iframes = $xpath->query('//iframe[not(@loading)]');
        $modified = false;
        
        foreach ($iframes as $iframe) {
            // Check exclusions
            $class = $iframe->getAttribute('class');
            $src = $iframe->getAttribute('src');
            
            if ($this->is_excluded($class) || $this->is_excluded($src)) {
                continue;
            }
            
            // Skip iframes without src
            if (empty($src)) {
                continue;
            }
            
            // Add loading attribute
            $iframe->setAttribute('loading', 'lazy');
            $modified = true;
        }
        
        return $modified;
    }
    
    /**
     * Process videos in DOM
     *
     * @param DOMDocument $dom DOM document
     * @param DOMXPath $xpath XPath object
     * @return bool Whether any videos were modified
     */
    private function process_videos(DOMDocument $dom, DOMXPath $xpath): bool {
        // For video elements, we'll add preload="none" to prevent automatic loading
        $videos = $xpath->query('//video[not(@preload) or @preload="auto" or @preload="metadata"]');
        $modified = false;
        
        foreach ($videos as $video) {
            // Check exclusions
            $class = $video->getAttribute('class');
            if ($this->is_excluded($class)) {
                continue;
            }
            
            // Set preload to none
            $video->setAttribute('preload', 'none');
            
            // Add data attribute for JavaScript enhancement
            $video->setAttribute('data-prorank-lazy', 'true');
            
            $modified = true;
        }
        
        return $modified;
    }
    
    /**
     * Process background images in DOM
     *
     * @param DOMDocument $dom DOM document
     * @param DOMXPath $xpath XPath object
     * @return bool Whether any elements were modified
     */
    private function process_background_images(DOMDocument $dom, DOMXPath $xpath): bool {
        // Find elements with inline background-image styles
        $elements = $xpath->query('//*[contains(@style, "background-image")]');
        $modified = false;
        
        foreach ($elements as $element) {
            $style = $element->getAttribute('style');
            
            // Extract background-image URL
            if (preg_match('/background-image\s*:\s*url\s*\(\s*[\'"]?([^\'")]+)[\'"]?\s*\)/i', $style, $matches)) {
                $url = $matches[1];
                
                // Check exclusions
                $class = $element->getAttribute('class');
                if ($this->is_excluded($class) || $this->is_excluded($url)) {
                    continue;
                }
                
                // Move background-image to data attribute
                $element->setAttribute('data-prorank-lazy-bg', $url);
                
                // Remove background-image from style
                $new_style = preg_replace('/background-image\s*:\s*url\s*\([^)]+\)\s*;?/i', '', $style);
                $element->setAttribute('style', trim($new_style));
                
                // Add marker class
                $classes = $element->getAttribute('class');
                $element->setAttribute('class', trim($classes . ' prorank-lazy-bg'));
                
                $modified = true;
                $this->js_added = true;
            }
        }
        
        return $modified;
    }
    
    /**
     * Add lazy load JavaScript
     *
     * @return void
     */
    public function add_lazy_load_script(): void {
        if (!$this->js_added) {
            return;
        }
        
        ?>
        <script id="prorank-lazy-load">
        (function() {
            'use strict';
            
            // Check for IntersectionObserver support
            if (!('IntersectionObserver' in window)) {
                // Fallback: load all background images immediately
                var elements = document.querySelectorAll('.prorank-lazy-bg[data-prorank-lazy-bg]');
                elements.forEach(function(el) {
                    var bg = el.getAttribute('data-prorank-lazy-bg');
                    if (bg) {
                        el.style.backgroundImage = 'url(' + bg + ')';
                        el.removeAttribute('data-prorank-lazy-bg');
                    }
                });
                return;
            }
            
            // Create intersection observer
            var lazyBgObserver = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        var el = entry.target;
                        var bg = el.getAttribute('data-prorank-lazy-bg');
                        
                        if (bg) {
                            el.style.backgroundImage = 'url(' + bg + ')';
                            el.removeAttribute('data-prorank-lazy-bg');
                            el.classList.remove('prorank-lazy-bg');
                            lazyBgObserver.unobserve(el);
                        }
                    }
                });
            }, {
                rootMargin: '50px 0px',
                threshold: 0.01
            });
            
            // Observe all lazy background elements
            document.addEventListener('DOMContentLoaded', function() {
                var elements = document.querySelectorAll('.prorank-lazy-bg[data-prorank-lazy-bg]');
                elements.forEach(function(el) {
                    lazyBgObserver.observe(el);
                });
            });
            
            // Also handle videos with data-prorank-lazy
            var lazyVideoObserver = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        var video = entry.target;
                        if (video.getAttribute('data-prorank-lazy') === 'true') {
                            video.setAttribute('preload', 'metadata');
                            video.removeAttribute('data-prorank-lazy');
                            lazyVideoObserver.unobserve(video);
                        }
                    }
                });
            }, {
                rootMargin: '50px 0px',
                threshold: 0.01
            });
            
            document.addEventListener('DOMContentLoaded', function() {
                var videos = document.querySelectorAll('video[data-prorank-lazy="true"]');
                videos.forEach(function(video) {
                    lazyVideoObserver.observe(video);
                });
            });
        })();
        </script>
        <?php
    }
    
    /**
     * Get DOM document from HTML content
     *
     * @param string $content HTML content
     * @return DOMDocument|null
     */
    private function get_dom_document(string $content): ?DOMDocument {
        if (empty($content)) {
            return null;
        }

        try {
            $dom = new DOMDocument();

            // Suppress errors for HTML5 elements
            $libxml_previous = libxml_use_internal_errors(true);

            // Wrap content to ensure proper parsing
            $wrapped = '<!DOCTYPE html><html><head><meta charset="utf-8"></head><body>' . $content . '</body></html>';

            // Load HTML
            $loaded = $dom->loadHTML($wrapped, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

            // Restore error handling
            libxml_clear_errors();
            libxml_use_internal_errors($libxml_previous);

            return $loaded ? $dom : null;
        } catch (\Throwable $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO: Failed to parse HTML for lazy loading - ' . $e->getMessage());
            }
            return null;
        }
    }
    
    /**
     * Check if element/URL is excluded
     *
     * @param string $value Class string or URL to check
     * @return bool
     */
    private function is_excluded(string $value): bool {
        if (empty($value)) {
            return false;
        }
        
        // Get exclusions (cached)
        if (empty($this->exclusions)) {
            $excluded = $this->get_setting('exclude', '');
            if (!empty($excluded)) {
                $this->exclusions = array_filter(
                    array_map('trim', explode("\n", $excluded))
                );
            }
        }
        
        // Always exclude certain classes
        $always_exclude = [
            'no-lazy',
            'skip-lazy',
            'prorank-no-lazy',
            'rev-slidebg', // Revolution Slider
            'rsImg', // RoyalSlider
        ];
        
        foreach ($always_exclude as $exclude) {
            if (strpos($value, $exclude) !== false) {
                return true;
            }
        }
        
        // Check user-defined exclusions
        foreach ($this->exclusions as $pattern) {
            if (strpos($value, $pattern) !== false) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check if current request is AMP
     *
     * @return bool
     */
    private function is_amp(): bool {
        return function_exists('amp_is_request') && amp_is_request();
    }
}