<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries throughout
/**
 * 404 Not Found Monitor Module
 *
 * Monitors and tracks 404 errors with automatic redirect suggestions
 *
 * @package ProRank\SEO\Modules\TechnicalSEO
 * @since   1.5.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Modules\TechnicalSEO;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\BaseModule;
use ProRank\SEO\Plugin;

/**
 * NotFoundMonitorModule class
 */
class NotFoundMonitorModule extends BaseModule {
    
    /**
     * Module slug
     *
     * @var string
     */
    protected string $slug = '404-monitor';
    
    /**
     * Module name
     *
     * @var string
     */
    protected string $name = '404 Error Monitor';
    
    /**
     * Module description
     *
     * @var string
     */
    protected string $description = 'Track 404 errors and automatically suggest redirects to improve user experience and SEO';
    
    /**
     * Module category
     *
     * @var string
     */
    protected string $category = 'technical-seo';
    
    /**
     * Module tier
     *
     * @var string
     */
    protected string $feature_tier = 'free';
    
    /**
     * Parent module
     *
     * @var string|null
     */
    protected ?string $parent_slug = 'redirects';
    
    /**
     * Table name cache
     *
     * @var string|null
     */
    private ?string $table_name = null;
    
    /**
     * Initialize module hooks
     *
     * @return void
     */
    public function init_hooks(): void {
        // Hook into 404 detection
        add_action('template_redirect', [$this, 'track_404_error'], 999);
        
        // Register REST routes
        add_action('rest_api_init', [$this, 'register_rest_routes']);
        
        // Add cron job for cleanup
        add_action('prorank_404_cleanup', [$this, 'cleanup_old_entries']);
        
        // Schedule cleanup if not already scheduled
        if (!wp_next_scheduled('prorank_404_cleanup')) {
            wp_schedule_event(time(), 'daily', 'prorank_404_cleanup');
        }
        
        // Add admin notices for high 404 counts
        add_action('admin_notices', [$this, 'show_404_alerts']);
    }
    
    /**
     * Track 404 errors
     *
     * @return void
     */
    public function track_404_error(): void {
        if (!is_404()) {
            return;
        }
        
        // Skip if in admin or doing AJAX
        if (is_admin() || wp_doing_ajax()) {
            return;
        }
        
        // Get current URL
        $url = $this->get_current_url();
        if (empty($url)) {
            return;
        }
        
        // Skip if URL should be ignored
        if ($this->should_ignore_url($url)) {
            return;
        }
        
        // Record the 404 error
        $this->record_404_error($url);
        
        // Try to suggest a redirect
        $this->suggest_redirect($url);
    }
    
    /**
     * Get current URL
     *
     * @return string
     */
    private function get_current_url(): string {
        $https = \prorank_get_server_var( 'HTTPS' );
        $protocol = $https === 'on' ? 'https' : 'http';
        $host = \prorank_get_server_var( 'HTTP_HOST' );
        $uri = \prorank_get_server_var( 'REQUEST_URI' );
        
        return $protocol . '://' . $host . $uri;
    }
    
    /**
     * Check if URL should be ignored
     *
     * @param string $url URL to check
     * @return bool
     */
    private function should_ignore_url(string $url): bool {
        // Default ignore patterns (always applied)
        $default_patterns = [
            '/wp-admin/',
            '/wp-content/uploads/',
            '/wp-includes/',
            '.php',
            'xmlrpc.php',
            'wp-login.php',
            '/feed/',
            'robots.txt',
            'sitemap.xml',
            '.jpg', '.jpeg', '.png', '.gif', '.svg', '.webp',
            '.css', '.js',
            '.woff', '.woff2', '.ttf', '.eot',
            '.ico',
            '.map',
        ];

        // Check default patterns
        foreach ($default_patterns as $pattern) {
            if (stripos($url, $pattern) !== false) {
                return true;
            }
        }

        // Get custom exclusion rules
        $custom_rules = get_option('prorank_404_exclusion_rules', []);

        if (!empty($custom_rules)) {
            foreach ($custom_rules as $rule) {
                if (empty($rule['pattern']) || empty($rule['enabled'])) {
                    continue;
                }

                $pattern = $rule['pattern'];
                $type = $rule['type'] ?? 'contains';

                switch ($type) {
                    case 'exact':
                        if ($url === $pattern) {
                            return true;
                        }
                        break;

                    case 'starts_with':
                        if (strpos($url, $pattern) === 0) {
                            return true;
                        }
                        break;

                    case 'ends_with':
                        if (substr($url, -strlen($pattern)) === $pattern) {
                            return true;
                        }
                        break;

                    case 'regex':
                        // Validate and test regex
                        if (@preg_match($pattern, $url)) {
                            return true;
                        }
                        break;

                    case 'contains':
                    default:
                        if (stripos($url, $pattern) !== false) {
                            return true;
                        }
                        break;
                }
            }
        }

        return false;
    }
    
    /**
     * Record 404 error in database
     *
     * @param string $url URL that triggered 404
     * @return void
     */
    private function record_404_error(string $url): void {
        global $wpdb;
        
        $table = $this->get_table_name();
        
        // Get additional data
        $referrer = \prorank_get_server_var( 'HTTP_REFERER' );
        $user_agent = \prorank_get_server_var( 'HTTP_USER_AGENT' );
        $ip_address = $this->get_client_ip();
        
        // Parse URL for path only
        $parsed = wp_parse_url($url);
        $path = $parsed['path'] ?? '/';
        if (!empty($parsed['query'])) {
            $path .= '?' . $parsed['query'];
        }
        
        // Check if URL already exists
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $existing = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT id, hits FROM {$table} WHERE url = %s",
                $path
            )
        );
        
        if ($existing) {
            // Update existing record
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->update(
                $table,
                [
                    'hits' => $existing->hits + 1,
                    'last_hit' => current_time('mysql'),
                    'referrer' => $referrer,
                    'user_agent' => $user_agent,
                    'ip_address' => $ip_address,
                ],
                ['id' => $existing->id],
                ['%d', '%s', '%s', '%s', '%s'],
                ['%d']
            );
        } else {
            // Insert new record
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->insert(
                $table,
                [
                    'url' => $path,
                    'referrer' => $referrer,
                    'user_agent' => $user_agent,
                    'ip_address' => $ip_address,
                    'hits' => 1,
                    'first_hit' => current_time('mysql'),
                    'last_hit' => current_time('mysql'),
                    'status' => 'unresolved',
                ],
                ['%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s']
            );
        }
    }
    
    /**
     * Get client IP address
     *
     * @return string|null
     */
    private function get_client_ip(): ?string {
        $ip_keys = ['HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR'];
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                $ip = sanitize_text_field( wp_unslash( $_SERVER[$key] ) );
                if (strpos($ip, ',') !== false) {
                    $ip = explode(',', $ip)[0];
                }
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return null;
    }
    
    /**
     * Suggest redirect for 404 URL
     *
     * @param string $url URL that triggered 404
     * @return void
     */
    private function suggest_redirect(string $url): void {
        global $wpdb;
        
        // Parse URL for path
        $parsed = wp_parse_url($url);
        $path = $parsed['path'] ?? '/';
        
        // Remove trailing slash for comparison
        $path = rtrim($path, '/');
        
        // Try to find similar existing pages
        $similar_pages = $this->find_similar_pages($path);
        
        if (!empty($similar_pages)) {
            $best_match = $similar_pages[0];
            $similarity = $this->calculate_similarity($path, $best_match['url']);
            
            // If similarity is high enough, update the 404 record with suggestion
            if ($similarity > 0.7) {
                $table = $this->get_table_name();
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $wpdb->update(
                    $table,
                    [
                        'auto_suggested' => 1,
                        'suggestion_url' => $best_match['url'],
                        'similarity_score' => $similarity * 100,
                    ],
                    ['url' => $path],
                    ['%d', '%s', '%f'],
                    ['%s']
                );
            }
        }
    }
    
    /**
     * Find similar existing pages
     *
     * @param string $path Path to match
     * @return array
     */
    private function find_similar_pages(string $path): array {
        global $wpdb;
        
        // Extract keywords from path
        $keywords = $this->extract_keywords($path);
        
        if (empty($keywords)) {
            return [];
        }
        
        // Search for posts/pages with similar URLs or titles
        $results = [];
        
        // Search in posts
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $posts = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT ID, post_name, post_title, guid 
                FROM {$wpdb->posts} 
                WHERE post_status = 'publish' 
                AND post_type IN ('post', 'page') 
                AND (post_name LIKE %s OR post_title LIKE %s)
                LIMIT 10",
                '%' . $wpdb->esc_like($keywords[0]) . '%',
                '%' . $wpdb->esc_like($keywords[0]) . '%'
            )
        );
        
        foreach ($posts as $post) {
            $results[] = [
                'url' => get_permalink($post->ID),
                'title' => $post->post_title,
                'slug' => $post->post_name,
            ];
        }
        
        return $results;
    }
    
    /**
     * Extract keywords from path
     *
     * @param string $path URL path
     * @return array
     */
    private function extract_keywords(string $path): array {
        // Remove common prefixes and suffixes
        $path = trim($path, '/');
        $path = preg_replace('/\.(html?|php|aspx?)$/i', '', $path);
        
        // Split by common delimiters
        $parts = preg_split('/[-_\/\s]+/', $path);
        
        // Filter out common words and short words
        $stopwords = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for'];
        $keywords = array_filter($parts, function($word) use ($stopwords) {
            return strlen($word) > 2 && !in_array(strtolower($word), $stopwords);
        });
        
        return array_values($keywords);
    }
    
    /**
     * Calculate similarity between two strings
     *
     * @param string $str1 First string
     * @param string $str2 Second string
     * @return float Similarity score (0-1)
     */
    private function calculate_similarity(string $str1, string $str2): float {
        $str1 = strtolower($str1);
        $str2 = strtolower($str2);
        
        // Use Levenshtein distance for similarity
        $max_len = max(strlen($str1), strlen($str2));
        if ($max_len === 0) {
            return 1.0;
        }
        
        $distance = levenshtein($str1, $str2);
        $similarity = 1 - ($distance / $max_len);
        
        return max(0, $similarity);
    }
    
    /**
     * Register REST routes
     *
     * @return void
     */
    public function register_rest_routes(): void {
        register_rest_route('prorank-seo/v1', '/404-monitor', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_404_errors'],
                'permission_callback' => [$this, 'check_permissions'],
            ],
        ]);
        
        register_rest_route('prorank-seo/v1', '/404-monitor/(?P<id>\d+)', [
            [
                'methods' => 'DELETE',
                'callback' => [$this, 'delete_404_error'],
                'permission_callback' => [$this, 'check_permissions'],
            ],
            [
                'methods' => 'POST',
                'callback' => [$this, 'resolve_404_error'],
                'permission_callback' => [$this, 'check_permissions'],
            ],
        ]);
        
        register_rest_route('prorank-seo/v1', '/404-monitor/stats', [
            'methods' => 'GET',
            'callback' => [$this, 'get_404_stats'],
            'permission_callback' => [$this, 'check_permissions'],
        ]);
        
        register_rest_route('prorank-seo/v1', '/404-monitor/create-redirect', [
            'methods' => 'POST',
            'callback' => [$this, 'create_redirect_from_404'],
            'permission_callback' => [$this, 'check_permissions'],
        ]);

        register_rest_route('prorank-seo/v1', '/404-monitor/exclusions', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_exclusion_rules'],
                'permission_callback' => [$this, 'check_permissions'],
            ],
            [
                'methods' => 'POST',
                'callback' => [$this, 'update_exclusion_rules'],
                'permission_callback' => [$this, 'check_permissions'],
            ],
        ]);
    }
    
    /**
     * Get 404 errors
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response
     */
    public function get_404_errors(\WP_REST_Request $request): \WP_REST_Response {
        global $wpdb;
        
        $table = $this->get_table_name();
        
        // Get parameters
        $page = max(1, (int) $request->get_param('page'));
        $per_page = max(1, min(100, (int) $request->get_param('per_page') ?: 20));
        $status = $request->get_param('status');
        $orderby = $request->get_param('orderby') ?: 'hits';
        $order = strtoupper($request->get_param('order') ?: 'DESC');
        
        // Build query
        $where = '1=1';
        $params = [];
        
        if ($status) {
            $where .= ' AND status = %s';
            $params[] = $status;
        }
        
        // Get total count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        $count_query = "SELECT COUNT(*) FROM {$table} WHERE {$where}";
        if (!empty($params)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
            $count_query = $wpdb->prepare($count_query, ...$params);
        }
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $total = (int) $wpdb->get_var($count_query);
        
        // Get 404 errors
        $offset = ($page - 1) * $per_page;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        $query = "SELECT * FROM {$table} WHERE {$where} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";
        $params[] = $per_page;
        $params[] = $offset;
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table/query is safe
        $errors = $wpdb->get_results($wpdb->prepare($query, ...$params));
        
        return new \WP_REST_Response([
            'errors' => $errors,
            'total' => $total,
            'total_pages' => ceil($total / $per_page),
            'page' => $page,
            'per_page' => $per_page,
        ]);
    }
    
    /**
     * Get 404 statistics
     *
     * @return \WP_REST_Response
     */
    public function get_404_stats(): \WP_REST_Response {
        global $wpdb;
        
        $table = $this->get_table_name();
        
        // Get overall stats
        $stats = $wpdb->get_row("
            SELECT 
                COUNT(*) as total_urls,
                SUM(hits) as total_hits,
                COUNT(CASE WHEN status = 'resolved' THEN 1 END) as resolved_count,
                COUNT(CASE WHEN auto_suggested = 1 THEN 1 END) as suggested_count
            FROM {$table}
        ");
        
        // Get top 404s
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $top_errors = $wpdb->get_results("
            SELECT url, hits, last_hit, auto_suggested, suggestion_url
            FROM {$table}
            WHERE status = 'unresolved'
            ORDER BY hits DESC
            LIMIT 10
        ");
        
        // Get recent 404s
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $recent_errors = $wpdb->get_results("
            SELECT url, hits, last_hit
            FROM {$table}
            WHERE status = 'unresolved'
            ORDER BY last_hit DESC
            LIMIT 10
        ");
        
        return new \WP_REST_Response([
            'stats' => $stats,
            'top_errors' => $top_errors,
            'recent_errors' => $recent_errors,
        ]);
    }
    
    /**
     * Create redirect from 404 error
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response|\WP_Error
     */
    public function create_redirect_from_404(\WP_REST_Request $request) {
        global $wpdb;
        
        $id = (int) $request->get_param('id');
        $target_url = $request->get_param('target_url');
        $redirect_type = (int) $request->get_param('type') ?: 301;
        
        // Get 404 error
        $table = $this->get_table_name();
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $error = $wpdb->get_row(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT * FROM {$table} WHERE id = %d",
                $id
            )
        );
        
        if (!$error) {
            return new \WP_Error('not_found', __('404 error not found', 'prorank-seo'), ['status' => 404]);
        }
        
        // Create redirect
        $redirects_table = $wpdb->prefix . 'prorank_redirects';
        $result = $wpdb->insert(
            $redirects_table,
            [
                'source_url' => $error->url,
                'target_url' => $target_url ?: $error->suggestion_url,
                'type' => $redirect_type,
                'status' => 'active',
                'notes' => sprintf(
                    /* translators: %s: ID */
                    __('Auto-created from 404 monitor (ID: %d)', 'prorank-seo'), $id),
            ],
            ['%s', '%s', '%d', '%s', '%s']
        );
        
        if ($result === false) {
            return new \WP_Error('db_error', __('Failed to create redirect', 'prorank-seo'), ['status' => 500]);
        }
        
        $redirect_id = $wpdb->insert_id;
        
        // Update 404 error status
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update(
            $table,
            [
                'status' => 'resolved',
                'redirect_id' => $redirect_id,
            ],
            ['id' => $id],
            ['%s', '%d'],
            ['%d']
        );
        
        return new \WP_REST_Response([
            'success' => true,
            'redirect_id' => $redirect_id,
            'message' => __('Redirect created successfully', 'prorank-seo'),
        ]);
    }
    
    /**
     * Delete 404 error
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response|\WP_Error
     */
    public function delete_404_error(\WP_REST_Request $request) {
        global $wpdb;
        
        $id = (int) $request->get_param('id');
        $table = $this->get_table_name();
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $result = $wpdb->delete(
            $table,
            ['id' => $id],
            ['%d']
        );
        
        if ($result === false) {
            return new \WP_Error('db_error', __('Failed to delete 404 error', 'prorank-seo'), ['status' => 500]);
        }
        
        return new \WP_REST_Response(null, 204);
    }
    
    /**
     * Resolve 404 error
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response|\WP_Error
     */
    public function resolve_404_error(\WP_REST_Request $request) {
        global $wpdb;
        
        $id = (int) $request->get_param('id');
        $table = $this->get_table_name();
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $result = $wpdb->update(
            $table,
            ['status' => 'resolved'],
            ['id' => $id],
            ['%s'],
            ['%d']
        );
        
        if ($result === false) {
            return new \WP_Error('db_error', __('Failed to resolve 404 error', 'prorank-seo'), ['status' => 500]);
        }
        
        return new \WP_REST_Response([
            'success' => true,
            'message' => __('404 error marked as resolved', 'prorank-seo'),
        ]);
    }
    
    /**
     * Get exclusion rules
     *
     * @return \WP_REST_Response
     */
    public function get_exclusion_rules(): \WP_REST_Response {
        $rules = get_option('prorank_404_exclusion_rules', []);

        // Ensure each rule has all expected fields
        $formatted_rules = array_map(function($rule) {
            return [
                'id' => $rule['id'] ?? uniqid(),
                'pattern' => $rule['pattern'] ?? '',
                'type' => $rule['type'] ?? 'contains',
                'enabled' => $rule['enabled'] ?? true,
                'description' => $rule['description'] ?? '',
            ];
        }, $rules);

        return new \WP_REST_Response([
            'rules' => $formatted_rules,
            'available_types' => [
                'contains' => __('Contains', 'prorank-seo'),
                'exact' => __('Exact match', 'prorank-seo'),
                'starts_with' => __('Starts with', 'prorank-seo'),
                'ends_with' => __('Ends with', 'prorank-seo'),
                'regex' => __('Regular expression', 'prorank-seo'),
            ],
        ]);
    }

    /**
     * Update exclusion rules
     *
     * @param \WP_REST_Request $request
     * @return \WP_REST_Response|\WP_Error
     */
    public function update_exclusion_rules(\WP_REST_Request $request) {
        $rules = $request->get_param('rules') ?: [];

        // Validate and sanitize rules
        $validated_rules = [];
        foreach ($rules as $rule) {
            $pattern = sanitize_text_field($rule['pattern'] ?? '');

            if (empty($pattern)) {
                continue;
            }

            $type = in_array($rule['type'] ?? '', ['contains', 'exact', 'starts_with', 'ends_with', 'regex'])
                ? $rule['type']
                : 'contains';

            // Validate regex patterns
            if ($type === 'regex') {
                if (@preg_match($pattern, '') === false) {
                    return new \WP_Error(
                        'invalid_regex',
                        sprintf(
                            /* translators: %s: placeholder value */
                            __('Invalid regex pattern: %s', 'prorank-seo'), $pattern),
                        ['status' => 400]
                    );
                }
            }

            $validated_rules[] = [
                'id' => $rule['id'] ?? uniqid(),
                'pattern' => $pattern,
                'type' => $type,
                'enabled' => (bool) ($rule['enabled'] ?? true),
                'description' => sanitize_text_field($rule['description'] ?? ''),
            ];
        }

        // Save rules
        update_option('prorank_404_exclusion_rules', $validated_rules);

        return new \WP_REST_Response([
            'success' => true,
            'rules' => $validated_rules,
            'message' => __('Exclusion rules updated successfully', 'prorank-seo'),
        ]);
    }

    /**
     * Check REST permissions
     *
     * @return bool
     */
    public function check_permissions(): bool {
        return current_user_can('manage_options');
    }
    
    /**
     * Cleanup old entries
     *
     * @return void
     */
    public function cleanup_old_entries(): void {
        global $wpdb;
        
        $table = $this->get_table_name();
        $days = (int) get_option('prorank_404_retention_days', 90);
        $cutoff = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        // Delete old resolved entries
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "DELETE FROM {$table} 
                WHERE status = 'resolved' 
                AND last_hit < %s",
                $cutoff
            )
        );
        
        // Delete low-hit unresolved entries older than 30 days
        $old_cutoff = gmdate('Y-m-d H:i:s', strtotime('-30 days'));
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "DELETE FROM {$table} 
                WHERE status = 'unresolved' 
                AND hits <= 2 
                AND last_hit < %s",
                $old_cutoff
            )
        );
    }
    
    /**
     * Show admin notices for high 404 counts
     *
     * @return void
     */
    public function show_404_alerts(): void {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Only show on dashboard and ProRank pages
        $screen = get_current_screen();
        if (!$screen || (!in_array($screen->id, ['dashboard', 'toplevel_page_prorank-seo']))) {
            return;
        }
        
        global $wpdb;
        $table = $this->get_table_name();
        
        // Check for high 404 count
        $threshold = (int) get_option('prorank_404_alert_threshold', 50);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $unresolved = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$table} 
            WHERE status = 'unresolved' 
            AND hits >= {$threshold}
        ");
        
        if ($unresolved > 0) {
            $url = admin_url('admin.php?page=prorank-seo-technical#redirects');
            ?>
            <div class="notice notice-warning is-dismissible">
                <p>
                    <strong><?php esc_html_e('ProRank SEO:', 'prorank-seo'); ?></strong>
                    <?php
                    echo wp_kses(
                        sprintf(
                            /* translators: 1: URL count 2: admin URL */
                            __('You have %1$d URLs with high 404 error counts. <a href="%2$s">Review and fix them</a> to improve user experience and SEO.', 'prorank-seo'),
                            absint($unresolved),
                            esc_url($url)
                        ),
                        ['a' => ['href' => true]]
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Get table name
     *
     * @return string
     */
    private function get_table_name(): string {
        if ($this->table_name === null) {
            global $wpdb;
            $this->table_name = $wpdb->prefix . 'prorank_404_monitor';
        }
        
        return $this->table_name;
    }
}
